import { PageProps } from "@/types";
import { Head, Link } from "@inertiajs/react";
import ApplicationLogo from "@/Components/ApplicationLogo";
import {
    CheckCircle,
    Wifi,
    CreditCard,
    Headphones,
    Zap,
    MessageSquare,
    Tv,
    FileText,
    Phone,
    Mail,
    MapPin,
    Shield,
    Clock,
    Users,
    ArrowRight,
    Menu,
    X,
} from "lucide-react";
import { useState, useEffect } from "react";

export default function Welcome({
    auth,
}: PageProps<{
    auth: { user: any };
}>) {
    const [activeSection, setActiveSection] = useState("home");
    const [isMobileMenuOpen, setIsMobileMenuOpen] = useState(false);
    const [isScrolled, setIsScrolled] = useState(false);

    const services = [
        {
            icon: <Wifi className="w-6 h-6" />,
            title: "Airtime & Data",
            desc: "Instant recharge for all networks",
        },
        {
            icon: <CreditCard className="w-6 h-6" />,
            title: "Bill Payments",
            desc: "Electricity, water bills & more",
        },
        {
            icon: <Tv className="w-6 h-6" />,
            title: "TV Subscription",
            desc: "DStv, GOtv, Startimes",
        },
        {
            icon: <Zap className="w-6 h-6" />,
            title: "Airtime to Cash",
            desc: "Convert airtime with low charges",
        },
        {
            icon: <MessageSquare className="w-6 h-6" />,
            title: "Bulk SMS",
            desc: "Send SMS in bulk at great rates",
        },
        {
            icon: <FileText className="w-6 h-6" />,
            title: "Result Checker",
            desc: "WAEC, NECO, GCE pins",
        },
    ];

    const testimonials = [
        {
            name: "Godly",
            role: "Regular User",
            text: "Hopay Sub has made staying connected effortless. Recharging is so simple!",
        },
        {
            name: "Abraham Agabi",
            role: "Business Owner",
            text: "The pricing is competitive and reliable. A fantastic platform!",
        },
        {
            name: "Aremu",
            role: "Student",
            text: "Customer support is fantastic—responsive and quick to resolve issues.",
        },
    ];

    const dataPlans = [
        {
            network: "MTN",
            color: "bg-yellow-100 text-yellow-800",
            plans: [
                { data: "1.0GB", price: "₦492", validity: "Daily" },
                { data: "2.0GB", price: "₦1,470", validity: "30 Days" },
                { data: "6.0GB", price: "₦2,425", validity: "7 Days" },
            ],
        },
        {
            network: "AIRTEL",
            color: "bg-red-100 text-red-800",
            plans: [
                { data: "1.0GB", price: "₦400", validity: "2 Days" },
                { data: "2.0GB", price: "₦735", validity: "Monthly" },
                { data: "10GB", price: "₦3,500", validity: "Monthly" },
            ],
        },
        {
            network: "GLO",
            color: "bg-green-100 text-green-800",
            plans: [
                { data: "1.0GB", price: "₦500", validity: "1 Month" },
                { data: "2.0GB", price: "₦1,000", validity: "1 Month" },
                { data: "10GB", price: "₦5,000", validity: "1 Month" },
            ],
        },
        {
            network: "9MOBILE",
            color: "bg-blue-100 text-blue-800",
            plans: [
                { data: "1.0GB", price: "₦280", validity: "1 Month" },
                { data: "2.0GB", price: "₦560", validity: "1 Month" },
                { data: "5GB", price: "₦1,400", validity: "1 Month" },
            ],
        },
    ];

    const faqs = [
        {
            question: "How do I buy data?",
            answer: "1. Log into your account 2. Fund your wallet 3. Select your network and plan 4. Confirm and receive instantly!",
        },
        {
            question: "How do I fund my wallet?",
            answer: "Fund securely via bank transfer, debit card, or purchase coupon codes from our agents.",
        },
        {
            question: "Is there customer support?",
            answer: "Yes! We offer 24/7 customer support via email, phone, and live chat.",
        },
    ];

    const navItems = [
        { id: "home", label: "Home" },
        { id: "about", label: "About Us" },
        { id: "services", label: "Services" },
        { id: "pricing", label: "Pricing" },
        { id: "agent", label: "Become Agent" },
        { id: "contact", label: "Contact" },
    ];

    const scrollToSection = (sectionId: string) => {
        setActiveSection(sectionId);
        setIsMobileMenuOpen(false);
        const element = document.getElementById(sectionId);
        if (element) {
            const offset = 80; // Height of fixed navbar
            const elementPosition = element.getBoundingClientRect().top;
            const offsetPosition =
                elementPosition + window.pageYOffset - offset;

            window.scrollTo({
                top: offsetPosition,
                behavior: "smooth",
            });
        }
    };

    // Handle scroll effect for navbar
    useEffect(() => {
        const handleScroll = () => {
            setIsScrolled(window.scrollY > 10);
        };

        window.addEventListener("scroll", handleScroll);
        return () => window.removeEventListener("scroll", handleScroll);
    }, []);

    // Close mobile menu when clicking outside
    useEffect(() => {
        const handleClickOutside = (event: MouseEvent) => {
            const target = event.target as HTMLElement;
            if (
                isMobileMenuOpen &&
                !target.closest(".mobile-menu") &&
                !target.closest(".hamburger-button")
            ) {
                setIsMobileMenuOpen(false);
            }
        };

        document.addEventListener("click", handleClickOutside);
        return () => document.removeEventListener("click", handleClickOutside);
    }, [isMobileMenuOpen]);

    // Close mobile menu on escape key
    useEffect(() => {
        const handleEscape = (event: KeyboardEvent) => {
            if (event.key === "Escape" && isMobileMenuOpen) {
                setIsMobileMenuOpen(false);
            }
        };

        document.addEventListener("keydown", handleEscape);
        return () => document.removeEventListener("keydown", handleEscape);
    }, [isMobileMenuOpen]);

    return (
        <>
            <Head title="Hopay Sub - Your Instant VTU Solution" />

            {/* Navigation */}
            <nav
                className={`fixed w-full z-50 transition-all duration-300 ${
                    isScrolled
                        ? "bg-white/95 backdrop-blur-sm shadow-md"
                        : "bg-white"
                }`}
            >
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <div className="flex justify-between items-center h-16">
                        <div className="flex items-center space-x-4">
                            <Link
                                href="#home"
                                onClick={() => scrollToSection("home")}
                            >
                                <ApplicationLogo className="h-10 w-auto" />
                            </Link>

                            {/* Desktop Navigation */}
                            <div className="hidden md:flex items-center space-x-1 ml-6">
                                {navItems.map((item) => (
                                    <button
                                        key={item.id}
                                        onClick={() => scrollToSection(item.id)}
                                        className={`px-3 py-2 text-sm font-medium rounded-md transition ${
                                            activeSection === item.id
                                                ? "text-blue-600 bg-blue-50"
                                                : "text-gray-700 hover:text-blue-600 hover:bg-gray-50"
                                        }`}
                                    >
                                        {item.label}
                                    </button>
                                ))}
                            </div>
                        </div>

                        {/* Desktop Auth Buttons */}
                        <div className="hidden md:flex items-center space-x-4">
                            {auth.user ? (
                                <Link
                                    href={route("dashboard")}
                                    className="bg-blue-600 text-white px-4 py-2 rounded-md text-sm font-medium hover:bg-blue-700 transition"
                                >
                                    Dashboard
                                </Link>
                            ) : (
                                <>
                                    <Link
                                        href={route("login")}
                                        className="text-gray-700 hover:text-blue-600 px-3 py-2 text-sm font-medium"
                                    >
                                        Log In
                                    </Link>
                                    <Link
                                        href={route("register")}
                                        className="bg-blue-600 text-white px-4 py-2 rounded-md text-sm font-medium hover:bg-blue-700 transition"
                                    >
                                        Sign Up
                                    </Link>
                                </>
                            )}
                        </div>

                        {/* Mobile Menu Button (Hamburger) */}
                        <div className="md:hidden flex items-center space-x-4">
                            {!auth.user && (
                                <Link
                                    href={route("login")}
                                    className="text-gray-700 hover:text-blue-600 px-3 py-2 text-sm font-medium"
                                >
                                    Log In
                                </Link>
                            )}
                            <button
                                onClick={() =>
                                    setIsMobileMenuOpen(!isMobileMenuOpen)
                                }
                                className="hamburger-button text-gray-700 hover:text-blue-600 p-2 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                                aria-label="Toggle menu"
                            >
                                {isMobileMenuOpen ? (
                                    <X className="w-6 h-6" />
                                ) : (
                                    <Menu className="w-6 h-6" />
                                )}
                            </button>
                        </div>
                    </div>
                </div>

                {/* Mobile Menu Overlay */}
                {isMobileMenuOpen && (
                    <div className="md:hidden mobile-menu">
                        {/* Backdrop */}
                        <div
                            className="fixed inset-0 bg-black bg-opacity-50 z-40"
                            onClick={() => setIsMobileMenuOpen(false)}
                        />

                        {/* Slide-in Menu */}
                        <div className="fixed right-0 top-0 h-full w-64 bg-white shadow-xl z-50 transform transition-transform duration-300 ease-in-out">
                            <div className="p-4 border-b">
                                <div className="flex items-center justify-between">
                                    <ApplicationLogo className="h-8 w-auto" />
                                    <button
                                        onClick={() =>
                                            setIsMobileMenuOpen(false)
                                        }
                                        className="text-gray-500 hover:text-gray-700 p-1 rounded-md"
                                    >
                                        <X className="w-5 h-5" />
                                    </button>
                                </div>
                            </div>

                            <div className="p-4 space-y-1">
                                {navItems.map((item) => (
                                    <button
                                        key={item.id}
                                        onClick={() => scrollToSection(item.id)}
                                        className={`w-full text-left px-4 py-3 rounded-lg transition flex items-center ${
                                            activeSection === item.id
                                                ? "bg-blue-50 text-blue-600 font-medium"
                                                : "text-gray-700 hover:bg-gray-100"
                                        }`}
                                    >
                                        <span>{item.label}</span>
                                        {activeSection === item.id && (
                                            <div className="ml-auto w-2 h-2 bg-blue-600 rounded-full"></div>
                                        )}
                                    </button>
                                ))}
                            </div>

                            <div className="absolute bottom-0 left-0 right-0 p-4 border-t bg-gray-50">
                                {auth.user ? (
                                    <Link
                                        href={route("dashboard")}
                                        className="block w-full bg-blue-600 text-white text-center px-4 py-3 rounded-md font-medium hover:bg-blue-700 transition mb-3"
                                        onClick={() =>
                                            setIsMobileMenuOpen(false)
                                        }
                                    >
                                        Dashboard
                                    </Link>
                                ) : (
                                    <Link
                                        href={route("register")}
                                        className="block w-full bg-blue-600 text-white text-center px-4 py-3 rounded-md font-medium hover:bg-blue-700 transition"
                                        onClick={() =>
                                            setIsMobileMenuOpen(false)
                                        }
                                    >
                                        Sign Up Free
                                    </Link>
                                )}

                                <div className="mt-4 pt-4 border-t text-center">
                                    <div className="text-sm text-gray-500">
                                        Need help?{" "}
                                        <span className="text-blue-600 font-medium">
                                            09015452077
                                        </span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                )}
            </nav>

            {/* Hero Section */}
            <section
                id="home"
                className="pt-20 bg-gradient-to-br from-blue-50 via-white to-blue-50"
            >
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-20 lg:py-32">
                    {/* Rest of your existing sections remain the same */}
                    <div className="grid lg:grid-cols-2 gap-12 items-center">
                        <div className="text-left">
                            <h1 className="text-4xl md:text-5xl lg:text-6xl font-bold text-gray-900 mb-6 leading-tight">
                                Welcome to{" "}
                                <span className="text-blue-600">Hopay Sub</span>
                            </h1>
                            <p className="text-lg text-gray-600 mb-8 leading-relaxed">
                                Your instant, reliable partner for seamless
                                connectivity. Buy airtime, mobile data, pay
                                bills, and manage all your VTU services in one
                                place. Fast, secure, and trusted by thousands.
                            </p>
                            <div className="flex flex-col sm:flex-row gap-4">
                                <Link
                                    href={route("register")}
                                    className="group bg-blue-600 text-white px-8 py-4 rounded-lg text-lg font-semibold hover:bg-blue-700 transition-all duration-300 shadow-lg hover:shadow-xl flex items-center justify-center"
                                >
                                    Create Free Account
                                    <ArrowRight className="ml-2 w-5 h-5 group-hover:translate-x-1 transition-transform" />
                                </Link>
                                <button
                                    onClick={() => scrollToSection("services")}
                                    className="border-2 border-blue-600 text-blue-600 px-8 py-4 rounded-lg text-lg font-semibold hover:bg-blue-50 transition"
                                >
                                    Explore Services
                                </button>
                            </div>
                            <div className="mt-12 grid grid-cols-2 md:grid-cols-4 gap-6">
                                <div className="text-center">
                                    <div className="text-3xl font-bold text-blue-600">
                                        10K+
                                    </div>
                                    <div className="text-gray-600 text-sm">
                                        Happy Users
                                    </div>
                                </div>
                                <div className="text-center">
                                    <div className="text-3xl font-bold text-blue-600">
                                        24/7
                                    </div>
                                    <div className="text-gray-600 text-sm">
                                        Support
                                    </div>
                                </div>
                                <div className="text-center">
                                    <div className="text-3xl font-bold text-blue-600">
                                        99%
                                    </div>
                                    <div className="text-gray-600 text-sm">
                                        Uptime
                                    </div>
                                </div>
                                <div className="text-center">
                                    <div className="text-3xl font-bold text-blue-600">
                                        Instant
                                    </div>
                                    <div className="text-gray-600 text-sm">
                                        Delivery
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div className="relative">
                            <div className="relative bg-gradient-to-br from-blue-500 to-blue-600 rounded-3xl p-8 shadow-2xl transform hover:scale-[1.02] transition-transform duration-300">
                                <div className="text-white">
                                    <h3 className="text-2xl font-bold mb-6">
                                        Why Choose Hopay Sub?
                                    </h3>
                                    <ul className="space-y-4">
                                        {[
                                            {
                                                icon: (
                                                    <Zap className="w-5 h-5" />
                                                ),
                                                text: "⚡ Instant Recharges & Top-ups",
                                            },
                                            {
                                                icon: (
                                                    <Shield className="w-5 h-5" />
                                                ),
                                                text: "🛡️ Secure & Reliable Transactions",
                                            },
                                            {
                                                icon: (
                                                    <Clock className="w-5 h-5" />
                                                ),
                                                text: "⏰ 24/7 Customer Support",
                                            },
                                            {
                                                icon: (
                                                    <Users className="w-5 h-5" />
                                                ),
                                                text: "👥 Agent Network Opportunities",
                                            },
                                        ].map((item, index) => (
                                            <li
                                                key={index}
                                                className="flex items-center space-x-4"
                                            >
                                                <div className="bg-white/20 p-2 rounded-lg">
                                                    {item.icon}
                                                </div>
                                                <span className="text-lg">
                                                    {item.text}
                                                </span>
                                            </li>
                                        ))}
                                    </ul>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </section>

            {/* About Us Section */}
            <section id="about" className="py-20 bg-white">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <div className="text-center mb-16">
                        <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
                            About Hopay Sub
                        </h2>
                        <p className="text-gray-600 max-w-3xl mx-auto text-lg">
                            A registered telecommunication services provider
                            dedicated to making your digital life effortless
                            with 20+ years of combined expertise in the
                            industry.
                        </p>
                    </div>

                    <div className="grid md:grid-cols-3 gap-8 mb-12">
                        <div className="bg-blue-50 rounded-2xl p-8 text-center">
                            <div className="text-blue-600 mb-4">
                                <Shield className="w-12 h-12 mx-auto" />
                            </div>
                            <h3 className="text-xl font-semibold text-gray-900 mb-3">
                                Trusted & Secure
                            </h3>
                            <p className="text-gray-600">
                                Bank-level security for all your transactions
                                and data protection.
                            </p>
                        </div>

                        <div className="bg-blue-50 rounded-2xl p-8 text-center">
                            <div className="text-blue-600 mb-4">
                                <Clock className="w-12 h-12 mx-auto" />
                            </div>
                            <h3 className="text-xl font-semibold text-gray-900 mb-3">
                                Always Available
                            </h3>
                            <p className="text-gray-600">
                                24/7 services with instant delivery and
                                round-the-clock support.
                            </p>
                        </div>

                        <div className="bg-blue-50 rounded-2xl p-8 text-center">
                            <div className="text-blue-600 mb-4">
                                <Users className="w-12 h-12 mx-auto" />
                            </div>
                            <h3 className="text-xl font-semibold text-gray-900 mb-3">
                                Community Focused
                            </h3>
                            <p className="text-gray-600">
                                Empowering entrepreneurs and students through
                                our agent network.
                            </p>
                        </div>
                    </div>
                </div>
            </section>

            {/* Services Section */}
            <section id="services" className="py-20 bg-gray-50">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <div className="text-center mb-16">
                        <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
                            Our Services
                        </h2>
                        <p className="text-gray-600 max-w-3xl mx-auto text-lg">
                            Comprehensive VTU services designed to keep you
                            connected, powered, and productive
                        </p>
                    </div>

                    <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
                        {services.map((service, index) => (
                            <div
                                key={index}
                                className="group bg-white rounded-2xl p-8 hover:shadow-xl transition-all duration-300 hover:-translate-y-1"
                            >
                                <div className="text-blue-600 mb-6 transform group-hover:scale-110 transition-transform duration-300">
                                    {service.icon}
                                </div>
                                <h3 className="text-xl font-semibold text-gray-900 mb-3 group-hover:text-blue-600 transition-colors">
                                    {service.title}
                                </h3>
                                <p className="text-gray-600 mb-4">
                                    {service.desc}
                                </p>
                                <button className="text-blue-600 font-medium inline-flex items-center group-hover:underline">
                                    Learn More
                                    <ArrowRight className="ml-2 w-4 h-4" />
                                </button>
                            </div>
                        ))}
                    </div>
                </div>
            </section>

            {/* Pricing Section */}
            <section id="pricing" className="py-20 bg-white">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <div className="text-center mb-16">
                        <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
                            Affordable Data Plans
                        </h2>
                        <p className="text-gray-600 max-w-3xl mx-auto text-lg">
                            Get the best rates for all networks with instant
                            activation
                        </p>
                    </div>

                    <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-8">
                        {dataPlans.map((network, index) => (
                            <div
                                key={index}
                                className="border border-gray-200 rounded-2xl overflow-hidden hover:shadow-xl transition-shadow"
                            >
                                <div className={`p-6 ${network.color}`}>
                                    <h3 className="text-xl font-bold">
                                        {network.network}
                                    </h3>
                                </div>
                                <div className="p-6">
                                    {network.plans.map((plan, planIndex) => (
                                        <div
                                            key={planIndex}
                                            className="mb-4 pb-4 border-b border-gray-100 last:border-b-0 last:mb-0 last:pb-0"
                                        >
                                            <div className="flex justify-between items-center mb-1">
                                                <span className="font-semibold text-gray-900">
                                                    {plan.data}
                                                </span>
                                                <span className="text-blue-600 font-bold">
                                                    {plan.price}
                                                </span>
                                            </div>
                                            <div className="text-sm text-gray-500">
                                                {plan.validity}
                                            </div>
                                        </div>
                                    ))}
                                </div>
                                <div className="px-6 pb-6">
                                    <Link
                                        href={
                                            auth.user
                                                ? route("dashboard")
                                                : route("register")
                                        }
                                        className="block w-full bg-blue-600 text-white text-center py-3 rounded-lg font-medium hover:bg-blue-700 transition"
                                    >
                                        Buy Now
                                    </Link>
                                </div>
                            </div>
                        ))}
                    </div>

                    <div className="text-center mt-12">
                        <Link
                            href="#"
                            className="inline-flex items-center text-blue-600 font-semibold text-lg hover:text-blue-700 group"
                        >
                            View All Data Plans
                            <ArrowRight className="ml-2 w-5 h-5 group-hover:translate-x-1 transition-transform" />
                        </Link>
                    </div>
                </div>
            </section>

            {/* Become Agent Section */}
            <section
                id="agent"
                className="py-20 bg-gradient-to-r from-blue-600 to-indigo-600"
            >
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <div className="text-center text-white mb-12">
                        <h2 className="text-3xl md:text-4xl font-bold mb-6">
                            Become a Hopay Sub Agent
                        </h2>
                        <p className="text-blue-100 text-lg max-w-3xl mx-auto mb-8">
                            Join our network of outstanding entrepreneurs!
                            Partner with Hopay Sub to bring easy-payment
                            services closer to your community and earn
                            attractive commissions on every transaction.
                        </p>
                    </div>

                    <div className="grid md:grid-cols-3 gap-8 mb-12">
                        <div className="bg-white/10 backdrop-blur-sm rounded-2xl p-8 border border-white/20">
                            <div className="text-3xl font-bold text-white mb-2">
                                ₦500K+
                            </div>
                            <div className="text-blue-100">
                                Monthly Earning Potential
                            </div>
                        </div>
                        <div className="bg-white/10 backdrop-blur-sm rounded-2xl p-8 border border-white/20">
                            <div className="text-3xl font-bold text-white mb-2">
                                24/7
                            </div>
                            <div className="text-blue-100">Agent Support</div>
                        </div>
                        <div className="bg-white/10 backdrop-blur-sm rounded-2xl p-8 border border-white/20">
                            <div className="text-3xl font-bold text-white mb-2">
                                0%
                            </div>
                            <div className="text-blue-100">Setup Fee</div>
                        </div>
                    </div>

                    <div className="text-center">
                        <Link
                            href={route("register")}
                            className="inline-flex items-center bg-white text-blue-600 px-8 py-4 rounded-lg text-lg font-semibold hover:bg-blue-50 transition shadow-lg hover:shadow-xl"
                        >
                            Start Earning as Agent
                            <ArrowRight className="ml-2 w-5 h-5" />
                        </Link>
                    </div>
                </div>
            </section>

            {/* Testimonials */}
            <section className="py-20 bg-gray-50">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <div className="text-center mb-16">
                        <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
                            Trusted by Thousands
                        </h2>
                    </div>

                    <div className="grid md:grid-cols-3 gap-8">
                        {testimonials.map((testimonial, index) => (
                            <div
                                key={index}
                                className="bg-white rounded-2xl p-8 shadow-lg hover:shadow-xl transition-shadow"
                            >
                                <div className="flex items-center mb-6">
                                    <div className="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center text-blue-600 font-bold text-lg">
                                        {testimonial.name.charAt(0)}
                                    </div>
                                    <div className="ml-4">
                                        <h4 className="font-semibold text-gray-900">
                                            {testimonial.name}
                                        </h4>
                                        <div className="text-gray-500 text-sm">
                                            {testimonial.role}
                                        </div>
                                    </div>
                                </div>
                                <p className="text-gray-600 italic">
                                    "{testimonial.text}"
                                </p>
                                <div className="flex text-yellow-400 mt-4">
                                    {"★".repeat(5)}
                                </div>
                            </div>
                        ))}
                    </div>
                </div>
            </section>

            {/* FAQ Section */}
            <section className="py-20 bg-white">
                <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
                    <div className="text-center mb-16">
                        <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
                            Frequently Asked Questions
                        </h2>
                    </div>

                    <div className="space-y-6">
                        {faqs.map((faq, index) => (
                            <div
                                key={index}
                                className="border border-gray-200 rounded-2xl p-6 hover:border-blue-300 transition-colors"
                            >
                                <h3 className="text-lg font-semibold text-gray-900 mb-3">
                                    {faq.question}
                                </h3>
                                <p className="text-gray-600">{faq.answer}</p>
                            </div>
                        ))}
                    </div>
                </div>
            </section>

            {/* Contact Section */}
            <section id="contact" className="py-20 bg-gray-900">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <div className="grid lg:grid-cols-2 gap-12">
                        <div className="text-white">
                            <h2 className="text-3xl md:text-4xl font-bold mb-6">
                                Get In Touch
                            </h2>
                            <p className="text-gray-400 mb-8 text-lg">
                                Have questions? Need support? Our team is here
                                to help you 24/7.
                            </p>

                            <div className="space-y-6">
                                <div className="flex items-start space-x-4">
                                    <Phone className="w-6 h-6 text-blue-400 mt-1" />
                                    <div>
                                        <h4 className="font-semibold text-white">
                                            Phone
                                        </h4>
                                        <p className="text-gray-400">
                                            09015452077
                                        </p>
                                    </div>
                                </div>

                                <div className="flex items-start space-x-4">
                                    <Mail className="w-6 h-6 text-blue-400 mt-1" />
                                    <div>
                                        <h4 className="font-semibold text-white">
                                            Email
                                        </h4>
                                        <p className="text-gray-400">
                                            HopaySub.com@gmail.com
                                        </p>
                                    </div>
                                </div>

                                <div className="flex items-start space-x-4">
                                    <Clock className="w-6 h-6 text-blue-400 mt-1" />
                                    <div>
                                        <h4 className="font-semibold text-white">
                                            Support Hours
                                        </h4>
                                        <p className="text-gray-400">
                                            24/7 Customer Support
                                        </p>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div className="bg-gray-800 rounded-2xl p-8">
                            <h3 className="text-2xl font-bold text-white mb-6">
                                Send us a Message
                            </h3>
                            <form className="space-y-6">
                                <div>
                                    <label className="block text-gray-300 text-sm font-medium mb-2">
                                        Name
                                    </label>
                                    <input
                                        type="text"
                                        className="w-full bg-gray-700 border border-gray-600 rounded-lg px-4 py-3 text-white focus:outline-none focus:border-blue-500 focus:ring-1 focus:ring-blue-500"
                                        placeholder="Your name"
                                    />
                                </div>

                                <div>
                                    <label className="block text-gray-300 text-sm font-medium mb-2">
                                        Email
                                    </label>
                                    <input
                                        type="email"
                                        className="w-full bg-gray-700 border border-gray-600 rounded-lg px-4 py-3 text-white focus:outline-none focus:border-blue-500 focus:ring-1 focus:ring-blue-500"
                                        placeholder="you@example.com"
                                    />
                                </div>

                                <div>
                                    <label className="block text-gray-300 text-sm font-medium mb-2">
                                        Message
                                    </label>
                                    <textarea
                                        rows={4}
                                        className="w-full bg-gray-700 border border-gray-600 rounded-lg px-4 py-3 text-white focus:outline-none focus:border-blue-500 focus:ring-1 focus:ring-blue-500"
                                        placeholder="Your message..."
                                    ></textarea>
                                </div>

                                <button
                                    type="submit"
                                    className="w-full bg-blue-600 text-white py-3 rounded-lg font-semibold hover:bg-blue-700 transition"
                                >
                                    Send Message
                                </button>
                            </form>
                        </div>
                    </div>
                </div>
            </section>

            {/* Footer */}
            <footer className="bg-gray-950 py-12">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <div className="grid md:grid-cols-4 gap-8">
                        <div>
                            <ApplicationLogo className="h-10 w-auto text-white mb-4" />
                            <p className="text-gray-400 text-sm">
                                Your trusted partner for all VTU services.
                                Experience seamless connectivity with Hopay Sub.
                            </p>
                        </div>

                        <div>
                            <h4 className="font-semibold text-white mb-4">
                                Quick Links
                            </h4>
                            <ul className="space-y-2">
                                {[
                                    "about",
                                    "services",
                                    "pricing",
                                    "agent",
                                    "contact",
                                ].map((section) => (
                                    <li key={section}>
                                        <button
                                            onClick={() =>
                                                scrollToSection(section)
                                            }
                                            className="text-gray-400 hover:text-white transition text-sm"
                                        >
                                            {section === "home"
                                                ? "Home"
                                                : section === "about"
                                                ? "About Us"
                                                : section === "agent"
                                                ? "Become Agent"
                                                : section
                                                      .charAt(0)
                                                      .toUpperCase() +
                                                  section.slice(1)}
                                        </button>
                                    </li>
                                ))}
                            </ul>
                        </div>

                        <div>
                            <h4 className="font-semibold text-white mb-4">
                                Services
                            </h4>
                            <ul className="space-y-2 text-sm text-gray-400">
                                <li>Airtime & Data</li>
                                <li>Bill Payments</li>
                                <li>Cable TV</li>
                                <li>Bulk SMS</li>
                                <li>Result Checker</li>
                            </ul>
                        </div>

                        <div>
                            <h4 className="font-semibold text-white mb-4">
                                Legal
                            </h4>
                            <ul className="space-y-2 text-sm text-gray-400">
                                <li>
                                    <Link
                                        href="#"
                                        className="hover:text-white transition"
                                    >
                                        Privacy Policy
                                    </Link>
                                </li>
                                <li>
                                    <Link
                                        href="#"
                                        className="hover:text-white transition"
                                    >
                                        Terms & Conditions
                                    </Link>
                                </li>
                                <li>
                                    <Link
                                        href="#"
                                        className="hover:text-white transition"
                                    >
                                        Cookie Policy
                                    </Link>
                                </li>
                            </ul>
                        </div>
                    </div>

                    <div className="border-t border-gray-800 mt-12 pt-8 flex flex-col md:flex-row justify-between items-center">
                        <p className="text-gray-500 text-sm text-center md:text-left">
                            Copyright © Hopay Sub {new Date().getFullYear()}.
                            All rights reserved.
                        </p>
                        <div className="flex space-x-4 mt-4 md:mt-0">
                            <a
                                href="#"
                                className="text-gray-500 hover:text-white transition"
                            >
                                FB
                            </a>
                            <a
                                href="#"
                                className="text-gray-500 hover:text-white transition"
                            >
                                TW
                            </a>
                            <a
                                href="#"
                                className="text-gray-500 hover:text-white transition"
                            >
                                IG
                            </a>
                            <a
                                href="#"
                                className="text-gray-500 hover:text-white transition"
                            >
                                WA
                            </a>
                        </div>
                    </div>
                </div>
            </footer>
        </>
    );
}
