import { Head, Link } from "@inertiajs/react";
import AppLayout from "@/Layouts/AppLayout";

interface WalletTransaction {
    id: number;
    reference: string;
    amount: number;
    type: "deposit" | "withdrawal" | "transfer" | "bonus" | "refund";
    status: string;
    payment_method: string;
    created_at: string;
}

interface PurchaseTransaction {
    id: number;
    reference: string;
    amount: number;
    type: string;
    status: string;
    network: string;
    phone: string;
    plan_description: string;
    created_at: string;
}

interface PageProps {
    walletTransactions: {
        data: WalletTransaction[];
        links: any[];
    };
    purchaseTransactions: {
        data: PurchaseTransaction[];
        links: any[];
    };
    wallet: {
        balance: number;
    };
}

export default function History({
    walletTransactions,
    purchaseTransactions,
    wallet,
}: PageProps) {
    const formatNumber = (amount: number) => {
        return new Intl.NumberFormat("en-NG", {
            minimumFractionDigits: 2,
            maximumFractionDigits: 2,
        }).format(amount);
    };

    const formatDate = (dateString: string) => {
        return new Date(dateString).toLocaleDateString("en-NG", {
            year: "numeric",
            month: "short",
            day: "numeric",
            hour: "2-digit",
            minute: "2-digit",
        });
    };

    return (
        <AppLayout pageTitle="Wallet History">
            <Head title="Wallet History" />

            <div className="p-6 space-y-8">
                {/* Header */}
                <div className="space text-left">
                    <h2 className="text-2xl font-bold">Wallet History</h2>
                    <p className="text-gray-600">
                        View your wallet and purchase transactions
                    </p>
                </div>

                {/* Current Balance */}
                <div className="bg-blue-50 border border-blue-200 rounded-xl p-6">
                    <div className="flex justify-between items-center">
                        <div>
                            <h3 className="font-bold text-blue-800 text-lg">
                                Current Balance
                            </h3>
                            <p className="text-3xl font-bold text-blue-800 mt-1">
                                ₦{formatNumber(wallet.balance)}
                            </p>
                        </div>
                        <Link
                            href={route("fund.wallet")}
                            className="bg-blue-600 hover:bg-blue-700 text-white font-semibold py-2 px-4 rounded-lg transition-all"
                        >
                            Fund Wallet
                        </Link>
                    </div>
                </div>

                {/* Tabs */}
                <div className="border-b border-gray-200">
                    <nav className="flex space-x-8">
                        <button className="border-b-2 border-blue-500 py-2 font-medium text-blue-600">
                            Wallet Transactions
                        </button>
                        <button className="py-2 font-medium text-gray-500 hover:text-gray-700">
                            Purchase History
                        </button>
                    </nav>
                </div>

                {/* Wallet Transactions Table */}
                <div className="bg-white rounded-xl border border-gray-200 shadow-sm">
                    <div className="p-6">
                        <h3 className="font-bold text-gray-800 text-lg mb-4">
                            Wallet Transactions
                        </h3>

                        {walletTransactions.data.length > 0 ? (
                            <div className="overflow-x-auto">
                                <table className="min-w-full divide-y divide-gray-200">
                                    <thead>
                                        <tr>
                                            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                Date
                                            </th>
                                            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                Reference
                                            </th>
                                            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                Type
                                            </th>
                                            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                Amount
                                            </th>
                                            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                Status
                                            </th>
                                            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                Method
                                            </th>
                                        </tr>
                                    </thead>
                                    <tbody className="divide-y divide-gray-200">
                                        {walletTransactions.data.map(
                                            (transaction) => (
                                                <tr
                                                    key={transaction.id}
                                                    className="hover:bg-gray-50"
                                                >
                                                    <td className="px-4 py-3 whitespace-nowrap text-sm text-gray-600">
                                                        {formatDate(
                                                            transaction.created_at
                                                        )}
                                                    </td>
                                                    <td className="px-4 py-3 whitespace-nowrap text-sm text-gray-900 font-mono">
                                                        {transaction.reference}
                                                    </td>
                                                    <td className="px-4 py-3 whitespace-nowrap">
                                                        <span
                                                            className={`
                                                        px-2 py-1 text-xs font-medium rounded-full
                                                        ${
                                                            transaction.type ===
                                                            "deposit"
                                                                ? "bg-green-100 text-green-800"
                                                                : "bg-red-100 text-red-800"
                                                        }
                                                    `}
                                                        >
                                                            {transaction.type}
                                                        </span>
                                                    </td>
                                                    <td className="px-4 py-3 whitespace-nowrap text-sm font-medium">
                                                        ₦
                                                        {formatNumber(
                                                            transaction.amount
                                                        )}
                                                    </td>
                                                    <td className="px-4 py-3 whitespace-nowrap">
                                                        <span
                                                            className={`
                                                        px-2 py-1 text-xs font-medium rounded-full
                                                        ${
                                                            transaction.status ===
                                                            "success"
                                                                ? "bg-green-100 text-green-800"
                                                                : transaction.status ===
                                                                  "pending"
                                                                ? "bg-yellow-100 text-yellow-800"
                                                                : "bg-red-100 text-red-800"
                                                        }
                                                    `}
                                                        >
                                                            {transaction.status}
                                                        </span>
                                                    </td>
                                                    <td className="px-4 py-3 whitespace-nowrap text-sm text-gray-600">
                                                        {
                                                            transaction.payment_method
                                                        }
                                                    </td>
                                                </tr>
                                            )
                                        )}
                                    </tbody>
                                </table>
                            </div>
                        ) : (
                            <div className="text-center py-12 text-gray-500">
                                <p>No wallet transactions found</p>
                            </div>
                        )}

                        {/* Pagination */}
                        {walletTransactions.links.length > 3 && (
                            <div className="mt-6 flex justify-center">
                                <nav className="flex items-center space-x-2">
                                    {walletTransactions.links.map(
                                        (link, index) => (
                                            <Link
                                                key={index}
                                                href={link.url || "#"}
                                                className={`
                                                px-3 py-1 rounded-md text-sm font-medium
                                                ${
                                                    link.active
                                                        ? "bg-blue-600 text-white"
                                                        : "text-gray-700 hover:bg-gray-100"
                                                }
                                                ${
                                                    !link.url
                                                        ? "opacity-50 cursor-not-allowed"
                                                        : ""
                                                }
                                            `}
                                                dangerouslySetInnerHTML={{
                                                    __html: link.label,
                                                }}
                                            />
                                        )
                                    )}
                                </nav>
                            </div>
                        )}
                    </div>
                </div>

                {/* Back to Wallet */}
                <div className="text-center">
                    <Link
                        href={route("fund.wallet")}
                        className="inline-flex items-center text-blue-600 hover:text-blue-700 font-medium"
                    >
                        <svg
                            className="w-4 h-4 mr-2"
                            fill="none"
                            stroke="currentColor"
                            viewBox="0 0 24 24"
                        >
                            <path
                                strokeLinecap="round"
                                strokeLinejoin="round"
                                strokeWidth="2"
                                d="M10 19l-7-7m0 0l7-7m-7 7h18"
                            />
                        </svg>
                        Back to Wallet
                    </Link>
                </div>
            </div>
        </AppLayout>
    );
}
