import { useState, useEffect } from "react";
import { Head, Link, router, usePage } from "@inertiajs/react";
import AppLayout from "@/Layouts/AppLayout";

interface UserShowData {
    id: number;
    name: string;
    email: string;
    kyc_status: string;
}

interface WalletShowData {
    balance: number;
    currency: string;
}

interface StaticAccountData {
    account_number?: string;
    bank_name?: string;
    account_name?: string;
}

interface CompanyAccountDetails {
    account_number: string;
    bank_name: string;
    account_name: string;
    notes?: string;
    active?: boolean;
}

// UPDATED: WalletTransaction interface (for deposits/withdrawals)
interface WalletTransaction {
    id: number;
    amount: number;
    type: "deposit" | "withdrawal" | "transfer" | "bonus" | "refund";
    status: string;
    description?: string;
    date: string;
    payment_method?: string;
}

// UPDATED: PurchaseTransaction interface (for service purchases)
interface PurchaseTransaction {
    id: number;
    amount: number;
    type: string;
    status: string;
    description?: string;
    date: string;
    network?: string;
    phone?: string;
    plan_description?: string;
}

interface PageProps {
    user: UserShowData;
    wallet: WalletShowData;
    paymentMethods: Array<{
        id: string;
        name: string;
        icon: string;
        requires_account_details?: boolean;
        requires_virtual_account?: boolean;
    }>;
    quickAmounts: number[];
    // UPDATED: Now only showing wallet transactions (deposits/withdrawals)
    walletHistory: WalletTransaction[];
    hasVirtualAccount: boolean;
    staticAccount: StaticAccountData | null;
    companyAccountDetails?: CompanyAccountDetails;
    flash?: {
        success?: string;
        error?: string;
    };
}

function Card({
    title,
    children,
    color = "blue",
}: {
    title: string;
    children: React.ReactNode;
    color?: string;
}) {
    const bgColor = {
        blue: "bg-blue-50",
        green: "bg-green-50",
        yellow: "bg-yellow-50",
        purple: "bg-purple-50",
        gray: "bg-gray-50",
    }[color];

    const borderColor = {
        blue: "border-blue-200",
        green: "border-green-200",
        yellow: "border-yellow-200",
        purple: "border-purple-200",
        gray: "border-gray-200",
    }[color];

    const textColor = {
        blue: "text-blue-800",
        green: "text-green-800",
        yellow: "text-yellow-800",
        purple: "text-purple-800",
        gray: "text-gray-800",
    }[color];

    return (
        <div
            className={`${bgColor} p-6 rounded-xl border ${borderColor} shadow-sm hover:shadow-md transition duration-200`}
        >
            <h3 className={`font-bold text-lg mb-2 ${textColor}`}>{title}</h3>
            {children}
        </div>
    );
}

export default function Fund({
    user,
    wallet,
    paymentMethods = [],
    quickAmounts = [],
    // UPDATED: Changed from transactionHistory to walletHistory
    walletHistory = [],
    hasVirtualAccount = false,
    staticAccount = null,
    companyAccountDetails = null,
}: PageProps) {
    const [selectedMethod, setSelectedMethod] = useState<string>("");
    const [selectedAmount, setSelectedAmount] = useState<number | null>(null);
    const [customAmount, setCustomAmount] = useState<string>("");
    const [customAmountError, setCustomAmountError] = useState<string>("");
    const [isProcessing, setIsProcessing] = useState(false);
    const [copyStatus, setCopyStatus] = useState<string>("");
    const [showCompanyDetails, setShowCompanyDetails] = useState(false);
    const [showVirtualAccountDetails, setShowVirtualAccountDetails] =
        useState(false);
    const [isPaymentMethodSelected, setIsPaymentMethodSelected] =
        useState(false);

    const { flash } = usePage<PageProps>().props;

    // Initialize with first quick amount
    useEffect(() => {
        if (quickAmounts.length > 0 && !selectedAmount) {
            setSelectedAmount(quickAmounts[0]);
            setCustomAmount(quickAmounts[0].toString());
        }
    }, [quickAmounts]);

    // Reset payment method selection when method is deselected
    useEffect(() => {
        if (!selectedMethod) {
            setIsPaymentMethodSelected(false);
        }
    }, [selectedMethod]);

    const formatCurrency = (amount: number | null) => {
        if (!amount) return "0.00";
        return new Intl.NumberFormat("en-NG", {
            minimumFractionDigits: 2,
            maximumFractionDigits: 2,
        }).format(amount);
    };

    const formatNumber = (amount: number | null) => {
        if (!amount) return "0";
        return amount.toLocaleString("en-NG");
    };

    const getSelectedMethodName = () => {
        if (!selectedMethod) return "Not selected";
        const method = paymentMethods.find((m) => m.id === selectedMethod);
        return method ? method.name : "Not selected";
    };

    const selectPaymentMethod = (methodId: string) => {
        setSelectedMethod(methodId);
        setIsPaymentMethodSelected(true);

        const method = paymentMethods.find((m) => m.id === methodId);
        if (
            method?.requires_account_details &&
            companyAccountDetails?.active !== false &&
            companyAccountDetails?.account_number
        ) {
            setShowCompanyDetails(true);
            setShowVirtualAccountDetails(false);
        } else if (method?.requires_virtual_account) {
            setShowVirtualAccountDetails(true);
            setShowCompanyDetails(false);
        } else {
            setShowCompanyDetails(false);
            setShowVirtualAccountDetails(false);
        }
    };

    const selectAmount = (amount: number) => {
        setSelectedAmount(amount);
        setCustomAmount(amount.toString());
        setCustomAmountError("");
    };

    const handleCustomAmount = (e: React.ChangeEvent<HTMLInputElement>) => {
        const value = e.target.value;
        setCustomAmount(value);

        const amount = parseInt(value) || 0;

        if (value === "") {
            setSelectedAmount(null);
            setCustomAmountError("");
        } else if (isNaN(amount)) {
            setCustomAmountError("Please enter a valid number");
            setSelectedAmount(null);
        } else if (amount < 100) {
            setCustomAmountError("Minimum amount is ₦100");
            setSelectedAmount(null);
        } else {
            setCustomAmountError("");
            setSelectedAmount(amount);
        }
    };

    const getCsrfToken = (): string => {
        const metaTag = document.querySelector('meta[name="csrf-token"]');
        if (metaTag) {
            return metaTag.getAttribute("content") || "";
        }

        if (window.Laravel?.csrfToken) {
            return window.Laravel.csrfToken;
        }

        const cookieMatch = document.cookie.match(/XSRF-TOKEN=([^;]+)/);
        if (cookieMatch) {
            return decodeURIComponent(cookieMatch[1]);
        }

        return "";
    };

    const proceedToPayment = async () => {
        if (!selectedMethod) {
            alert("Please select a payment method");
            return;
        }

        // For direct bank transfer, just show account details
        if (selectedMethod === "bank_transfer" && companyAccountDetails) {
            const amount = selectedAmount || 0;
            alert(
                `Please transfer ${
                    amount > 0 ? `₦${formatNumber(amount)} ` : ""
                }to the company account:\n\n` +
                    `Account Number: ${companyAccountDetails.account_number}\n` +
                    `Bank: ${companyAccountDetails.bank_name}\n` +
                    `Account Name: ${companyAccountDetails.account_name}\n\n` +
                    `Important: Use your registered name/email as depositor.\n` +
                    `Funds will reflect in 1-2 hours after confirmation.`
            );
            return;
        }

        // For virtual account, just show virtual account details
        if (selectedMethod === "virtual_account" && staticAccount) {
            const amount = selectedAmount || 0;
            alert(
                `Please transfer ${
                    amount > 0 ? `₦${formatNumber(amount)} ` : ""
                }to your virtual account:\n\n` +
                    `Account Number: ${staticAccount.account_number}\n` +
                    `Bank: ${staticAccount.bank_name}\n` +
                    `Account Name: ${staticAccount.account_name}\n\n` +
                    `Important: Transfer will reflect instantly in your wallet.`
            );
            return;
        }

        // For other payment methods, check amount
        if (!selectedAmount || selectedAmount < 100) {
            alert("Please enter a valid amount (minimum ₦100)");
            return;
        }

        setIsProcessing(true);

        try {
            const csrfToken = getCsrfToken();

            if (!csrfToken) {
                throw new Error(
                    "CSRF token not found. Please refresh the page."
                );
            }

            const requestBody = {
                amount: selectedAmount,
                payment_method: selectedMethod,
            };

            console.log("Sending deposit request:", requestBody);

            const response = await fetch(route("wallet.deposit.initialize"), {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                    Accept: "application/json",
                    "X-CSRF-TOKEN": csrfToken,
                    "X-Requested-With": "XMLHttpRequest",
                },
                credentials: "include",
                body: JSON.stringify(requestBody),
            });

            console.log("Response status:", response.status);

            if (response.status === 419) {
                alert(
                    "Your session has expired. Please refresh the page and try again."
                );
                window.location.reload();
                return;
            }

            if (response.status === 401 || response.status === 403) {
                alert("You need to be logged in to perform this action.");
                window.location.href = route("login");
                return;
            }

            if (response.status === 422) {
                const text = await response.text();
                console.log("Validation error response:", text);

                try {
                    const data = JSON.parse(text);
                    const errors =
                        data.errors || data.message || "Validation failed";

                    if (typeof errors === "object") {
                        const errorMessages = Object.values(errors).flat();
                        alert(`Validation Error:\n${errorMessages.join("\n")}`);
                    } else {
                        alert(`Validation Error: ${errors}`);
                    }
                } catch {
                    alert("Validation failed. Please check your input.");
                }

                return;
            }

            if (!response.ok) {
                throw new Error(`Server error: ${response.status}`);
            }

            const text = await response.text();
            let data;
            try {
                data = JSON.parse(text);
            } catch (parseError) {
                console.error("Failed to parse JSON:", parseError);
                throw new Error("Server returned an invalid response.");
            }

            if (data.success && data.paymentUrl) {
                console.log("Redirecting to payment URL");
                window.location.href = data.paymentUrl;
            } else if (data.success && data.bank_details) {
                alert(`Please transfer ₦${formatNumber(selectedAmount)} to:\n
                    Account Number: ${data.bank_details.account_number}\n
                    Bank: ${data.bank_details.bank_name}\n
                    Account Name: ${data.bank_details.account_name}\n\n
                    Reference: ${data.reference}`);
                router.reload();
            } else if (data.success) {
                alert(data.message || "Transaction initiated successfully!");
                router.reload();
            } else if (data.error) {
                alert(`Payment Error: ${data.error}`);
            } else {
                alert("Failed to initiate payment. Please try again.");
            }
        } catch (error: any) {
            console.error("Payment error:", error);
            alert(
                error.message || "Failed to initiate payment. Please try again."
            );
        } finally {
            setIsProcessing(false);
        }
    };

    const copyToClipboard = async (text: string | undefined) => {
        if (!text) return;

        try {
            await navigator.clipboard.writeText(text);
            setCopyStatus("Copied!");
            setTimeout(() => setCopyStatus(""), 2000);
        } catch (err) {
            console.error("Failed to copy:", err);
            setCopyStatus("Failed to copy");
            setTimeout(() => setCopyStatus(""), 2000);
        }
    };

    // Show flash messages
    useEffect(() => {
        if (flash?.success) {
            alert(flash.success);
        }
        if (flash?.error) {
            alert(flash.error);
        }
    }, [flash]);

    // Add null check for user
    if (!user || !wallet) {
        return (
            <AppLayout pageTitle="Fund Wallet">
                <Head title="Fund Wallet" />
                <div className="p-6">
                    <div className="text-center py-12">
                        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto"></div>
                        <p className="mt-4 text-gray-600">
                            Loading wallet information...
                        </p>
                    </div>
                </div>
            </AppLayout>
        );
    }

    // Helper function to get transaction description
    const getTransactionDescription = (transaction: WalletTransaction) => {
        if (transaction.description) return transaction.description;

        if (transaction.type === "deposit") {
            return `${
                transaction.payment_method
                    ? transaction.payment_method.charAt(0).toUpperCase() +
                      transaction.payment_method.slice(1)
                    : "Manual"
            } Deposit`;
        } else if (transaction.type === "withdrawal") {
            return `${
                transaction.payment_method
                    ? transaction.payment_method.charAt(0).toUpperCase() +
                      transaction.payment_method.slice(1)
                    : "Bank"
            } Withdrawal`;
        }

        return (
            transaction.type.charAt(0).toUpperCase() + transaction.type.slice(1)
        );
    };

    // Check which payment method is selected
    const isDirectBankTransfer = selectedMethod === "bank_transfer";
    const isVirtualAccount = selectedMethod === "virtual_account";
    const showAmountSelection = !isDirectBankTransfer && !isVirtualAccount;
    const showTransactionSummary = !isDirectBankTransfer && !isVirtualAccount;

    return (
        <AppLayout pageTitle="Fund Wallet">
            <Head title="Fund Wallet" />

            <div className="p-6 space-y-8">
                {/* Header */}
                <div className="space text-left">
                    <h2 className="text-2xl font-bold">Fund Your Wallet</h2>
                    <p className="text-gray-600">
                        Add money to your wallet using any payment method
                    </p>
                </div>

                {/* Wallet Balance Card - ALWAYS VISIBLE */}
                <Card title="Current Balance" color="blue">
                    <div className="flex justify-between items-center">
                        <div>
                            <p className="text-3xl font-bold text-blue-800 mt-1">
                                ₦{formatCurrency(wallet.balance)}
                            </p>
                            <p className="text-sm text-blue-600 mt-1">
                                {wallet.currency}
                            </p>
                        </div>
                        <div className="bg-blue-100 p-3 rounded-full">
                            <svg
                                className="w-8 h-8 text-blue-600"
                                fill="none"
                                stroke="currentColor"
                                viewBox="0 0 24 24"
                            >
                                <path
                                    strokeLinecap="round"
                                    strokeLinejoin="round"
                                    strokeWidth="2"
                                    d="M17 9V7a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2m2 4h10a2 2 0 002-2v-6a2 2 0 00-2-2H9a2 2 0 00-2 2v6a2 2 0 002 2zm7-5a2 2 0 11-4 0 2 2 0 014 0z"
                                ></path>
                            </svg>
                        </div>
                    </div>
                </Card>

                <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
                    {/* Left Column: Payment Methods & Amount */}
                    <div className="lg:col-span-2 space-y-6">
                        {/* Payment Methods */}
                        <Card title="Choose Payment Method" color="gray">
                            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                {paymentMethods
                                    .filter((method) => method.id !== "ussd") // Remove USSD payment method
                                    .map((method) => (
                                        <button
                                            key={method.id}
                                            onClick={() =>
                                                selectPaymentMethod(method.id)
                                            }
                                            className={`
                                                p-4 border rounded-lg transition-all duration-200 text-left
                                                ${
                                                    selectedMethod === method.id
                                                        ? "border-blue-500 bg-blue-50 ring-2 ring-blue-100"
                                                        : "border-gray-200 hover:border-blue-300 hover:bg-gray-50"
                                                }
                                                ${
                                                    (method.requires_account_details &&
                                                        (!companyAccountDetails?.active ||
                                                            !companyAccountDetails?.account_number)) ||
                                                    (method.requires_virtual_account &&
                                                        !hasVirtualAccount)
                                                        ? "opacity-50 cursor-not-allowed"
                                                        : ""
                                                }
                                            `}
                                            disabled={
                                                (method.requires_account_details &&
                                                    (!companyAccountDetails?.active ||
                                                        !companyAccountDetails?.account_number)) ||
                                                (method.requires_virtual_account &&
                                                    !hasVirtualAccount)
                                            }
                                        >
                                            <div className="flex items-center space-x-3">
                                                <div className="p-2 bg-blue-100 rounded-lg">
                                                    {method.icon === "Bank" && (
                                                        <svg
                                                            className="w-6 h-6 text-blue-600"
                                                            fill="none"
                                                            stroke="currentColor"
                                                            viewBox="0 0 24 24"
                                                        >
                                                            <path
                                                                strokeLinecap="round"
                                                                strokeLinejoin="round"
                                                                strokeWidth="2"
                                                                d="M3 6l3 1m0 0l-3 9a5.002 5.002 0 006.001 0M6 7l3 9M6 7l6-2m6 2l3-1m-3 1l-3 9a5.002 5.002 0 006.001 0M18 7l3 9m-3-9l-6-2m0-2v2m0 16V5m0 16H9m3 0h3"
                                                            ></path>
                                                        </svg>
                                                    )}
                                                    {method.icon ===
                                                        "CreditCard" && (
                                                        <svg
                                                            className="w-6 h-6 text-blue-600"
                                                            fill="none"
                                                            stroke="currentColor"
                                                            viewBox="0 0 24 24"
                                                        >
                                                            <path
                                                                strokeLinecap="round"
                                                                strokeLinejoin="round"
                                                                strokeWidth="2"
                                                                d="M3 10h18M7 15h1m4 0h1m-7 4h12a3 3 0 003-3V8a3 3 0 00-3-3H6a3 3 0 00-3 3v8a3 3 0 003 3z"
                                                            ></path>
                                                        </svg>
                                                    )}
                                                    {method.icon ===
                                                        "BuildingBank" && (
                                                        <svg
                                                            className="w-6 h-6 text-blue-600"
                                                            fill="none"
                                                            stroke="currentColor"
                                                            viewBox="0 0 24 24"
                                                        >
                                                            <path
                                                                strokeLinecap="round"
                                                                strokeLinejoin="round"
                                                                strokeWidth="2"
                                                                d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4"
                                                            ></path>
                                                        </svg>
                                                    )}
                                                </div>
                                                <div>
                                                    <div className="font-medium text-gray-900">
                                                        {method.name}
                                                    </div>
                                                    <div className="text-sm text-gray-500">
                                                        Instant funding
                                                    </div>
                                                </div>
                                            </div>
                                            {method.requires_account_details &&
                                                (!companyAccountDetails?.active ||
                                                    !companyAccountDetails?.account_number) && (
                                                    <div className="mt-2 text-xs text-red-500">
                                                        Temporarily unavailable
                                                    </div>
                                                )}
                                            {method.requires_virtual_account &&
                                                !hasVirtualAccount && (
                                                    <div className="mt-2 text-xs text-red-500">
                                                        Complete KYC to enable
                                                    </div>
                                                )}
                                        </button>
                                    ))}
                            </div>

                            {/* Company Account Details Section */}
                            {showCompanyDetails && companyAccountDetails && (
                                <div className="mt-6 p-4 bg-gray-50 border border-gray-200 rounded-lg animate-fade-in">
                                    <div className="flex justify-between items-center mb-4">
                                        <h4 className="font-bold text-gray-800 text-lg">
                                            Company Bank Account Details
                                        </h4>
                                        <span className="bg-green-100 text-green-800 text-xs font-medium px-3 py-1 rounded-full">
                                            Available 24/7
                                        </span>
                                    </div>

                                    <div className="space-y-4">
                                        {/* Account Details Grid */}
                                        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                                            <div className="bg-white p-4 rounded-lg border border-gray-200 shadow-sm">
                                                <div className="flex items-center space-x-2 mb-2">
                                                    <svg
                                                        className="w-5 h-5 text-blue-600"
                                                        fill="none"
                                                        stroke="currentColor"
                                                        viewBox="0 0 24 24"
                                                    >
                                                        <path
                                                            strokeLinecap="round"
                                                            strokeLinejoin="round"
                                                            strokeWidth="2"
                                                            d="M17 9V7a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2m2 4h10a2 2 0 002-2v-6a2 2 0 00-2-2H9a2 2 0 00-2 2v6a2 2 0 002 2zm7-5a2 2 0 11-4 0 2 2 0 014 0z"
                                                        ></path>
                                                    </svg>
                                                    <div className="text-xs text-gray-500 font-medium">
                                                        Account Number
                                                    </div>
                                                </div>
                                                <div className="font-bold text-xl text-gray-900 font-mono">
                                                    {
                                                        companyAccountDetails.account_number
                                                    }
                                                </div>
                                            </div>

                                            <div className="bg-white p-4 rounded-lg border border-gray-200 shadow-sm">
                                                <div className="flex items-center space-x-2 mb-2">
                                                    <svg
                                                        className="w-5 h-5 text-blue-600"
                                                        fill="none"
                                                        stroke="currentColor"
                                                        viewBox="0 0 24 24"
                                                    >
                                                        <path
                                                            strokeLinecap="round"
                                                            strokeLinejoin="round"
                                                            strokeWidth="2"
                                                            d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4"
                                                        ></path>
                                                    </svg>
                                                    <div className="text-xs text-gray-500 font-medium">
                                                        Bank Name
                                                    </div>
                                                </div>
                                                <div className="font-bold text-lg text-gray-900">
                                                    {
                                                        companyAccountDetails.bank_name
                                                    }
                                                </div>
                                            </div>

                                            <div className="bg-white p-4 rounded-lg border border-gray-200 shadow-sm">
                                                <div className="flex items-center space-x-2 mb-2">
                                                    <svg
                                                        className="w-5 h-5 text-blue-600"
                                                        fill="none"
                                                        stroke="currentColor"
                                                        viewBox="0 0 24 24"
                                                    >
                                                        <path
                                                            strokeLinecap="round"
                                                            strokeLinejoin="round"
                                                            strokeWidth="2"
                                                            d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"
                                                        ></path>
                                                    </svg>
                                                    <div className="text-xs text-gray-500 font-medium">
                                                        Account Name
                                                    </div>
                                                </div>
                                                <div className="font-bold text-lg text-gray-900">
                                                    {
                                                        companyAccountDetails.account_name
                                                    }
                                                </div>
                                            </div>
                                        </div>

                                        {/* Instructions */}
                                        {companyAccountDetails.notes && (
                                            <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                                                <div className="flex items-start">
                                                    <svg
                                                        className="w-5 h-5 text-blue-600 mt-0.5 mr-3 flex-shrink-0"
                                                        fill="none"
                                                        stroke="currentColor"
                                                        viewBox="0 0 24 24"
                                                    >
                                                        <path
                                                            strokeLinecap="round"
                                                            strokeLinejoin="round"
                                                            strokeWidth="2"
                                                            d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"
                                                        ></path>
                                                    </svg>
                                                    <div>
                                                        <div className="font-bold text-blue-800 mb-1">
                                                            Important
                                                            Instructions:
                                                        </div>
                                                        <div className="text-sm text-blue-700">
                                                            {
                                                                companyAccountDetails.notes
                                                            }
                                                        </div>
                                                        <div className="mt-2 text-xs text-blue-600">
                                                            After transferring,
                                                            please contact
                                                            support with your
                                                            transaction receipt
                                                            for faster
                                                            processing.
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        )}

                                        {/* Action Buttons */}
                                        <div className="flex flex-col sm:flex-row space-y-3 sm:space-y-0 sm:space-x-3">
                                            <button
                                                onClick={() =>
                                                    copyToClipboard(
                                                        companyAccountDetails.account_number
                                                    )
                                                }
                                                className="flex-1 bg-blue-600 hover:bg-blue-700 text-white font-semibold py-3 px-4 rounded-lg transition-all flex items-center justify-center space-x-2"
                                            >
                                                <svg
                                                    className="w-5 h-5"
                                                    fill="none"
                                                    stroke="currentColor"
                                                    viewBox="0 0 24 24"
                                                >
                                                    <path
                                                        strokeLinecap="round"
                                                        strokeLinejoin="round"
                                                        strokeWidth="2"
                                                        d="M8 16H6a2 2 0 01-2-2V6a2 2 0 012-2h8a2 2 0 012 2v2m-6 12h8a2 2 0 002-2v-8a2 2 0 00-2-2h-8a2 2 0 00-2 2v8a2 2 0 002 2z"
                                                    ></path>
                                                </svg>
                                                <span>
                                                    {copyStatus ||
                                                        "Copy Account Number"}
                                                </span>
                                            </button>

                                            <button
                                                onClick={() => {
                                                    const content = `COMPANY BANK ACCOUNT DETAILS
=======================================
Account Number: ${companyAccountDetails.account_number}
Bank Name: ${companyAccountDetails.bank_name}
Account Name: ${companyAccountDetails.account_name}

INSTRUCTIONS:
=======================================
${
    companyAccountDetails.notes ||
    "Use your registered name as depositor. Funds reflect in 1-2 hours after confirmation."
}

PROCESS:
1. Transfer any amount to this account
2. Use your registered name/email as depositor name
3. Keep your transaction receipt/reference
4. Funds will be verified and credited within 1-2 hours
5. Contact support if funds don't reflect within 24 hours

=======================================
Generated on: ${new Date().toLocaleDateString()}`;

                                                    const blob = new Blob(
                                                        [content],
                                                        { type: "text/plain" }
                                                    );
                                                    const url =
                                                        window.URL.createObjectURL(
                                                            blob
                                                        );
                                                    const a =
                                                        document.createElement(
                                                            "a"
                                                        );
                                                    a.href = url;
                                                    a.download =
                                                        "company-bank-details.txt";
                                                    document.body.appendChild(
                                                        a
                                                    );
                                                    a.click();
                                                    document.body.removeChild(
                                                        a
                                                    );
                                                    window.URL.revokeObjectURL(
                                                        url
                                                    );
                                                }}
                                                className="flex-1 bg-green-600 hover:bg-green-700 text-white font-semibold py-3 px-4 rounded-lg transition-all flex items-center justify-center space-x-2"
                                            >
                                                <svg
                                                    className="w-5 h-5"
                                                    fill="none"
                                                    stroke="currentColor"
                                                    viewBox="0 0 24 24"
                                                >
                                                    <path
                                                        strokeLinecap="round"
                                                        strokeLinejoin="round"
                                                        strokeWidth="2"
                                                        d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"
                                                    ></path>
                                                </svg>
                                                <span>Save Details</span>
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            )}

                            {/* Virtual Account Details Section */}
                            {showVirtualAccountDetails && (
                                <div className="mt-6 p-4 bg-gray-50 border border-gray-200 rounded-lg animate-fade-in">
                                    <div className="flex justify-between items-center mb-4">
                                        <h4 className="font-bold text-gray-800 text-lg">
                                            Your Virtual Account Details
                                        </h4>
                                        <span className="bg-blue-100 text-blue-800 text-xs font-medium px-3 py-1 rounded-full">
                                            Instant Funding
                                        </span>
                                    </div>

                                    {hasVirtualAccount && staticAccount ? (
                                        <div className="space-y-4">
                                            {/* Account Details Grid */}
                                            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                                                <div className="bg-white p-4 rounded-lg border border-gray-200 shadow-sm">
                                                    <div className="flex items-center space-x-2 mb-2">
                                                        <svg
                                                            className="w-5 h-5 text-blue-600"
                                                            fill="none"
                                                            stroke="currentColor"
                                                            viewBox="0 0 24 24"
                                                        >
                                                            <path
                                                                strokeLinecap="round"
                                                                strokeLinejoin="round"
                                                                strokeWidth="2"
                                                                d="M17 9V7a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2m2 4h10a2 2 0 002-2v-6a2 2 0 00-2-2H9a2 2 0 00-2 2v6a2 2 0 002 2zm7-5a2 2 0 11-4 0 2 2 0 014 0z"
                                                            ></path>
                                                        </svg>
                                                        <div className="text-xs text-gray-500 font-medium">
                                                            Account Number
                                                        </div>
                                                    </div>
                                                    <div className="font-bold text-xl text-gray-900 font-mono">
                                                        {
                                                            staticAccount.account_number
                                                        }
                                                    </div>
                                                </div>

                                                <div className="bg-white p-4 rounded-lg border border-gray-200 shadow-sm">
                                                    <div className="flex items-center space-x-2 mb-2">
                                                        <svg
                                                            className="w-5 h-5 text-blue-600"
                                                            fill="none"
                                                            stroke="currentColor"
                                                            viewBox="0 0 24 24"
                                                        >
                                                            <path
                                                                strokeLinecap="round"
                                                                strokeLinejoin="round"
                                                                strokeWidth="2"
                                                                d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4"
                                                            ></path>
                                                        </svg>
                                                        <div className="text-xs text-gray-500 font-medium">
                                                            Bank Name
                                                        </div>
                                                    </div>
                                                    <div className="font-bold text-lg text-gray-900">
                                                        {
                                                            staticAccount.bank_name
                                                        }
                                                    </div>
                                                </div>

                                                <div className="bg-white p-4 rounded-lg border border-gray-200 shadow-sm">
                                                    <div className="flex items-center space-x-2 mb-2">
                                                        <svg
                                                            className="w-5 h-5 text-blue-600"
                                                            fill="none"
                                                            stroke="currentColor"
                                                            viewBox="0 0 24 24"
                                                        >
                                                            <path
                                                                strokeLinecap="round"
                                                                strokeLinejoin="round"
                                                                strokeWidth="2"
                                                                d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"
                                                            ></path>
                                                        </svg>
                                                        <div className="text-xs text-gray-500 font-medium">
                                                            Account Name
                                                        </div>
                                                    </div>
                                                    <div className="font-bold text-lg text-gray-900">
                                                        {
                                                            staticAccount.account_name
                                                        }
                                                    </div>
                                                </div>
                                            </div>

                                            {/* Instructions */}
                                            <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                                                <div className="flex items-start">
                                                    <svg
                                                        className="w-5 h-5 text-blue-600 mt-0.5 mr-3 flex-shrink-0"
                                                        fill="none"
                                                        stroke="currentColor"
                                                        viewBox="0 0 24 24"
                                                    >
                                                        <path
                                                            strokeLinecap="round"
                                                            strokeLinejoin="round"
                                                            strokeWidth="2"
                                                            d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"
                                                        ></path>
                                                    </svg>
                                                    <div>
                                                        <div className="font-bold text-blue-800 mb-1">
                                                            Important
                                                            Instructions:
                                                        </div>
                                                        <div className="text-sm text-blue-700">
                                                            Transfer any amount
                                                            to this account and
                                                            it will reflect
                                                            instantly in your
                                                            wallet.
                                                        </div>
                                                        <div className="mt-2 text-xs text-blue-600">
                                                            This is your
                                                            dedicated virtual
                                                            account for instant
                                                            wallet funding.
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>

                                            {/* Action Buttons */}
                                            <div className="flex flex-col sm:flex-row space-y-3 sm:space-y-0 sm:space-x-3">
                                                <button
                                                    onClick={() =>
                                                        copyToClipboard(
                                                            staticAccount.account_number
                                                        )
                                                    }
                                                    className="flex-1 bg-blue-600 hover:bg-blue-700 text-white font-semibold py-3 px-4 rounded-lg transition-all flex items-center justify-center space-x-2"
                                                >
                                                    <svg
                                                        className="w-5 h-5"
                                                        fill="none"
                                                        stroke="currentColor"
                                                        viewBox="0 0 24 24"
                                                    >
                                                        <path
                                                            strokeLinecap="round"
                                                            strokeLinejoin="round"
                                                            strokeWidth="2"
                                                            d="M8 16H6a2 2 0 01-2-2V6a2 2 0 012-2h8a2 2 0 012 2v2m-6 12h8a2 2 0 002-2v-8a2 2 0 00-2-2h-8a2 2 0 00-2 2v8a2 2 0 002 2z"
                                                        ></path>
                                                    </svg>
                                                    <span>
                                                        {copyStatus ||
                                                            "Copy Account Number"}
                                                    </span>
                                                </button>

                                                <button
                                                    onClick={() => {
                                                        const content = `YOUR VIRTUAL ACCOUNT DETAILS
=======================================
Account Number: ${staticAccount.account_number}
Bank Name: ${staticAccount.bank_name}
Account Name: ${staticAccount.account_name}

INSTRUCTIONS:
=======================================
Transfer any amount to this account and it will reflect instantly in your wallet.

This is your dedicated virtual account for instant wallet funding.

PROCESS:
1. Transfer any amount to this account
2. Funds will reflect instantly in your wallet
3. No need to contact support for transfers
4. Use as your personal funding account

=======================================
Generated on: ${new Date().toLocaleDateString()}`;

                                                        const blob = new Blob(
                                                            [content],
                                                            {
                                                                type: "text/plain",
                                                            }
                                                        );
                                                        const url =
                                                            window.URL.createObjectURL(
                                                                blob
                                                            );
                                                        const a =
                                                            document.createElement(
                                                                "a"
                                                            );
                                                        a.href = url;
                                                        a.download =
                                                            "virtual-account-details.txt";
                                                        document.body.appendChild(
                                                            a
                                                        );
                                                        a.click();
                                                        document.body.removeChild(
                                                            a
                                                        );
                                                        window.URL.revokeObjectURL(
                                                            url
                                                        );
                                                    }}
                                                    className="flex-1 bg-green-600 hover:bg-green-700 text-white font-semibold py-3 px-4 rounded-lg transition-all flex items-center justify-center space-x-2"
                                                >
                                                    <svg
                                                        className="w-5 h-5"
                                                        fill="none"
                                                        stroke="currentColor"
                                                        viewBox="0 0 24 24"
                                                    >
                                                        <path
                                                            strokeLinecap="round"
                                                            strokeLinejoin="round"
                                                            strokeWidth="2"
                                                            d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"
                                                        ></path>
                                                    </svg>
                                                    <span>Save Details</span>
                                                </button>
                                            </div>
                                        </div>
                                    ) : (
                                        <div className="text-center py-8">
                                            <svg
                                                className="w-16 h-16 text-gray-300 mx-auto mb-4"
                                                fill="none"
                                                stroke="currentColor"
                                                viewBox="0 0 24 24"
                                            >
                                                <path
                                                    strokeLinecap="round"
                                                    strokeLinejoin="round"
                                                    strokeWidth="2"
                                                    d="M9 12l2 2 4-4m5.618-4.016A11.955 11.955 0 0112 2.944a11.955 11.955 0 01-8.618 3.04A12.02 12.02 0 003 9c0 5.591 3.824 10.29 9 11.622 5.176-1.332 9-6.03 9-11.622 0-1.042-.133-2.052-.382-3.016z"
                                                ></path>
                                            </svg>
                                            <h4 className="font-bold text-gray-800 text-xl mb-2">
                                                No Linked Account Found
                                            </h4>
                                            <p className="text-gray-600 mb-6 max-w-md mx-auto">
                                                Complete your KYC using BVN or
                                                NIN to generate your virtual
                                                account.
                                            </p>
                                            <Link
                                                href={route("profile.edit")}
                                                className="inline-block bg-blue-600 hover:bg-blue-700 text-white font-semibold py-3 px-6 rounded-lg transition-all"
                                            >
                                                Ready to complete your KYC?
                                                Click here to get started.
                                            </Link>
                                        </div>
                                    )}
                                </div>
                            )}
                        </Card>

                        {/* Show placeholder message when no payment method is selected */}
                        {!isPaymentMethodSelected && (
                            <Card
                                title="Ready to Fund Your Wallet?"
                                color="blue"
                            >
                                <div className="text-center py-8">
                                    <svg
                                        className="w-16 h-16 text-blue-300 mx-auto mb-4"
                                        fill="none"
                                        stroke="currentColor"
                                        viewBox="0 0 24 24"
                                    >
                                        <path
                                            strokeLinecap="round"
                                            strokeLinejoin="round"
                                            strokeWidth="2"
                                            d="M9 5l7 7-7 7"
                                        ></path>
                                    </svg>
                                    <h4 className="font-bold text-gray-800 text-xl mb-2">
                                        Select a Payment Method
                                    </h4>
                                    <p className="text-gray-600 max-w-md mx-auto">
                                        Choose your preferred payment method
                                        above to continue. Once selected, you'll
                                        see amount options and transaction
                                        details here.
                                    </p>
                                    <div className="mt-6 flex justify-center space-x-2">
                                        <div className="w-2 h-2 bg-blue-400 rounded-full"></div>
                                        <div className="w-2 h-2 bg-blue-300 rounded-full"></div>
                                        <div className="w-2 h-2 bg-blue-200 rounded-full"></div>
                                    </div>
                                </div>
                            </Card>
                        )}

                        {/* Only show amount selection when a payment method is selected AND it's not bank transfer or virtual account */}
                        {isPaymentMethodSelected && showAmountSelection && (
                            <Card title="Select Amount" color="gray">
                                {/* Quick Amounts */}
                                <div className="mb-6">
                                    <div className="text-sm text-gray-600 mb-3">
                                        Quick Select
                                    </div>
                                    <div className="grid grid-cols-2 sm:grid-cols-3 gap-3">
                                        {quickAmounts.map((amount) => (
                                            <button
                                                key={amount}
                                                onClick={() =>
                                                    selectAmount(amount)
                                                }
                                                className={`
                                                    py-3 px-4 border rounded-lg text-center transition-all font-medium
                                                    ${
                                                        selectedAmount ===
                                                        amount
                                                            ? "border-blue-500 bg-blue-50 text-blue-700"
                                                            : "border-gray-200 hover:border-blue-300 hover:bg-gray-50 text-gray-700"
                                                    }
                                                `}
                                            >
                                                ₦{formatNumber(amount)}
                                            </button>
                                        ))}
                                    </div>
                                </div>

                                {/* Custom Amount */}
                                <div>
                                    <label
                                        htmlFor="customAmount"
                                        className="block text-sm font-medium text-gray-700 mb-2"
                                    >
                                        Or Enter Custom Amount
                                    </label>
                                    <div className="relative">
                                        <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                            <span className="text-gray-500 font-medium">
                                                ₦
                                            </span>
                                        </div>
                                        <input
                                            type="number"
                                            id="customAmount"
                                            value={customAmount}
                                            onChange={handleCustomAmount}
                                            placeholder="0.00"
                                            className={`
                                                pl-8 block w-full rounded-lg shadow-sm focus:ring-blue-500 focus:border-blue-500 py-3 pr-3
                                                ${
                                                    customAmountError
                                                        ? "border-red-500"
                                                        : "border-gray-300"
                                                }
                                            `}
                                            min="100"
                                            step="100"
                                        />
                                    </div>
                                    {customAmountError && (
                                        <p className="mt-1 text-sm text-red-500">
                                            {customAmountError}
                                        </p>
                                    )}
                                    <p className="mt-2 text-sm text-gray-500">
                                        Minimum amount: ₦100
                                    </p>
                                </div>
                            </Card>
                        )}

                        {/* Transaction Summary - Only show when a payment method is selected AND it's not bank transfer or virtual account */}
                        {isPaymentMethodSelected && showTransactionSummary && (
                            <Card title="Transaction Summary" color="gray">
                                <div className="space-y-3">
                                    <div className="flex justify-between">
                                        <span className="text-gray-600">
                                            Amount to fund:
                                        </span>
                                        <span className="font-medium">
                                            ₦{formatNumber(selectedAmount)}
                                        </span>
                                    </div>
                                    <div className="flex justify-between">
                                        <span className="text-gray-600">
                                            Payment method:
                                        </span>
                                        <span className="font-medium">
                                            {getSelectedMethodName()}
                                        </span>
                                    </div>
                                    <div className="border-t pt-3">
                                        <div className="flex justify-between text-lg font-bold">
                                            <span>Total to pay:</span>
                                            <span className="text-green-600">
                                                ₦{formatNumber(selectedAmount)}
                                            </span>
                                        </div>
                                    </div>
                                </div>

                                <button
                                    onClick={proceedToPayment}
                                    disabled={
                                        isProcessing ||
                                        !selectedMethod ||
                                        !selectedAmount ||
                                        selectedAmount < 100
                                    }
                                    className={`
                                        w-full mt-6 py-3 px-4 rounded-lg font-semibold transition-colors flex items-center justify-center space-x-2
                                        ${
                                            isProcessing ||
                                            !selectedMethod ||
                                            !selectedAmount ||
                                            selectedAmount < 100
                                                ? "bg-gray-300 cursor-not-allowed text-gray-500"
                                                : "bg-blue-600 hover:bg-blue-700 text-white"
                                        }
                                    `}
                                >
                                    {isProcessing ? (
                                        <>
                                            <svg
                                                className="animate-spin -ml-1 mr-2 h-5 w-5 text-white"
                                                xmlns="http://www.w3.org/2000/svg"
                                                fill="none"
                                                viewBox="0 0 24 24"
                                            >
                                                <circle
                                                    className="opacity-25"
                                                    cx="12"
                                                    cy="12"
                                                    r="10"
                                                    stroke="currentColor"
                                                    strokeWidth="4"
                                                ></circle>
                                                <path
                                                    className="opacity-75"
                                                    fill="currentColor"
                                                    d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"
                                                ></path>
                                            </svg>
                                            Processing...
                                        </>
                                    ) : (
                                        <>
                                            <svg
                                                className="w-5 h-5"
                                                fill="none"
                                                stroke="currentColor"
                                                viewBox="0 0 24 24"
                                            >
                                                <path
                                                    strokeLinecap="round"
                                                    strokeLinejoin="round"
                                                    strokeWidth="2"
                                                    d="M14 5l7 7m0 0l-7 7m7-7H3"
                                                ></path>
                                            </svg>
                                            <span>Proceed to Payment</span>
                                        </>
                                    )}
                                </button>
                            </Card>
                        )}

                        {/* Show alternative button for direct bank transfer */}
                        {isDirectBankTransfer && (
                            <Card title="Direct Bank Transfer" color="green">
                                <div className="space-y-4">
                                    <div className="bg-green-50 border border-green-200 rounded-lg p-4">
                                        <div className="flex items-start">
                                            <svg
                                                className="w-6 h-6 text-green-600 mt-0.5 mr-3 flex-shrink-0"
                                                fill="none"
                                                stroke="currentColor"
                                                viewBox="0 0 24 24"
                                            >
                                                <path
                                                    strokeLinecap="round"
                                                    strokeLinejoin="round"
                                                    strokeWidth="2"
                                                    d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"
                                                ></path>
                                            </svg>
                                            <div>
                                                <h4 className="font-bold text-green-800 text-lg mb-2">
                                                    How to Fund via Direct Bank
                                                    Transfer
                                                </h4>
                                                <ol className="list-decimal list-inside space-y-2 text-green-700">
                                                    <li>
                                                        Copy the company account
                                                        details above
                                                    </li>
                                                    <li>
                                                        Transfer any amount from
                                                        your bank app
                                                    </li>
                                                    <li>
                                                        Use your registered
                                                        name/email as depositor
                                                    </li>
                                                    <li>
                                                        Keep your transaction
                                                        receipt
                                                    </li>
                                                    <li>
                                                        Contact support with
                                                        receipt for faster
                                                        processing
                                                    </li>
                                                </ol>
                                            </div>
                                        </div>
                                    </div>

                                    <button
                                        onClick={proceedToPayment}
                                        className="w-full bg-green-600 hover:bg-green-700 text-white font-semibold py-3 px-4 rounded-lg transition-all flex items-center justify-center space-x-2"
                                    >
                                        <svg
                                            className="w-5 h-5"
                                            fill="none"
                                            stroke="currentColor"
                                            viewBox="0 0 24 24"
                                        >
                                            <path
                                                strokeLinecap="round"
                                                strokeLinejoin="round"
                                                strokeWidth="2"
                                                d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"
                                            ></path>
                                        </svg>
                                        <span>Show Transfer Instructions</span>
                                    </button>
                                </div>
                            </Card>
                        )}

                        {/* Show alternative button for virtual account */}
                        {isVirtualAccount &&
                            hasVirtualAccount &&
                            staticAccount && (
                                <Card
                                    title="Virtual Account Transfer"
                                    color="blue"
                                >
                                    <div className="space-y-4">
                                        <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                                            <div className="flex items-start">
                                                <svg
                                                    className="w-6 h-6 text-blue-600 mt-0.5 mr-3 flex-shrink-0"
                                                    fill="none"
                                                    stroke="currentColor"
                                                    viewBox="0 0 24 24"
                                                >
                                                    <path
                                                        strokeLinecap="round"
                                                        strokeLinejoin="round"
                                                        strokeWidth="2"
                                                        d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"
                                                    ></path>
                                                </svg>
                                                <div>
                                                    <h4 className="font-bold text-blue-800 text-lg mb-2">
                                                        How to Fund via Virtual
                                                        Account
                                                    </h4>
                                                    <ol className="list-decimal list-inside space-y-2 text-blue-700">
                                                        <li>
                                                            Copy your virtual
                                                            account details
                                                            above
                                                        </li>
                                                        <li>
                                                            Transfer any amount
                                                            from your bank app
                                                        </li>
                                                        <li>
                                                            Funds will reflect
                                                            instantly in your
                                                            wallet
                                                        </li>
                                                        <li>
                                                            No need to contact
                                                            support
                                                        </li>
                                                        <li>
                                                            Use this account for
                                                            all future deposits
                                                        </li>
                                                    </ol>
                                                </div>
                                            </div>
                                        </div>

                                        <button
                                            onClick={proceedToPayment}
                                            className="w-full bg-blue-600 hover:bg-blue-700 text-white font-semibold py-3 px-4 rounded-lg transition-all flex items-center justify-center space-x-2"
                                        >
                                            <svg
                                                className="w-5 h-5"
                                                fill="none"
                                                stroke="currentColor"
                                                viewBox="0 0 24 24"
                                            >
                                                <path
                                                    strokeLinecap="round"
                                                    strokeLinejoin="round"
                                                    strokeWidth="2"
                                                    d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"
                                                ></path>
                                            </svg>
                                            <span>
                                                Show Transfer Instructions
                                            </span>
                                        </button>
                                    </div>
                                </Card>
                            )}
                    </div>

                    {/* Right Column: Recent Wallet Transactions & Help - ALWAYS VISIBLE */}
                    <div className="space-y-6">
                        {/* Recent Wallet Transactions */}
                        <Card title="Recent Wallet Activity" color="gray">
                            {walletHistory.length > 0 ? (
                                <div className="space-y-4">
                                    {walletHistory.map((transaction) => (
                                        <div
                                            key={transaction.id}
                                            className={`
                                                border-l-4 pl-4 py-2
                                                ${
                                                    transaction.type ===
                                                    "deposit"
                                                        ? "border-l-green-500 bg-green-50"
                                                        : transaction.type ===
                                                          "withdrawal"
                                                        ? "border-l-red-500 bg-red-50"
                                                        : "border-l-blue-500 bg-blue-50"
                                                }
                                            `}
                                        >
                                            <div className="flex justify-between items-start">
                                                <div>
                                                    <div className="font-bold text-gray-900">
                                                        ₦
                                                        {formatNumber(
                                                            transaction.amount
                                                        )}
                                                    </div>
                                                    <div className="text-sm text-gray-500">
                                                        {getTransactionDescription(
                                                            transaction
                                                        )}
                                                        {transaction.payment_method && (
                                                            <span className="ml-2 text-xs bg-gray-100 text-gray-600 px-1.5 py-0.5 rounded">
                                                                {
                                                                    transaction.payment_method
                                                                }
                                                            </span>
                                                        )}
                                                    </div>
                                                </div>
                                                <div className="text-right">
                                                    <div
                                                        className={`
                                                        text-xs font-medium px-2 py-1 rounded-full
                                                        ${
                                                            transaction.status ===
                                                            "success"
                                                                ? "bg-green-100 text-green-800"
                                                                : transaction.status ===
                                                                  "pending"
                                                                ? "bg-yellow-100 text-yellow-800"
                                                                : "bg-red-100 text-red-800"
                                                        }
                                                    `}
                                                    >
                                                        {transaction.status}
                                                    </div>
                                                    <div className="text-xs text-gray-500 mt-1">
                                                        {transaction.date}
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    ))}
                                </div>
                            ) : (
                                <div className="text-center py-8 text-gray-500">
                                    <svg
                                        className="w-12 h-12 mx-auto text-gray-300 mb-3"
                                        fill="none"
                                        stroke="currentColor"
                                        viewBox="0 0 24 24"
                                    >
                                        <path
                                            strokeLinecap="round"
                                            strokeLinejoin="round"
                                            strokeWidth="2"
                                            d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2"
                                        ></path>
                                    </svg>
                                    <p>No recent wallet activity</p>
                                </div>
                            )}

                            <Link
                                href={route("wallet.history")}
                                className="mt-4 block text-center text-blue-600 hover:text-blue-700 font-medium text-sm"
                            >
                                View all wallet transactions →
                            </Link>
                        </Card>

                        {/* Help Section - Updated for different payment methods */}
                        <Card title="Need Help?" color="yellow">
                            <ul className="space-y-3">
                                <li className="flex items-start space-x-3">
                                    <svg
                                        className="w-5 h-5 text-blue-600 mt-0.5"
                                        fill="none"
                                        stroke="currentColor"
                                        viewBox="0 0 24 24"
                                    >
                                        <path
                                            strokeLinecap="round"
                                            strokeLinejoin="round"
                                            strokeWidth="2"
                                            d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"
                                        ></path>
                                    </svg>
                                    <span className="text-sm text-gray-700">
                                        {isDirectBankTransfer
                                            ? "Bank transfers process in 1-2 hours"
                                            : isVirtualAccount
                                            ? "Virtual account transfers are instant"
                                            : "Funds reflect instantly for online payments"}
                                    </span>
                                </li>
                                <li className="flex items-start space-x-3">
                                    <svg
                                        className="w-5 h-5 text-blue-600 mt-0.5"
                                        fill="none"
                                        stroke="currentColor"
                                        viewBox="0 0 24 24"
                                    >
                                        <path
                                            strokeLinecap="round"
                                            strokeLinejoin="round"
                                            strokeWidth="2"
                                            d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"
                                        ></path>
                                    </svg>
                                    <span className="text-sm text-gray-700">
                                        {isDirectBankTransfer
                                            ? "Always use your registered name as depositor"
                                            : isVirtualAccount
                                            ? "No need to contact support for transfers"
                                            : "Contact support for failed transactions"}
                                    </span>
                                </li>
                                <li className="flex items-start space-x-3">
                                    <svg
                                        className="w-5 h-5 text-blue-600 mt-0.5"
                                        fill="none"
                                        stroke="currentColor"
                                        viewBox="0 0 24 24"
                                    >
                                        <path
                                            strokeLinecap="round"
                                            strokeLinejoin="round"
                                            strokeWidth="2"
                                            d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"
                                        ></path>
                                    </svg>
                                    <span className="text-sm text-gray-700">
                                        {isDirectBankTransfer
                                            ? "Keep transaction receipt for verification"
                                            : isVirtualAccount
                                            ? "Transfer any amount, no minimum required"
                                            : "Minimum deposit amount is ₦100"}
                                    </span>
                                </li>
                            </ul>
                            <button className="mt-4 text-blue-600 hover:text-blue-700 text-sm font-medium">
                                <svg
                                    className="w-4 h-4 inline mr-1"
                                    fill="none"
                                    stroke="currentColor"
                                    viewBox="0 0 24 24"
                                >
                                    <path
                                        strokeLinecap="round"
                                        strokeLinejoin="round"
                                        strokeWidth="2"
                                        d="M8 12h.01M12 12h.01M16 12h.01M21 12c0 4.418-4.03 8-9 8a9.863 9.863 0 01-4.255-.949L3 20l1.395-3.72C3.512 15.042 3 13.574 3 12c0-4.418 4.03-8 9-8s9 3.582 9 8z"
                                    ></path>
                                </svg>
                                Contact Support
                            </button>
                        </Card>
                    </div>
                </div>
            </div>
        </AppLayout>
    );
}
