import AppLayout from "@/Layouts/AppLayout";
import { Head, Link } from "@inertiajs/react";
import { useState } from "react";

interface Transaction {
    id: number;
    reference: string;
    type: string;
    amount: number;
    amount_charged: number;
    status: string;
    network?: string;
    phone?: string;
    plan_description?: string;
    plan_code?: string;
    created_at: string;
    is_test?: boolean;
    remark?: string;
    order_id?: string;
}

interface PaginatedData {
    data: Transaction[];
    links: Array<{
        url: string | null;
        label: string;
        active: boolean;
    }>;
    current_page: number;
    last_page: number;
    per_page: number;
    total: number;
}

interface TransactionsIndexProps {
    transactions: PaginatedData;
}

export default function TransactionsIndex({
    transactions,
}: TransactionsIndexProps) {
    const transactionList = transactions?.data || [];
    const [selectedTransaction, setSelectedTransaction] =
        useState<Transaction | null>(null);
    const [showDetails, setShowDetails] = useState(false);

    // Helper function to get product description
    const getProductDescription = (transaction: Transaction): string => {
        if (transaction.type === "airtime") {
            return (
                transaction.plan_description ||
                `${transaction.amount} Credit` ||
                "Airtime Purchase"
            );
        } else if (transaction.type === "data") {
            return transaction.plan_description || "Data Bundle";
        } else if (transaction.type === "deposit") {
            return "Deposit Fund";
        } else {
            return transaction.plan_description || "Purchase";
        }
    };

    // Helper function to get transaction type (debit/credit)
    const getTransactionTypeDisplay = (
        transaction: Transaction
    ): { text: string; color: string } => {
        if (transaction.type === "deposit") {
            return {
                text: "Credit",
                color: "text-green-600 bg-green-50",
            };
        } else {
            return {
                text: "Debit",
                color: "text-red-600 bg-red-50",
            };
        }
    };

    // Helper function to get display amount
    const getDisplayAmount = (transaction: Transaction): number => {
        if (transaction.type === "deposit") {
            return transaction.amount; // For deposits, show original amount
        } else {
            return transaction.amount_charged || transaction.amount; // For purchases, show charged amount
        }
    };

    const getStatusColor = (status: string) => {
        switch (status.toLowerCase()) {
            case "success":
            case "completed":
                return "bg-green-100 text-green-800";
            case "failed":
                return "bg-red-100 text-red-800";
            case "pending":
                return "bg-yellow-100 text-yellow-800";
            default:
                return "bg-gray-100 text-gray-800";
        }
    };

    const formatDate = (date: string) => {
        try {
            return new Date(date)
                .toLocaleDateString("en-GB", {
                    day: "numeric",
                    month: "numeric",
                    year: "numeric",
                    hour: "2-digit",
                    minute: "2-digit",
                    hour12: true,
                })
                .replace(",", "");
        } catch (error) {
            return date;
        }
    };

    const formatCurrency = (amount: number | undefined) => {
        if (amount === undefined || amount === null) return "-";
        return new Intl.NumberFormat("en-NG", {
            style: "currency",
            currency: "NGN",
            minimumFractionDigits: 2,
            maximumFractionDigits: 2,
        }).format(amount);
    };

    const handleViewDetails = (transaction: Transaction) => {
        setSelectedTransaction(transaction);
        setShowDetails(true);
    };

    const handleDownloadReceipt = (transaction: Transaction) => {
        const link = document.createElement("a");
        link.href = `/transactions/${transaction.id}/receipt`;
        link.download = `receipt-${transaction.reference}.pdf`;
        link.target = "_blank";
        link.style.display = "none";

        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
    };

    return (
        <AppLayout pageTitle="Transaction History">
            <div className="py-6">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    <div className="bg-white overflow-hidden shadow-sm sm:rounded-lg">
                        <div className="p-6">
                            {/* Header */}
                            <div className="flex justify-between items-center mb-6">
                                <div>
                                    <h1 className="text-2xl font-bold text-gray-900">
                                        Transaction History
                                    </h1>
                                    <p className="text-sm text-gray-600 mt-1">
                                        View all your transactions
                                    </p>
                                </div>
                                <div className="text-sm text-gray-500">
                                    Total: {transactions.total} transactions
                                </div>
                            </div>

                            {/* Transaction Table */}
                            {transactionList.length === 0 ? (
                                <div className="text-center py-10">
                                    <div className="mb-4">
                                        <svg
                                            className="w-16 h-16 text-gray-300 mx-auto"
                                            fill="none"
                                            stroke="currentColor"
                                            viewBox="0 0 24 24"
                                        >
                                            <path
                                                strokeLinecap="round"
                                                strokeLinejoin="round"
                                                strokeWidth="1"
                                                d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2"
                                            />
                                        </svg>
                                    </div>
                                    <p className="text-gray-500 text-lg mb-2">
                                        No transactions yet
                                    </p>
                                    <p className="text-gray-400 text-sm mb-4">
                                        Your transaction history will appear
                                        here
                                    </p>
                                    <div className="flex gap-3 justify-center">
                                        <Link
                                            href="/buy-airtime"
                                            className="inline-flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition"
                                        >
                                            Buy Airtime
                                        </Link>
                                        <Link
                                            href="/deposit"
                                            className="inline-flex items-center px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition"
                                        >
                                            Deposit Funds
                                        </Link>
                                    </div>
                                </div>
                            ) : (
                                <>
                                    <div className="overflow-x-auto border border-gray-200 rounded-lg">
                                        <table className="min-w-full divide-y divide-gray-200">
                                            <thead className="bg-gray-50">
                                                <tr>
                                                    <th className="px-4 py-3 text-left text-xs font-medium text-gray-700 uppercase tracking-wider border-r border-gray-200">
                                                        Transaction Date
                                                    </th>
                                                    <th className="px-4 py-3 text-left text-xs font-medium text-gray-700 uppercase tracking-wider border-r border-gray-200">
                                                        Transaction Status
                                                    </th>
                                                    <th className="px-4 py-3 text-left text-xs font-medium text-gray-700 uppercase tracking-wider border-r border-gray-200">
                                                        Transaction ID
                                                    </th>
                                                    <th className="px-4 py-3 text-left text-xs font-medium text-gray-700 uppercase tracking-wider border-r border-gray-200">
                                                        Mobile Network
                                                    </th>
                                                    <th className="px-4 py-3 text-left text-xs font-medium text-gray-700 uppercase tracking-wider border-r border-gray-200">
                                                        Product Description
                                                    </th>
                                                    <th className="px-4 py-3 text-left text-xs font-medium text-gray-700 uppercase tracking-wider border-r border-gray-200">
                                                        Mobile Number
                                                    </th>
                                                    <th className="px-4 py-3 text-left text-xs font-medium text-gray-700 uppercase tracking-wider border-r border-gray-200">
                                                        Amount (₦)
                                                    </th>
                                                    <th className="px-4 py-3 text-left text-xs font-medium text-gray-700 uppercase tracking-wider border-r border-gray-200">
                                                        Transaction Type
                                                    </th>
                                                    <th className="px-4 py-3 text-left text-xs font-medium text-gray-700 uppercase tracking-wider">
                                                        Payment Option
                                                    </th>
                                                </tr>
                                            </thead>
                                            <tbody className="bg-white divide-y divide-gray-200">
                                                {transactionList.map(
                                                    (transaction) => {
                                                        const transactionType =
                                                            getTransactionTypeDisplay(
                                                                transaction
                                                            );
                                                        const displayAmount =
                                                            getDisplayAmount(
                                                                transaction
                                                            );

                                                        return (
                                                            <tr
                                                                key={
                                                                    transaction.id
                                                                }
                                                                className="hover:bg-gray-50"
                                                            >
                                                                {/* Transaction Date */}
                                                                <td className="px-4 py-3 whitespace-nowrap text-sm text-gray-900 border-r border-gray-200">
                                                                    {formatDate(
                                                                        transaction.created_at
                                                                    )}
                                                                </td>

                                                                {/* Transaction Status */}
                                                                <td className="px-4 py-3 whitespace-nowrap border-r border-gray-200">
                                                                    <span
                                                                        className={`inline-flex items-center px-3 py-1 rounded-full text-xs font-medium ${getStatusColor(
                                                                            transaction.status
                                                                        )}`}
                                                                    >
                                                                        {transaction.status
                                                                            .charAt(
                                                                                0
                                                                            )
                                                                            .toUpperCase() +
                                                                            transaction.status.slice(
                                                                                1
                                                                            )}
                                                                    </span>
                                                                </td>

                                                                {/* Transaction ID */}
                                                                <td className="px-4 py-3 whitespace-nowrap border-r border-gray-200">
                                                                    <div className="flex flex-col">
                                                                        <code className="text-xs font-mono bg-gray-100 px-2 py-1 rounded mb-1">
                                                                            {
                                                                                transaction.reference
                                                                            }
                                                                        </code>
                                                                        <div className="flex space-x-2">
                                                                            <button
                                                                                onClick={() =>
                                                                                    handleViewDetails(
                                                                                        transaction
                                                                                    )
                                                                                }
                                                                                className="text-xs text-blue-600 hover:text-blue-800 hover:underline"
                                                                            >
                                                                                View
                                                                            </button>
                                                                            <span className="text-gray-300">
                                                                                ›
                                                                            </span>
                                                                            <button
                                                                                onClick={() =>
                                                                                    handleDownloadReceipt(
                                                                                        transaction
                                                                                    )
                                                                                }
                                                                                className="text-xs text-green-600 hover:text-green-800 hover:underline"
                                                                            >
                                                                                Download
                                                                            </button>
                                                                        </div>
                                                                    </div>
                                                                </td>

                                                                {/* Mobile Network */}
                                                                <td className="px-4 py-3 whitespace-nowrap text-sm text-gray-900 border-r border-gray-200">
                                                                    {transaction.network ||
                                                                        "-"}
                                                                </td>

                                                                {/* Product Description */}
                                                                <td className="px-4 py-3 whitespace-nowrap text-sm text-gray-900 border-r border-gray-200">
                                                                    {getProductDescription(
                                                                        transaction
                                                                    )}
                                                                </td>

                                                                {/* Mobile Number */}
                                                                <td className="px-4 py-3 whitespace-nowrap text-sm text-gray-900 border-r border-gray-200">
                                                                    {transaction.phone ||
                                                                        "-"}
                                                                </td>

                                                                {/* Amount (₦) */}
                                                                <td className="px-4 py-3 whitespace-nowrap text-sm text-gray-900 border-r border-gray-200 text-right font-medium">
                                                                    <span
                                                                        className={
                                                                            transactionType.text ===
                                                                            "Credit"
                                                                                ? "text-green-600"
                                                                                : "text-red-600"
                                                                        }
                                                                    >
                                                                        {formatCurrency(
                                                                            displayAmount
                                                                        )}
                                                                    </span>
                                                                </td>

                                                                {/* Transaction Type (Debit/Credit) */}
                                                                <td className="px-4 py-3 whitespace-nowrap border-r border-gray-200">
                                                                    <span
                                                                        className={`inline-flex items-center px-3 py-1 rounded-full text-xs font-medium ${transactionType.color}`}
                                                                    >
                                                                        {
                                                                            transactionType.text
                                                                        }
                                                                    </span>
                                                                </td>

                                                                {/* Payment Option */}
                                                                <td className="px-4 py-3 whitespace-nowrap text-sm text-gray-900">
                                                                    <span className="px-2 py-1 bg-gray-100 rounded text-xs">
                                                                        {transaction.type ===
                                                                        "deposit"
                                                                            ? "Bank Deposit"
                                                                            : "Wallet"}
                                                                    </span>
                                                                </td>
                                                            </tr>
                                                        );
                                                    }
                                                )}
                                            </tbody>
                                        </table>
                                    </div>

                                    {/* Transaction Details Modal */}
                                    {showDetails && selectedTransaction && (
                                        <div className="fixed inset-0 bg-gray-500 bg-opacity-75 flex items-center justify-center z-50 p-4">
                                            <div className="bg-white rounded-lg max-w-2xl w-full max-h-[90vh] overflow-y-auto">
                                                <div className="px-6 py-4 border-b border-gray-200">
                                                    <div className="flex justify-between items-center">
                                                        <h3 className="text-lg font-semibold text-gray-900">
                                                            Transaction Details
                                                        </h3>
                                                        <button
                                                            onClick={() =>
                                                                setShowDetails(
                                                                    false
                                                                )
                                                            }
                                                            className="text-gray-400 hover:text-gray-500"
                                                        >
                                                            <span className="sr-only">
                                                                Close
                                                            </span>
                                                            <svg
                                                                className="h-6 w-6"
                                                                fill="none"
                                                                viewBox="0 0 24 24"
                                                                stroke="currentColor"
                                                            >
                                                                <path
                                                                    strokeLinecap="round"
                                                                    strokeLinejoin="round"
                                                                    strokeWidth={
                                                                        2
                                                                    }
                                                                    d="M6 18L18 6M6 6l12 12"
                                                                />
                                                            </svg>
                                                        </button>
                                                    </div>
                                                </div>
                                                <div className="px-6 py-4">
                                                    <div className="space-y-4">
                                                        <div className="grid grid-cols-2 gap-4">
                                                            <div>
                                                                <label className="text-sm font-medium text-gray-500">
                                                                    Transaction
                                                                    ID
                                                                </label>
                                                                <p className="mt-1 text-sm text-gray-900 font-mono">
                                                                    {
                                                                        selectedTransaction.reference
                                                                    }
                                                                </p>
                                                            </div>
                                                            <div>
                                                                <label className="text-sm font-medium text-gray-500">
                                                                    Status
                                                                </label>
                                                                <p className="mt-1">
                                                                    <span
                                                                        className={`inline-flex items-center px-3 py-1 rounded-full text-xs font-medium ${getStatusColor(
                                                                            selectedTransaction.status
                                                                        )}`}
                                                                    >
                                                                        {selectedTransaction.status.toUpperCase()}
                                                                    </span>
                                                                </p>
                                                            </div>
                                                        </div>
                                                        <div className="grid grid-cols-2 gap-4">
                                                            <div>
                                                                <label className="text-sm font-medium text-gray-500">
                                                                    Date & Time
                                                                </label>
                                                                <p className="mt-1 text-sm text-gray-900">
                                                                    {formatDate(
                                                                        selectedTransaction.created_at
                                                                    )}
                                                                </p>
                                                            </div>
                                                            <div>
                                                                <label className="text-sm font-medium text-gray-500">
                                                                    Transaction
                                                                    Type
                                                                </label>
                                                                <p className="mt-1">
                                                                    {selectedTransaction.type ===
                                                                    "deposit" ? (
                                                                        <span className="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium text-green-600 bg-green-50">
                                                                            Credit
                                                                            (Deposit)
                                                                        </span>
                                                                    ) : (
                                                                        <span className="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium text-red-600 bg-red-50">
                                                                            Debit
                                                                            (Purchase)
                                                                        </span>
                                                                    )}
                                                                </p>
                                                            </div>
                                                        </div>
                                                        {selectedTransaction.order_id && (
                                                            <div>
                                                                <label className="text-sm font-medium text-gray-500">
                                                                    Order ID
                                                                </label>
                                                                <p className="mt-1 text-sm text-gray-900 font-mono">
                                                                    {
                                                                        selectedTransaction.order_id
                                                                    }
                                                                </p>
                                                            </div>
                                                        )}
                                                        <div className="border-t border-gray-200 pt-4">
                                                            <h4 className="text-sm font-medium text-gray-900 mb-3">
                                                                Transaction
                                                                Summary
                                                            </h4>
                                                            <div className="space-y-2">
                                                                <div className="flex justify-between">
                                                                    <span className="text-sm text-gray-600">
                                                                        Product
                                                                    </span>
                                                                    <span className="text-sm text-gray-900">
                                                                        {getProductDescription(
                                                                            selectedTransaction
                                                                        )}
                                                                    </span>
                                                                </div>
                                                                {selectedTransaction.network && (
                                                                    <div className="flex justify-between">
                                                                        <span className="text-sm text-gray-600">
                                                                            Network
                                                                        </span>
                                                                        <span className="text-sm text-gray-900">
                                                                            {
                                                                                selectedTransaction.network
                                                                            }
                                                                        </span>
                                                                    </div>
                                                                )}
                                                                {selectedTransaction.phone && (
                                                                    <div className="flex justify-between">
                                                                        <span className="text-sm text-gray-600">
                                                                            Phone
                                                                            Number
                                                                        </span>
                                                                        <span className="text-sm text-gray-900">
                                                                            {
                                                                                selectedTransaction.phone
                                                                            }
                                                                        </span>
                                                                    </div>
                                                                )}
                                                                <div className="flex justify-between">
                                                                    <span className="text-sm text-gray-600">
                                                                        Original
                                                                        Amount
                                                                    </span>
                                                                    <span className="text-sm text-gray-900">
                                                                        {formatCurrency(
                                                                            selectedTransaction.amount
                                                                        )}
                                                                    </span>
                                                                </div>
                                                                {selectedTransaction.amount_charged && (
                                                                    <div className="flex justify-between">
                                                                        <span className="text-sm text-gray-600">
                                                                            Amount
                                                                            Charged
                                                                        </span>
                                                                        <span className="text-sm font-medium text-gray-900">
                                                                            {formatCurrency(
                                                                                selectedTransaction.amount_charged
                                                                            )}
                                                                        </span>
                                                                    </div>
                                                                )}
                                                                <div className="flex justify-between">
                                                                    <span className="text-sm text-gray-600">
                                                                        Payment
                                                                        Method
                                                                    </span>
                                                                    <span className="text-sm text-gray-900">
                                                                        {selectedTransaction.type ===
                                                                        "deposit"
                                                                            ? "Bank Deposit"
                                                                            : "Wallet"}
                                                                    </span>
                                                                </div>
                                                                {selectedTransaction.is_test && (
                                                                    <div className="flex justify-between">
                                                                        <span className="text-sm text-gray-600">
                                                                            Test
                                                                            Mode
                                                                        </span>
                                                                        <span className="text-sm font-medium text-yellow-600">
                                                                            Yes
                                                                            (No
                                                                            real
                                                                            money
                                                                            deducted)
                                                                        </span>
                                                                    </div>
                                                                )}
                                                            </div>
                                                        </div>
                                                        {selectedTransaction.remark && (
                                                            <div>
                                                                <label className="text-sm font-medium text-gray-500">
                                                                    Remarks
                                                                </label>
                                                                <p className="mt-1 text-sm text-gray-900">
                                                                    {
                                                                        selectedTransaction.remark
                                                                    }
                                                                </p>
                                                            </div>
                                                        )}
                                                    </div>
                                                </div>
                                                <div className="px-6 py-4 border-t border-gray-200 bg-gray-50 flex justify-end space-x-3">
                                                    <button
                                                        onClick={() =>
                                                            handleDownloadReceipt(
                                                                selectedTransaction
                                                            )
                                                        }
                                                        className="px-4 py-2 bg-green-600 text-white text-sm font-medium rounded-md hover:bg-green-700"
                                                    >
                                                        Download Receipt
                                                    </button>
                                                    <button
                                                        onClick={() =>
                                                            setShowDetails(
                                                                false
                                                            )
                                                        }
                                                        className="px-4 py-2 bg-gray-200 text-gray-800 text-sm font-medium rounded-md hover:bg-gray-300"
                                                    >
                                                        Close
                                                    </button>
                                                </div>
                                            </div>
                                        </div>
                                    )}

                                    {/* Pagination */}
                                    {transactions.links &&
                                        transactions.links.length > 2 && (
                                            <div className="mt-6 flex items-center justify-between border-t border-gray-200 px-4 py-3 sm:px-6">
                                                <div className="flex flex-1 justify-between sm:hidden">
                                                    <Link
                                                        href={
                                                            transactions
                                                                .links[0].url ||
                                                            "#"
                                                        }
                                                        className="relative inline-flex items-center rounded-md border border-gray-300 bg-white px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-50"
                                                        preserveScroll
                                                    >
                                                        Previous
                                                    </Link>
                                                    <Link
                                                        href={
                                                            transactions.links[
                                                                transactions
                                                                    .links
                                                                    .length - 1
                                                            ].url || "#"
                                                        }
                                                        className="relative ml-3 inline-flex items-center rounded-md border border-gray-300 bg-white px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-50"
                                                        preserveScroll
                                                    >
                                                        Next
                                                    </Link>
                                                </div>
                                                <div className="hidden sm:flex sm:flex-1 sm:items-center sm:justify-between">
                                                    <div>
                                                        <p className="text-sm text-gray-700">
                                                            Showing{" "}
                                                            <span className="font-medium">
                                                                {(transactions.current_page -
                                                                    1) *
                                                                    transactions.per_page +
                                                                    1}
                                                            </span>{" "}
                                                            to{" "}
                                                            <span className="font-medium">
                                                                {Math.min(
                                                                    transactions.current_page *
                                                                        transactions.per_page,
                                                                    transactions.total
                                                                )}
                                                            </span>{" "}
                                                            of{" "}
                                                            <span className="font-medium">
                                                                {
                                                                    transactions.total
                                                                }
                                                            </span>{" "}
                                                            results
                                                        </p>
                                                    </div>
                                                    <div>
                                                        <nav
                                                            className="isolate inline-flex -space-x-px rounded-md shadow-sm"
                                                            aria-label="Pagination"
                                                        >
                                                            {transactions.links.map(
                                                                (
                                                                    link,
                                                                    index
                                                                ) => (
                                                                    <Link
                                                                        key={
                                                                            index
                                                                        }
                                                                        href={
                                                                            link.url ||
                                                                            "#"
                                                                        }
                                                                        className={`relative inline-flex items-center px-4 py-2 text-sm font-semibold ${
                                                                            link.active
                                                                                ? "z-10 bg-blue-600 text-white focus-visible:outline focus-visible:outline-2 focus-visible:outline-offset-2 focus-visible:outline-blue-600"
                                                                                : "text-gray-900 ring-1 ring-inset ring-gray-300 hover:bg-gray-50 focus:outline-offset-0"
                                                                        } ${
                                                                            index ===
                                                                            0
                                                                                ? "rounded-l-md"
                                                                                : ""
                                                                        } ${
                                                                            index ===
                                                                            transactions
                                                                                .links
                                                                                .length -
                                                                                1
                                                                                ? "rounded-r-md"
                                                                                : ""
                                                                        }`}
                                                                        preserveScroll
                                                                    >
                                                                        {link.label
                                                                            .replace(
                                                                                "&laquo;",
                                                                                "«"
                                                                            )
                                                                            .replace(
                                                                                "&raquo;",
                                                                                "»"
                                                                            )}
                                                                    </Link>
                                                                )
                                                            )}
                                                        </nav>
                                                    </div>
                                                </div>
                                            </div>
                                        )}
                                </>
                            )}
                        </div>
                    </div>
                </div>
            </div>
        </AppLayout>
    );
}
