import InputError from "@/Components/InputError";
import InputLabel from "@/Components/InputLabel";
import TextInput from "@/Components/TextInput";
import AvatarUpload from "@/Components/AvatarUpload"; // This is in Components root
import { Button } from "@/Components/ui/button"; // Updated path
import { Transition } from "@headlessui/react";
import { Link, useForm, usePage } from "@inertiajs/react";
import { FormEventHandler } from "react";

export default function UpdateProfileInformation({
    mustVerifyEmail,
    status,
    className = "",
}: {
    mustVerifyEmail: boolean;
    status?: string;
    className?: string;
}) {
    const { auth } = usePage().props;
    const user = auth.user;

    const { data, setData, patch, errors, processing, recentlySuccessful } =
        useForm({
            name: user.name ?? "",
            email: user.email ?? "",
            bvn: user.bvn ?? "",
            nin: user.nin ?? "",
        });

    const isNameLocked = true;
    const isEmailLocked = true;
    const isBvnLocked = Boolean(user.bvn);
    const isNinLocked = Boolean(user.nin);

    const nothingToUpdate =
        (isBvnLocked || !data.bvn) && (isNinLocked || !data.nin);

    const submit: FormEventHandler = (e) => {
        e.preventDefault();
        patch(route("profile.update"));
    };

    return (
        <section className={className}>
            <header>
                <h2 className="text-lg font-medium text-gray-900">
                    Profile Information
                </h2>

                <p className="mt-1 text-sm text-gray-600">
                    Update your account's profile information and avatar.
                </p>
            </header>

            {/* Avatar Upload Section */}
            <div className="mt-6 border-b pb-6">
                <AvatarUpload
                    currentAvatar={user.avatar_url}
                    userName={user.name}
                />
            </div>

            <form onSubmit={submit} className="mt-6 space-y-6">
                <div>
                    <InputLabel htmlFor="name" value="Name" />
                    <TextInput
                        id="name"
                        className="mt-1 block w-full bg-gray-100"
                        value={data.name}
                        disabled={isNameLocked}
                        readOnly
                    />
                    <p className="mt-1 text-xs text-gray-500">
                        Name changes require support approval
                    </p>
                </div>

                <div>
                    <InputLabel htmlFor="email" value="Email" />
                    <TextInput
                        id="email"
                        type="email"
                        className="mt-1 block w-full bg-gray-100"
                        value={data.email}
                        disabled={isEmailLocked}
                        readOnly
                    />
                    <p className="mt-1 text-xs text-gray-500">
                        Email changes require support approval
                    </p>
                </div>

                <div className="border-t pt-6">
                    <h2 className="text-lg font-medium text-gray-900">
                        Additional Identification
                    </h2>

                    <p className="mt-1 text-sm text-gray-600">
                        Submit either BVN or NIN to enable your static account
                        number.
                    </p>
                </div>

                <div>
                    <InputLabel
                        htmlFor="bvn"
                        value="BVN (Bank Verification Number)"
                    />
                    <TextInput
                        id="bvn"
                        type="text"
                        className="mt-1 block w-full"
                        value={data.bvn}
                        disabled={isBvnLocked}
                        onChange={(e) => setData("bvn", e.target.value)}
                        placeholder="Enter your 11 digit BVN"
                        maxLength={11}
                    />

                    {isBvnLocked && (
                        <p className="mt-1 text-xs text-green-600">
                            BVN submitted and locked
                        </p>
                    )}

                    <InputError className="mt-2" message={errors.bvn} />
                </div>

                <div>
                    <InputLabel
                        htmlFor="nin"
                        value="NIN (National Identification Number)"
                    />
                    <TextInput
                        id="nin"
                        type="text"
                        className="mt-1 block w-full"
                        value={data.nin}
                        disabled={isNinLocked}
                        onChange={(e) => setData("nin", e.target.value)}
                        placeholder="Enter your 11 digit NIN"
                        maxLength={11}
                    />

                    {isNinLocked && (
                        <p className="mt-1 text-xs text-green-600">
                            NIN submitted and locked
                        </p>
                    )}

                    <InputError className="mt-2" message={errors.nin} />
                </div>

                {mustVerifyEmail && user.email_verified_at === null && (
                    <p className="text-sm text-gray-800">
                        Your email address is unverified.
                        <Link
                            href={route("verification.send")}
                            method="post"
                            as="button"
                            className="ml-1 underline text-sm text-gray-600 hover:text-gray-900"
                        >
                            Resend verification email
                        </Link>
                    </p>
                )}

                <div className="flex items-center gap-4">
                    <Button
                        type="submit"
                        disabled={processing || nothingToUpdate}
                        className="bg-primary hover:bg-primary/90"
                    >
                        Save Changes
                    </Button>

                    <Transition
                        show={recentlySuccessful}
                        enter="transition ease-in-out"
                        enterFrom="opacity-0"
                        leave="transition ease-in-out"
                        leaveTo="opacity-0"
                    >
                        <p className="text-sm text-gray-600">Saved.</p>
                    </Transition>
                </div>
            </form>
        </section>
    );
}
