import AppLayout from "@/Layouts/AppLayout";
import { Head, useForm, Link } from "@inertiajs/react";
import { useState, useEffect } from "react";
import { toast, ToastContainer } from "react-toastify";
import "react-toastify/dist/ReactToastify.css";

interface Network {
    code: string;
    name: string;
}

interface DataPlan {
    id: string;
    description: string;
    price: number;
    display_price: string;
    validity: string;
    data_size: string;
}

interface Props {
    networks: Network[];
    wallet: { balance: number };
}

interface PurchaseResult {
    phone: string;
    success: boolean;
    message: string;
    order_id?: string;
    status: string;
}

export default function BuyData({ networks = [], wallet }: Props) {
    const { data, setData, post, processing, errors, reset } = useForm({
        network: "",
        plan_id: "",
        phone_numbers: "",
        bypass_validator: false,
    });

    const [plans, setPlans] = useState<DataPlan[]>([]);
    const [selectedPlan, setSelectedPlan] = useState<DataPlan | null>(null);
    const [isLoadingPlans, setIsLoadingPlans] = useState(false);
    const [isPurchasing, setIsPurchasing] = useState(false);
    const [purchaseResults, setPurchaseResults] = useState<PurchaseResult[]>(
        []
    );
    const [currentBalance, setCurrentBalance] = useState<number>(
        wallet?.balance || 0
    );

    // Format number helper
    const formatNumber = (num: number) => {
        return new Intl.NumberFormat("en-NG").format(num);
    };

    // Load plans when network is selected
    useEffect(() => {
        if (data.network) {
            loadPlans(data.network);
        } else {
            setPlans([]);
            setSelectedPlan(null);
            setData("plan_id", "");
        }
    }, [data.network]);

    const loadPlans = async (networkCode: string) => {
        setIsLoadingPlans(true);
        try {
            const response = await fetch(`/data/plans/${networkCode}`);

            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }

            const result = await response.json();

            if (result.success) {
                setPlans(result.plans);

                if (result.plans.length > 0 && !data.plan_id) {
                    const firstPlan = result.plans[0];
                    setSelectedPlan(firstPlan);
                    setData("plan_id", firstPlan.id);
                } else if (result.plans.length === 0) {
                    setSelectedPlan(null);
                    setData("plan_id", "");
                }
            } else {
                toast.error(result.message || "Failed to load data plans");
                setPlans([]);
                setSelectedPlan(null);
                setData("plan_id", "");
            }
        } catch (error: any) {
            console.error("Error loading data plans:", error);
            toast.error("Error loading data plans: " + error.message);
            setPlans([]);
            setSelectedPlan(null);
            setData("plan_id", "");
        } finally {
            setIsLoadingPlans(false);
        }
    };

    const handlePlanSelect = (planId: string) => {
        const plan = plans.find((p) => p.id === planId);
        if (plan) {
            setSelectedPlan(plan);
            setData("plan_id", planId);
        } else {
            setSelectedPlan(null);
        }
    };

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();

        // Basic validation
        if (!data.network) {
            toast.error("Please select a mobile network");
            return;
        }

        if (!data.plan_id) {
            toast.error("Please select a data bundle plan");
            return;
        }

        if (!data.phone_numbers.trim()) {
            toast.error("Please enter phone numbers");
            return;
        }

        const phoneCount = countPhoneNumbers();
        if (phoneCount === 0) {
            toast.error("Please enter valid phone numbers");
            return;
        }

        // Validate user balance
        const totalAmount = calculateTotalAmount();
        if (currentBalance < totalAmount) {
            toast.error(
                `Insufficient balance. You need ₦${formatNumber(
                    totalAmount
                )} but have ₦${formatNumber(currentBalance)}`
            );
            return;
        }

        // Start purchase
        setIsPurchasing(true);
        setPurchaseResults([]);

        // Confirm purchase
        if (
            !confirm(
                `Confirm purchase of ₦${formatNumber(
                    totalAmount
                )} data bundle for ${phoneCount} number(s)?`
            )
        ) {
            setIsPurchasing(false);
            return;
        }

        try {
            // Get CSRF token
            const csrfToken =
                document
                    .querySelector('meta[name="csrf-token"]')
                    ?.getAttribute("content") || "";

            // Make API call
            const response = await fetch(route("data.purchase"), {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                    "X-CSRF-TOKEN": csrfToken,
                    "X-Requested-With": "XMLHttpRequest",
                },
                body: JSON.stringify(data),
            });

            const result = await response.json();

            if (result.success) {
                // Update balance
                if (result.wallet_balance !== undefined) {
                    setCurrentBalance(result.wallet_balance);
                }

                // Show success message
                if (result.successful > 0) {
                    toast.success(
                        `✅ Purchase completed! ${result.successful} successful`
                    );
                }

                if (result.failed > 0) {
                    toast.warning(`⚠️ ${result.failed} failed`);
                }

                // Store results for display
                setPurchaseResults(result.results || []);

                // Reset form if all successful
                if (result.failed === 0 && result.successful > 0) {
                    reset();
                    setSelectedPlan(null);
                    setPlans([]);
                }
            } else {
                // Show generic error message
                toast.error(
                    result.message || "Service unavailable, try again later"
                );
            }
        } catch (error: any) {
            console.error("Purchase error:", error);
            toast.error("Service unavailable, try again later");
        } finally {
            setIsPurchasing(false);
        }
    };

    const formatPrice = (price: number) => {
        return new Intl.NumberFormat("en-NG", {
            style: "currency",
            currency: "NGN",
        }).format(price);
    };

    // Calculate total amount
    const calculateTotalAmount = () => {
        if (!selectedPlan) return 0;

        const phoneCount = countPhoneNumbers();
        if (phoneCount === 0) {
            return selectedPlan.price;
        }

        return selectedPlan.price * phoneCount;
    };

    // Count valid phone numbers
    const countPhoneNumbers = () => {
        if (!data.phone_numbers || data.phone_numbers.trim() === "") {
            return 0;
        }

        const phones = data.phone_numbers
            .split(/[,|\n]/)
            .filter((num) => num.trim().length > 0);

        return phones.length;
    };

    // Get valid phone numbers for display
    const getPhoneNumbersArray = () => {
        if (!data.phone_numbers || data.phone_numbers.trim() === "") {
            return [];
        }

        return data.phone_numbers
            .split(/[,|\n]/)
            .map((num) => num.trim())
            .filter((num) => num.length > 0);
    };

    // Check if form is ready for submission
    const isFormReady = () => {
        return (
            data.network &&
            data.plan_id &&
            data.phone_numbers.trim() &&
            countPhoneNumbers() > 0 &&
            currentBalance >= calculateTotalAmount()
        );
    };

    return (
        <AppLayout pageTitle="Buy Databundle">
            <Head title="Buy Databundle" />
            <ToastContainer
                position="top-right"
                theme="colored"
                autoClose={5000}
                hideProgressBar={false}
                newestOnTop
                closeOnClick
                rtl={false}
                pauseOnFocusLoss
                draggable
                pauseOnHover
            />

            <div className="py-6 px-4 sm:px-6 lg:px-8">
                <div className="max-w-4xl mx-auto">
                    {/* Page Header */}
                    <div className="mb-8">
                        <h1 className="text-3xl font-bold text-gray-900">
                            Buy Databundle
                        </h1>
                        <p className="text-gray-600 mt-2">
                            Purchase data bundles for all networks
                        </p>
                    </div>

                    {/* Current Balance */}
                    <div className="bg-blue-50 border border-blue-200 rounded-xl p-6 mb-6">
                        <div className="flex justify-between items-center">
                            <div>
                                <h3 className="font-bold text-blue-800 text-lg">
                                    Current Balance
                                </h3>
                                <p className="text-3xl font-bold text-blue-800 mt-1">
                                    ₦{formatNumber(currentBalance)}
                                </p>
                                <p className="text-sm text-blue-600 mt-2">
                                    {countPhoneNumbers()} number(s) selected
                                </p>
                            </div>
                            <Link
                                href={route("fund.wallet")}
                                className="bg-blue-600 hover:bg-blue-700 text-white font-semibold py-2 px-4 rounded-lg transition-all"
                            >
                                Top Up Wallet
                            </Link>
                        </div>
                    </div>

                    <div className="bg-white rounded-lg shadow-lg mb-6">
                        <div className="p-6">
                            <form onSubmit={handleSubmit} className="space-y-8">
                                {/* Mobile Network Selection */}
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 mb-2">
                                        MOBILE NETWORK
                                    </label>
                                    <select
                                        value={data.network}
                                        onChange={(e) => {
                                            setData("network", e.target.value);
                                            setData("plan_id", "");
                                            setSelectedPlan(null);
                                            setPlans([]);
                                        }}
                                        disabled={isPurchasing}
                                        className="w-full p-3 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500 disabled:bg-gray-100 disabled:cursor-not-allowed"
                                    >
                                        <option value="">- Select -</option>
                                        {networks.map((network) => (
                                            <option
                                                key={network.code}
                                                value={network.code}
                                            >
                                                {network.name}
                                            </option>
                                        ))}
                                    </select>
                                    {errors.network && (
                                        <p className="text-red-500 text-sm mt-1">
                                            {errors.network}
                                        </p>
                                    )}
                                </div>

                                {/* Databundle Plan Selection */}
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 mb-2">
                                        DATABUNDLE PLAN
                                    </label>
                                    {!data.network ? (
                                        <div className="text-gray-500 text-sm">
                                            Select a mobile network first
                                        </div>
                                    ) : isLoadingPlans ? (
                                        <div className="flex items-center space-x-2">
                                            <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-blue-500"></div>
                                            <span className="text-gray-500">
                                                Loading plans...
                                            </span>
                                        </div>
                                    ) : plans.length === 0 ? (
                                        <div className="text-gray-500 text-sm">
                                            No data plans available for this
                                            network
                                        </div>
                                    ) : (
                                        <select
                                            value={data.plan_id}
                                            onChange={(e) => {
                                                handlePlanSelect(
                                                    e.target.value
                                                );
                                            }}
                                            disabled={isPurchasing}
                                            className="w-full p-3 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500 disabled:bg-gray-100 disabled:cursor-not-allowed"
                                        >
                                            <option value="">- Select -</option>
                                            {plans.map((plan, index) => {
                                                const displayText = `${
                                                    plan.description
                                                } - ${formatPrice(plan.price)}${
                                                    plan.validity
                                                        ? ` - ${plan.validity}`
                                                        : ""
                                                }${
                                                    plan.data_size
                                                        ? ` (${plan.data_size})`
                                                        : ""
                                                }`;

                                                return (
                                                    <option
                                                        key={plan.id || index}
                                                        value={plan.id}
                                                    >
                                                        {displayText}
                                                    </option>
                                                );
                                            })}
                                        </select>
                                    )}
                                    {errors.plan_id && (
                                        <p className="text-red-500 text-sm mt-1">
                                            {errors.plan_id}
                                        </p>
                                    )}
                                </div>

                                {/* Mobile Numbers Input */}
                                <div>
                                    <div className="flex justify-between items-center mb-2">
                                        <label className="block text-sm font-medium text-gray-700">
                                            MOBILE NUMBERS
                                        </label>
                                        <span className="text-sm text-gray-500">
                                            {countPhoneNumbers()} number(s)
                                        </span>
                                    </div>
                                    <textarea
                                        value={data.phone_numbers}
                                        onChange={(e) =>
                                            setData(
                                                "phone_numbers",
                                                e.target.value
                                            )
                                        }
                                        placeholder="Enter one mobile number per line or separate with comma e.g. 09062547110, 08149659347"
                                        rows={4}
                                        disabled={isPurchasing}
                                        className="w-full p-3 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500 disabled:bg-gray-100 disabled:cursor-not-allowed"
                                    />
                                    <p className="text-sm text-gray-500 mt-1">
                                        Enter 11-digit numbers (e.g.,
                                        08012345678, 08123456789)
                                    </p>
                                    {errors.phone_numbers && (
                                        <p className="text-red-500 text-sm mt-1">
                                            {errors.phone_numbers}
                                        </p>
                                    )}
                                </div>

                                {/* Selected Numbers Preview */}
                                {getPhoneNumbersArray().length > 0 && (
                                    <div className="bg-gray-50 border border-gray-200 rounded-md p-4">
                                        <h4 className="font-medium text-gray-700 mb-2">
                                            Numbers to recharge:
                                        </h4>
                                        <div className="text-sm text-gray-600">
                                            {getPhoneNumbersArray().map(
                                                (num, index) => (
                                                    <div
                                                        key={index}
                                                        className="flex items-center mb-1"
                                                    >
                                                        <span className="mr-2">
                                                            •
                                                        </span>
                                                        <span>{num}</span>
                                                    </div>
                                                )
                                            )}
                                        </div>
                                    </div>
                                )}

                                {/* Amount Display */}
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 mb-2">
                                        AMOUNT TO PAY
                                    </label>
                                    <div className="p-4 bg-gray-50 border border-gray-300 rounded-md">
                                        {selectedPlan ? (
                                            <div>
                                                <div className="font-semibold text-2xl text-gray-900">
                                                    {formatPrice(
                                                        calculateTotalAmount()
                                                    )}
                                                </div>
                                                <div className="text-sm text-gray-500 mt-2">
                                                    {selectedPlan.price > 0 && (
                                                        <>
                                                            <div>
                                                                Plan price:{" "}
                                                                {formatPrice(
                                                                    selectedPlan.price
                                                                )}
                                                            </div>
                                                            {countPhoneNumbers() >
                                                                0 && (
                                                                <div>
                                                                    Number of
                                                                    lines:{" "}
                                                                    {countPhoneNumbers()}
                                                                </div>
                                                            )}
                                                        </>
                                                    )}
                                                </div>
                                                {currentBalance <
                                                    calculateTotalAmount() && (
                                                    <div className="mt-3 p-2 bg-red-50 border border-red-200 text-red-700 rounded text-sm">
                                                        ⚠️ Your balance is
                                                        insufficient for this
                                                        transaction
                                                    </div>
                                                )}
                                            </div>
                                        ) : (
                                            <span className="text-gray-500">
                                                Select a plan to see amount
                                            </span>
                                        )}
                                    </div>
                                </div>

                                {/* Bypass Validator Checkbox */}
                                <div className="flex items-center space-x-3">
                                    <input
                                        type="checkbox"
                                        id="bypass_validator"
                                        checked={data.bypass_validator}
                                        onChange={(e) =>
                                            setData(
                                                "bypass_validator",
                                                e.target.checked
                                            )
                                        }
                                        disabled={isPurchasing}
                                        className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded disabled:bg-gray-100"
                                    />
                                    <label
                                        htmlFor="bypass_validator"
                                        className="text-sm text-gray-700"
                                    >
                                        Bypass number validator (Use with
                                        caution)
                                    </label>
                                </div>

                                {/* Submit Button */}
                                <div>
                                    <button
                                        type="submit"
                                        disabled={
                                            isPurchasing || !isFormReady()
                                        }
                                        className={`w-full py-3 px-4 rounded-md font-medium text-lg ${
                                            isPurchasing || !isFormReady()
                                                ? "bg-gray-400 cursor-not-allowed"
                                                : "bg-gradient-to-r from-blue-600 to-blue-700 hover:from-blue-700 hover:to-blue-800 text-white shadow-lg hover:shadow-xl transition-all duration-200"
                                        }`}
                                    >
                                        {isPurchasing ? (
                                            <span className="flex items-center justify-center">
                                                <svg
                                                    className="animate-spin h-6 w-6 mr-3 text-white"
                                                    viewBox="0 0 24 24"
                                                >
                                                    <circle
                                                        className="opacity-25"
                                                        cx="12"
                                                        cy="12"
                                                        r="10"
                                                        stroke="currentColor"
                                                        strokeWidth="4"
                                                        fill="none"
                                                    />
                                                    <path
                                                        className="opacity-75"
                                                        fill="currentColor"
                                                        d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"
                                                    />
                                                </svg>
                                                Processing...
                                            </span>
                                        ) : (
                                            `BUY NOW - ${formatPrice(
                                                calculateTotalAmount()
                                            )}`
                                        )}
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>

                    {/* Purchase Results */}
                    {purchaseResults.length > 0 && (
                        <div className="bg-white rounded-lg shadow-lg">
                            <div className="p-6">
                                <h3 className="text-lg font-bold text-gray-900 mb-4">
                                    Purchase Results
                                </h3>
                                <div className="space-y-3">
                                    {purchaseResults.map((result, index) => (
                                        <div
                                            key={index}
                                            className={`p-3 rounded border ${
                                                result.success
                                                    ? "bg-green-50 border-green-200"
                                                    : "bg-red-50 border-red-200"
                                            }`}
                                        >
                                            <div className="flex justify-between items-center">
                                                <span className="font-medium">
                                                    {result.phone}
                                                </span>
                                                <span
                                                    className={`px-2 py-1 rounded text-xs font-medium ${
                                                        result.success
                                                            ? "bg-green-100 text-green-800"
                                                            : "bg-red-100 text-red-800"
                                                    }`}
                                                >
                                                    {result.success
                                                        ? "SUCCESS"
                                                        : "FAILED"}
                                                </span>
                                            </div>
                                            <p className="text-sm text-gray-600 mt-1">
                                                {result.message}
                                            </p>
                                            {result.order_id && (
                                                <p className="text-xs text-gray-500 mt-1">
                                                    Order ID: {result.order_id}
                                                </p>
                                            )}
                                        </div>
                                    ))}
                                </div>
                            </div>
                        </div>
                    )}
                </div>
            </div>
        </AppLayout>
    );
}
