import AppLayout from "@/Layouts/AppLayout";
import { Head, useForm, Link } from "@inertiajs/react";
import { useState, useEffect } from "react";
import { toast, ToastContainer } from "react-toastify";
import "react-toastify/dist/ReactToastify.css";

interface CableProvider {
    code: string;
    name: string;
    display_name: string;
}

interface CablePackage {
    id: string;
    package_code: string;
    name: string;
    price: number;
    display_price: string;
    description: string;
    validity?: string;
}

interface SmartcardInfo {
    customer_name: string;
    status: string;
    message: string;
}

interface Props {
    providers: CableProvider[];
    wallet: { balance: number };
}

interface PurchaseResult {
    smartcard_number: string;
    phone: string;
    success: boolean;
    message: string;
    order_id?: string;
    status: string;
}

export default function BuyCableTV({ providers = [], wallet }: Props) {
    const { data, setData, post, processing, errors, reset } = useForm({
        provider: "",
        package_code: "",
        smartcard_number: "",
        phone_number: "",
        subscription_type: "purchase", // purchase or renew
        auto_renew: false,
    });

    const [packages, setPackages] = useState<CablePackage[]>([]);
    const [selectedPackage, setSelectedPackage] = useState<CablePackage | null>(
        null
    );
    const [isLoadingPackages, setIsLoadingPackages] = useState(false);
    const [isPurchasing, setIsPurchasing] = useState(false);
    const [isVerifying, setIsVerifying] = useState(false);
    const [smartcardInfo, setSmartcardInfo] = useState<SmartcardInfo | null>(
        null
    );
    const [purchaseResults, setPurchaseResults] = useState<PurchaseResult[]>(
        []
    );
    const [currentBalance, setCurrentBalance] = useState<number>(
        wallet?.balance || 0
    );

    // Format number helper
    const formatNumber = (num: number) => {
        return new Intl.NumberFormat("en-NG").format(num);
    };

    // Load packages when provider is selected
    useEffect(() => {
        if (data.provider) {
            loadPackages(data.provider);
        } else {
            setPackages([]);
            setSelectedPackage(null);
            setData("package_code", "");
        }
    }, [data.provider]);

    const loadPackages = async (providerCode: string) => {
        setIsLoadingPackages(true);
        try {
            const response = await fetch(`/cable/packages/${providerCode}`);

            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }

            const result = await response.json();

            if (result.success) {
                setPackages(result.packages);

                if (result.packages.length > 0 && !data.package_code) {
                    const firstPackage = result.packages[0];
                    setSelectedPackage(firstPackage);
                    setData("package_code", firstPackage.package_code);
                } else if (result.packages.length === 0) {
                    setSelectedPackage(null);
                    setData("package_code", "");
                }
            } else {
                toast.error(result.message || "Failed to load cable packages");
                setPackages([]);
                setSelectedPackage(null);
                setData("package_code", "");
            }
        } catch (error: any) {
            console.error("Error loading cable packages:", error);
            toast.error("Error loading cable packages: " + error.message);
            setPackages([]);
            setSelectedPackage(null);
            setData("package_code", "");
        } finally {
            setIsLoadingPackages(false);
        }
    };

    const handlePackageSelect = (packageCode: string) => {
        const pkg = packages.find((p) => p.package_code === packageCode);
        if (pkg) {
            setSelectedPackage(pkg);
            setData("package_code", packageCode);
        } else {
            setSelectedPackage(null);
        }
    };

    const handleVerifySmartcard = async () => {
        if (!data.provider) {
            toast.error("Please select a cable provider");
            return;
        }

        if (!data.smartcard_number.trim()) {
            toast.error("Please enter a smartcard/IUC number");
            return;
        }

        setIsVerifying(true);
        setSmartcardInfo(null);

        try {
            const csrfToken =
                document
                    .querySelector('meta[name="csrf-token"]')
                    ?.getAttribute("content") || "";

            const response = await fetch("/cable/verify", {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                    "X-CSRF-TOKEN": csrfToken,
                    "X-Requested-With": "XMLHttpRequest",
                },
                body: JSON.stringify({
                    provider: data.provider,
                    smartcard_number: data.smartcard_number,
                }),
            });

            const result = await response.json();

            if (result.success) {
                setSmartcardInfo(result.info);
                toast.success("Smartcard verified successfully");
            } else {
                setSmartcardInfo({
                    customer_name: "",
                    status: "failed",
                    message: result.message || "Verification failed",
                });
                toast.error(result.message || "Failed to verify smartcard");
            }
        } catch (error: any) {
            console.error("Verification error:", error);
            toast.error("Verification service unavailable");
            setSmartcardInfo({
                customer_name: "",
                status: "error",
                message: "Service error",
            });
        } finally {
            setIsVerifying(false);
        }
    };

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();

        // Validation
        if (!data.provider) {
            toast.error("Please select a cable provider");
            return;
        }

        if (!data.package_code) {
            toast.error("Please select a bouquet/package");
            return;
        }

        if (!data.smartcard_number.trim()) {
            toast.error("Please enter a smartcard/IUC number");
            return;
        }

        if (!data.phone_number.trim()) {
            toast.error("Please enter a phone number");
            return;
        }

        // Basic phone validation
        const phoneRegex = /^(0[7-9][0-1]\d{8})$/;
        if (!phoneRegex.test(data.phone_number)) {
            toast.error("Please enter a valid 11-digit Nigerian phone number");
            return;
        }

        // Validate user balance
        const totalAmount = selectedPackage?.price || 0;
        if (currentBalance < totalAmount) {
            toast.error(
                `Insufficient balance. You need ₦${formatNumber(
                    totalAmount
                )} but have ₦${formatNumber(currentBalance)}`
            );
            return;
        }

        // Confirm purchase
        const providerName = getProviderName(data.provider);
        const packageName = selectedPackage?.name || "";

        if (
            !confirm(
                `Confirm ${
                    data.subscription_type === "purchase"
                        ? "purchase"
                        : "renewal"
                } of ₦${formatNumber(
                    totalAmount
                )} ${packageName} for ${providerName} smartcard ${
                    data.smartcard_number
                }?`
            )
        ) {
            return;
        }

        // Start purchase
        setIsPurchasing(true);
        setPurchaseResults([]);

        try {
            const csrfToken =
                document
                    .querySelector('meta[name="csrf-token"]')
                    ?.getAttribute("content") || "";

            const response = await fetch("/cable/purchase", {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                    "X-CSRF-TOKEN": csrfToken,
                    "X-Requested-With": "XMLHttpRequest",
                },
                body: JSON.stringify(data),
            });

            const result = await response.json();

            if (result.success) {
                // Update balance
                if (result.wallet_balance !== undefined) {
                    setCurrentBalance(result.wallet_balance);
                }

                // Show success message
                toast.success(
                    `✅ ${
                        data.subscription_type === "purchase"
                            ? "Purchase"
                            : "Renewal"
                    } successful! Order ID: ${result.order_id || "N/A"}`
                );

                // Store result for display
                setPurchaseResults([
                    {
                        smartcard_number: data.smartcard_number,
                        phone: data.phone_number,
                        success: true,
                        message:
                            result.message ||
                            "Subscription processed successfully",
                        order_id: result.order_id,
                        status: result.status || "success",
                    },
                ]);

                // Reset form if successful
                if (result.status === "success") {
                    reset();
                    setSelectedPackage(null);
                    setPackages([]);
                    setSmartcardInfo(null);
                }
            } else {
                toast.error(result.message || "Subscription failed");

                setPurchaseResults([
                    {
                        smartcard_number: data.smartcard_number,
                        phone: data.phone_number,
                        success: false,
                        message: result.message || "Subscription failed",
                        order_id: result.order_id,
                        status: result.status || "failed",
                    },
                ]);
            }
        } catch (error: any) {
            console.error("Purchase error:", error);
            toast.error("Service unavailable, try again later");

            setPurchaseResults([
                {
                    smartcard_number: data.smartcard_number,
                    phone: data.phone_number,
                    success: false,
                    message: "Service unavailable, try again later",
                    status: "error",
                },
            ]);
        } finally {
            setIsPurchasing(false);
        }
    };

    const formatPrice = (price: number) => {
        return new Intl.NumberFormat("en-NG", {
            style: "currency",
            currency: "NGN",
        }).format(price);
    };

    const getProviderName = (code: string) => {
        const provider = providers.find((p) => p.code === code);
        return provider ? provider.display_name : code.toUpperCase();
    };

    // Check if form is ready for submission
    const isFormReady = () => {
        return (
            data.provider &&
            data.package_code &&
            data.smartcard_number.trim() &&
            data.phone_number.trim() &&
            currentBalance >= (selectedPackage?.price || 0)
        );
    };

    return (
        <AppLayout pageTitle="Pay Cable TV Subscription">
            <Head title="Pay DStv, GOtv, StarTimes and Showmax Subscription - Instant Activation" />
            <ToastContainer
                position="top-right"
                theme="colored"
                autoClose={5000}
                hideProgressBar={false}
                newestOnTop
                closeOnClick
                rtl={false}
                pauseOnFocusLoss
                draggable
                pauseOnHover
            />

            <div className="py-6 px-4 sm:px-6 lg:px-8">
                <div className="max-w-4xl mx-auto">
                    {/* Page Header */}
                    <div className="mb-8">
                        <h1 className="text-3xl font-bold text-gray-900">
                            Pay DStv, GOtv, StarTimes and Showmax Subscription
                        </h1>
                        <p className="text-gray-600 mt-2">
                            Instant Activation - Get your cable TV subscription
                            instantly
                        </p>
                    </div>

                    {/* Payment Option Section */}
                    <div className="bg-blue-50 border border-blue-200 rounded-xl p-6 mb-6">
                        <h3 className="font-bold text-blue-800 text-lg mb-4">
                            Payment Option
                        </h3>
                        <div className="flex justify-between items-center">
                            <div>
                                <p className="text-2xl font-bold text-blue-800">
                                    Wallet - ₦{formatNumber(currentBalance)}
                                </p>
                                <p className="text-sm text-blue-600 mt-1">
                                    Your available balance for subscription
                                </p>
                            </div>
                            <Link
                                href={route("fund.wallet")}
                                className="bg-blue-600 hover:bg-blue-700 text-white font-semibold py-2 px-4 rounded-lg transition-all"
                            >
                                Top Up Wallet
                            </Link>
                        </div>
                    </div>

                    <div className="bg-white rounded-lg shadow-lg mb-6">
                        <div className="p-6">
                            <form onSubmit={handleSubmit} className="space-y-6">
                                {/* TV Subscription Selection */}
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 mb-2">
                                        TV Subscription
                                    </label>
                                    <select
                                        value={data.provider}
                                        onChange={(e) => {
                                            setData("provider", e.target.value);
                                            setData("package_code", "");
                                            setSelectedPackage(null);
                                            setPackages([]);
                                            setSmartcardInfo(null);
                                        }}
                                        disabled={isPurchasing}
                                        className="w-full p-3 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500 disabled:bg-gray-100 disabled:cursor-not-allowed"
                                    >
                                        <option value="">
                                            - Select TV Subscription -
                                        </option>
                                        {providers.map((provider) => (
                                            <option
                                                key={provider.code}
                                                value={provider.code}
                                            >
                                                {provider.display_name}
                                            </option>
                                        ))}
                                    </select>
                                    {errors.provider && (
                                        <p className="text-red-500 text-sm mt-1">
                                            {errors.provider}
                                        </p>
                                    )}
                                </div>

                                {/* Bouquet/Package Selection */}
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 mb-2">
                                        Bouquet/Package
                                    </label>
                                    {!data.provider ? (
                                        <div className="text-gray-500 text-sm p-3 border border-gray-300 rounded-md bg-gray-50">
                                            Select a TV subscription first
                                        </div>
                                    ) : isLoadingPackages ? (
                                        <div className="flex items-center space-x-2 p-3 border border-gray-300 rounded-md bg-gray-50">
                                            <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-blue-500"></div>
                                            <span className="text-gray-500">
                                                Loading bouquets/packages...
                                            </span>
                                        </div>
                                    ) : packages.length === 0 ? (
                                        <div className="text-gray-500 text-sm p-3 border border-gray-300 rounded-md bg-gray-50">
                                            No bouquets/packages available for
                                            this provider
                                        </div>
                                    ) : (
                                        <select
                                            value={data.package_code}
                                            onChange={(e) => {
                                                handlePackageSelect(
                                                    e.target.value
                                                );
                                            }}
                                            disabled={isPurchasing}
                                            className="w-full p-3 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500 disabled:bg-gray-100 disabled:cursor-not-allowed"
                                        >
                                            <option value="">
                                                - Select Bouquet/Package -
                                            </option>
                                            {packages.map((pkg) => (
                                                <option
                                                    key={pkg.package_code}
                                                    value={pkg.package_code}
                                                >
                                                    {pkg.name} -{" "}
                                                    {formatPrice(pkg.price)}
                                                </option>
                                            ))}
                                        </select>
                                    )}
                                    {errors.package_code && (
                                        <p className="text-red-500 text-sm mt-1">
                                            {errors.package_code}
                                        </p>
                                    )}
                                </div>

                                {/* SmartCard/IUC Number Input */}
                                <div>
                                    <div className="flex justify-between items-center mb-2">
                                        <label className="block text-sm font-medium text-gray-700">
                                            SmartCard/IUC Number
                                        </label>
                                        <button
                                            type="button"
                                            onClick={handleVerifySmartcard}
                                            disabled={
                                                !data.provider ||
                                                !data.smartcard_number.trim() ||
                                                isVerifying ||
                                                isPurchasing
                                            }
                                            className="text-sm bg-gray-100 hover:bg-gray-200 text-gray-800 py-1 px-3 rounded-md transition-all disabled:opacity-50 disabled:cursor-not-allowed"
                                        >
                                            {isVerifying
                                                ? "Verifying..."
                                                : "Verify SmartCard"}
                                        </button>
                                    </div>
                                    <input
                                        type="text"
                                        value={data.smartcard_number}
                                        onChange={(e) => {
                                            setData(
                                                "smartcard_number",
                                                e.target.value
                                            );
                                            setSmartcardInfo(null);
                                        }}
                                        placeholder="Enter your smartcard or IUC number"
                                        disabled={isPurchasing}
                                        className="w-full p-3 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500 disabled:bg-gray-100 disabled:cursor-not-allowed"
                                    />
                                    {errors.smartcard_number && (
                                        <p className="text-red-500 text-sm mt-1">
                                            {errors.smartcard_number}
                                        </p>
                                    )}

                                    {/* Smartcard Verification Result */}
                                    {smartcardInfo && (
                                        <div
                                            className={`mt-3 p-3 rounded-md ${
                                                smartcardInfo.status ===
                                                "success"
                                                    ? "bg-green-50 border border-green-200"
                                                    : "bg-red-50 border border-red-200"
                                            }`}
                                        >
                                            {smartcardInfo.status ===
                                            "success" ? (
                                                <>
                                                    <div className="font-medium text-green-800">
                                                        ✓ SmartCard Verified
                                                    </div>
                                                    {smartcardInfo.customer_name && (
                                                        <div className="text-sm text-green-700 mt-1">
                                                            Customer:{" "}
                                                            {
                                                                smartcardInfo.customer_name
                                                            }
                                                        </div>
                                                    )}
                                                </>
                                            ) : (
                                                <div className="text-red-800">
                                                    {smartcardInfo.message}
                                                </div>
                                            )}
                                        </div>
                                    )}
                                </div>

                                {/* Phone Number Input */}
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 mb-2">
                                        Phone Number
                                    </label>
                                    <input
                                        type="tel"
                                        value={data.phone_number}
                                        onChange={(e) =>
                                            setData(
                                                "phone_number",
                                                e.target.value
                                            )
                                        }
                                        placeholder="Enter phone number for notifications"
                                        disabled={isPurchasing}
                                        className="w-full p-3 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500 disabled:bg-gray-100 disabled:cursor-not-allowed"
                                    />
                                    <p className="text-sm text-gray-500 mt-1">
                                        11-digit Nigerian number (e.g.,
                                        08012345678)
                                    </p>
                                    {errors.phone_number && (
                                        <p className="text-red-500 text-sm mt-1">
                                            {errors.phone_number}
                                        </p>
                                    )}
                                </div>

                                {/* Amount Display */}
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 mb-2">
                                        Amount
                                    </label>
                                    <div className="p-4 bg-gray-50 border border-gray-300 rounded-md">
                                        {selectedPackage ? (
                                            <div>
                                                <div className="font-semibold text-2xl text-gray-900">
                                                    {formatPrice(
                                                        selectedPackage.price
                                                    )}
                                                </div>
                                                <div className="text-sm text-gray-500 mt-2">
                                                    <div>
                                                        Provider:{" "}
                                                        {getProviderName(
                                                            data.provider
                                                        )}
                                                    </div>
                                                    <div>
                                                        Package:{" "}
                                                        {selectedPackage.name}
                                                    </div>
                                                    {selectedPackage.description && (
                                                        <div className="mt-1 text-gray-600">
                                                            {
                                                                selectedPackage.description
                                                            }
                                                        </div>
                                                    )}
                                                </div>
                                                {currentBalance <
                                                    selectedPackage.price && (
                                                    <div className="mt-3 p-2 bg-red-50 border border-red-200 text-red-700 rounded text-sm">
                                                        ⚠️ Your wallet balance
                                                        is insufficient for this
                                                        transaction
                                                    </div>
                                                )}
                                            </div>
                                        ) : (
                                            <span className="text-gray-500">
                                                Select a bouquet/package to see
                                                amount
                                            </span>
                                        )}
                                    </div>
                                </div>

                                {/* Subscription Type Selection */}
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 mb-2">
                                        Subscription Type
                                    </label>
                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                                        <label
                                            className={`flex items-center p-3 border rounded-md cursor-pointer transition-all ${
                                                data.subscription_type ===
                                                "purchase"
                                                    ? "bg-blue-50 border-blue-300"
                                                    : "bg-gray-50 border-gray-300 hover:bg-gray-100"
                                            }`}
                                        >
                                            <input
                                                type="radio"
                                                name="subscription_type"
                                                value="purchase"
                                                checked={
                                                    data.subscription_type ===
                                                    "purchase"
                                                }
                                                onChange={(e) =>
                                                    setData(
                                                        "subscription_type",
                                                        e.target.value
                                                    )
                                                }
                                                className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300"
                                            />
                                            <span className="ml-3 font-medium text-gray-700">
                                                Purchase Bouquet/Package
                                            </span>
                                        </label>
                                        <label
                                            className={`flex items-center p-3 border rounded-md cursor-pointer transition-all ${
                                                data.subscription_type ===
                                                "renew"
                                                    ? "bg-blue-50 border-blue-300"
                                                    : "bg-gray-50 border-gray-300 hover:bg-gray-100"
                                            }`}
                                        >
                                            <input
                                                type="radio"
                                                name="subscription_type"
                                                value="renew"
                                                checked={
                                                    data.subscription_type ===
                                                    "renew"
                                                }
                                                onChange={(e) =>
                                                    setData(
                                                        "subscription_type",
                                                        e.target.value
                                                    )
                                                }
                                                className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300"
                                            />
                                            <span className="ml-3 font-medium text-gray-700">
                                                Renew Subscription
                                            </span>
                                        </label>
                                    </div>
                                </div>

                                {/* Auto Renew Option */}
                                <div className="flex items-center space-x-3">
                                    <input
                                        type="checkbox"
                                        id="auto_renew"
                                        checked={data.auto_renew}
                                        onChange={(e) =>
                                            setData(
                                                "auto_renew",
                                                e.target.checked
                                            )
                                        }
                                        disabled={isPurchasing}
                                        className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded disabled:bg-gray-100"
                                    />
                                    <label
                                        htmlFor="auto_renew"
                                        className="text-sm text-gray-700"
                                    >
                                        Auto renew (Automatically renew this
                                        subscription when it expires)
                                    </label>
                                </div>

                                {/* Submit Button */}
                                <div>
                                    <button
                                        type="submit"
                                        disabled={
                                            isPurchasing || !isFormReady()
                                        }
                                        className={`w-full py-3 px-4 rounded-md font-medium text-lg ${
                                            isPurchasing || !isFormReady()
                                                ? "bg-gray-400 cursor-not-allowed"
                                                : "bg-gradient-to-r from-blue-600 to-blue-700 hover:from-blue-700 hover:to-blue-800 text-white shadow-lg hover:shadow-xl transition-all duration-200"
                                        }`}
                                    >
                                        {isPurchasing ? (
                                            <span className="flex items-center justify-center">
                                                <svg
                                                    className="animate-spin h-6 w-6 mr-3 text-white"
                                                    viewBox="0 0 24 24"
                                                >
                                                    <circle
                                                        className="opacity-25"
                                                        cx="12"
                                                        cy="12"
                                                        r="10"
                                                        stroke="currentColor"
                                                        strokeWidth="4"
                                                        fill="none"
                                                    />
                                                    <path
                                                        className="opacity-75"
                                                        fill="currentColor"
                                                        d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"
                                                    />
                                                </svg>
                                                Processing...
                                            </span>
                                        ) : (
                                            `PAY NOW - ${
                                                selectedPackage
                                                    ? formatPrice(
                                                          selectedPackage.price
                                                      )
                                                    : "₦0.00"
                                            }`
                                        )}
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>

                    {/* Purchase Results */}
                    {purchaseResults.length > 0 && (
                        <div className="bg-white rounded-lg shadow-lg">
                            <div className="p-6">
                                <h3 className="text-lg font-bold text-gray-900 mb-4">
                                    Subscription Result
                                </h3>
                                <div className="space-y-3">
                                    {purchaseResults.map((result, index) => (
                                        <div
                                            key={index}
                                            className={`p-4 rounded border ${
                                                result.success
                                                    ? "bg-green-50 border-green-200"
                                                    : "bg-red-50 border-red-200"
                                            }`}
                                        >
                                            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                                <div>
                                                    <div className="text-sm text-gray-500">
                                                        SmartCard/IUC
                                                    </div>
                                                    <div className="font-medium">
                                                        {
                                                            result.smartcard_number
                                                        }
                                                    </div>
                                                </div>
                                                <div>
                                                    <div className="text-sm text-gray-500">
                                                        Phone Number
                                                    </div>
                                                    <div className="font-medium">
                                                        {result.phone}
                                                    </div>
                                                </div>
                                            </div>
                                            <div className="mt-3 flex justify-between items-center">
                                                <span
                                                    className={`px-3 py-1 rounded-full text-sm font-medium ${
                                                        result.success
                                                            ? "bg-green-100 text-green-800"
                                                            : "bg-red-100 text-red-800"
                                                    }`}
                                                >
                                                    {result.success
                                                        ? "SUCCESS"
                                                        : "FAILED"}
                                                </span>
                                                {result.order_id && (
                                                    <span className="text-sm text-gray-500">
                                                        Order ID:{" "}
                                                        {result.order_id}
                                                    </span>
                                                )}
                                            </div>
                                            <p className="mt-2 text-gray-700">
                                                {result.message}
                                            </p>
                                        </div>
                                    ))}
                                </div>
                            </div>
                        </div>
                    )}
                </div>
            </div>
        </AppLayout>
    );
}
