import AppLayout from "@/Layouts/AppLayout";
import { Head, useForm, Link } from "@inertiajs/react";
import { useState } from "react";
import { toast, ToastContainer } from "react-toastify";
import "react-toastify/dist/ReactToastify.css";

interface Network {
    code: string;
    name: string;
}

interface Props {
    networks: Network[];
    wallet: { balance: number };
}

export default function BuyAirtime({ networks = [], wallet }: Props) {
    const { data, setData, post, processing, errors, reset } = useForm({
        network: "",
        amount: "",
        phone: "",
    });

    const [isPurchasing, setIsPurchasing] = useState(false);
    const [currentBalance, setCurrentBalance] = useState<number>(
        wallet?.balance || 0
    );

    // Common airtime amounts
    const amountOptions = [100, 200, 500, 1000, 1500, 2000, 3000, 5000];

    // Format number helper
    const formatNumber = (num: number) => {
        return new Intl.NumberFormat("en-NG").format(num);
    };

    // Format price
    const formatPrice = (price: number) => {
        return new Intl.NumberFormat("en-NG", {
            style: "currency",
            currency: "NGN",
        }).format(price);
    };

    // Calculate final amount (with discount if any)
    const calculateFinalAmount = () => {
        const amount = parseFloat(data.amount) || 0;
        return amount; // If you have discount, apply it here: amount - (amount * discountPercent / 100)
    };

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();

        // Basic validation
        if (!data.network) {
            toast.error("Please select a mobile network");
            return;
        }

        if (!data.amount || parseFloat(data.amount) < 50) {
            toast.error("Amount must be at least ₦50");
            return;
        }

        if (!data.phone || !/^0[0-9]{10}$/.test(data.phone)) {
            toast.error(
                "Please enter a valid 11-digit phone number starting with 0"
            );
            return;
        }

        const finalAmount = calculateFinalAmount();

        // Validate user balance
        if (currentBalance < finalAmount) {
            toast.error(
                `Insufficient balance. You need ₦${formatNumber(
                    finalAmount
                )} but have ₦${formatNumber(currentBalance)}`
            );
            return;
        }

        // Start purchase
        setIsPurchasing(true);

        // Confirm purchase
        if (
            !confirm(
                `Confirm airtime purchase of ₦${formatNumber(
                    finalAmount
                )} for ${data.phone}?`
            )
        ) {
            setIsPurchasing(false);
            return;
        }

        try {
            // Get CSRF token
            const csrfToken =
                document
                    .querySelector('meta[name="csrf-token"]')
                    ?.getAttribute("content") || "";

            // Make API call
            const response = await fetch(route("airtime.purchase"), {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                    "X-CSRF-TOKEN": csrfToken,
                    "X-Requested-With": "XMLHttpRequest",
                },
                body: JSON.stringify(data),
            });

            const result = await response.json();

            if (result.success) {
                // Update balance
                if (result.wallet_balance !== undefined) {
                    setCurrentBalance(result.wallet_balance);
                }

                // Show success message
                toast.success(
                    `✅ Airtime purchased successfully! ₦${formatNumber(
                        finalAmount
                    )} sent to ${data.phone}`
                );

                // Reset form
                reset();
            } else {
                // Show generic error message
                toast.error(
                    result.message || "Service unavailable, try again later"
                );
            }
        } catch (error: any) {
            console.error("Purchase error:", error);
            toast.error("Service unavailable, try again later");
        } finally {
            setIsPurchasing(false);
        }
    };

    // Check if form is ready for submission
    const isFormReady = () => {
        const finalAmount = calculateFinalAmount();
        return (
            data.network &&
            data.amount &&
            parseFloat(data.amount) >= 50 &&
            data.phone &&
            /^0[0-9]{10}$/.test(data.phone) &&
            currentBalance >= finalAmount
        );
    };

    const finalAmount = calculateFinalAmount();

    return (
        <AppLayout pageTitle="Buy Airtime">
            <Head title="Buy Airtime" />
            <ToastContainer
                position="top-right"
                theme="colored"
                autoClose={5000}
                hideProgressBar={false}
                newestOnTop
                closeOnClick
                rtl={false}
                pauseOnFocusLoss
                draggable
                pauseOnHover
            />

            <div className="py-6 px-4 sm:px-6 lg:px-8">
                <div className="max-w-4xl mx-auto">
                    {/* Page Header */}
                    <div className="mb-8">
                        <h1 className="text-3xl font-bold text-gray-900">
                            Buy Airtime
                        </h1>
                        <p className="text-gray-600 mt-2">
                            Recharge airtime for all networks with instant
                            delivery
                        </p>
                    </div>

                    {/* Current Balance */}
                    <div className="bg-blue-50 border border-blue-200 rounded-xl p-6 mb-6">
                        <div className="flex justify-between items-center">
                            <div>
                                <h3 className="font-bold text-blue-800 text-lg">
                                    Current Balance
                                </h3>
                                <p className="text-3xl font-bold text-blue-800 mt-1">
                                    ₦{formatNumber(currentBalance)}
                                </p>
                            </div>
                            <Link
                                href={route("fund.wallet")}
                                className="bg-blue-600 hover:bg-blue-700 text-white font-semibold py-2 px-4 rounded-lg transition-all"
                            >
                                Top Up Wallet
                            </Link>
                        </div>
                    </div>

                    <div className="bg-white rounded-lg shadow-lg mb-6">
                        <div className="p-6">
                            <form onSubmit={handleSubmit} className="space-y-8">
                                {/* Mobile Network Selection */}
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 mb-2">
                                        MOBILE NETWORK
                                    </label>
                                    <select
                                        value={data.network}
                                        onChange={(e) =>
                                            setData("network", e.target.value)
                                        }
                                        disabled={isPurchasing}
                                        className="w-full p-3 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500 disabled:bg-gray-100 disabled:cursor-not-allowed"
                                    >
                                        <option value="">- Select -</option>
                                        {networks.map((network) => (
                                            <option
                                                key={network.code}
                                                value={network.code}
                                            >
                                                {network.name}
                                            </option>
                                        ))}
                                    </select>
                                    {errors.network && (
                                        <p className="text-red-500 text-sm mt-1">
                                            {errors.network}
                                        </p>
                                    )}
                                </div>

                                {/* Phone Number Input */}
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 mb-2">
                                        PHONE NUMBER
                                    </label>
                                    <input
                                        type="tel"
                                        value={data.phone}
                                        onChange={(e) =>
                                            setData("phone", e.target.value)
                                        }
                                        placeholder="08012345678"
                                        maxLength={11}
                                        disabled={isPurchasing}
                                        className="w-full p-3 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500 disabled:bg-gray-100 disabled:cursor-not-allowed"
                                    />
                                    <p className="text-sm text-gray-500 mt-1">
                                        Enter 11-digit number (e.g.,
                                        08012345678)
                                    </p>
                                    {errors.phone && (
                                        <p className="text-red-500 text-sm mt-1">
                                            {errors.phone}
                                        </p>
                                    )}
                                </div>

                                {/* Amount Selection */}
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 mb-2">
                                        SELECT AMOUNT
                                    </label>

                                    {/* Quick Amount Buttons */}
                                    <div className="grid grid-cols-2 sm:grid-cols-4 gap-3 mb-4">
                                        {amountOptions.map((amount) => (
                                            <button
                                                key={amount}
                                                type="button"
                                                onClick={() =>
                                                    setData(
                                                        "amount",
                                                        amount.toString()
                                                    )
                                                }
                                                className={`p-3 text-center rounded-lg border transition-all ${
                                                    data.amount ===
                                                    amount.toString()
                                                        ? "bg-blue-100 border-blue-500 text-blue-700 font-semibold"
                                                        : "bg-gray-50 border-gray-300 text-gray-700 hover:bg-gray-100"
                                                }`}
                                            >
                                                ₦{formatNumber(amount)}
                                            </button>
                                        ))}
                                    </div>

                                    {/* Custom Amount Input */}
                                    <div className="relative">
                                        <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                            <span className="text-gray-500">
                                                ₦
                                            </span>
                                        </div>
                                        <input
                                            type="number"
                                            value={data.amount}
                                            onChange={(e) =>
                                                setData(
                                                    "amount",
                                                    e.target.value
                                                )
                                            }
                                            min="50"
                                            max="200000"
                                            step="50"
                                            placeholder="Enter custom amount"
                                            disabled={isPurchasing}
                                            className="w-full pl-8 p-3 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500 disabled:bg-gray-100 disabled:cursor-not-allowed"
                                        />
                                    </div>
                                    <p className="text-sm text-gray-500 mt-1">
                                        Minimum: ₦50, Maximum: ₦200,000
                                    </p>
                                    {errors.amount && (
                                        <p className="text-red-500 text-sm mt-1">
                                            {errors.amount}
                                        </p>
                                    )}
                                </div>

                                {/* Amount to Pay Display */}
                                {data.amount &&
                                    parseFloat(data.amount) >= 50 && (
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 mb-2">
                                                AMOUNT TO PAY
                                            </label>
                                            <div className="p-4 bg-gray-50 border border-gray-300 rounded-md">
                                                <div className="font-semibold text-2xl text-gray-900">
                                                    {formatPrice(finalAmount)}
                                                </div>
                                                {currentBalance <
                                                    finalAmount && (
                                                    <div className="mt-3 p-2 bg-red-50 border border-red-200 text-red-700 rounded text-sm">
                                                        ⚠️ Your balance is
                                                        insufficient for this
                                                        transaction
                                                    </div>
                                                )}
                                            </div>
                                        </div>
                                    )}

                                {/* Submit Button */}
                                <div>
                                    <button
                                        type="submit"
                                        disabled={
                                            isPurchasing || !isFormReady()
                                        }
                                        className={`w-full py-3 px-4 rounded-md font-medium text-lg ${
                                            isPurchasing || !isFormReady()
                                                ? "bg-gray-400 cursor-not-allowed"
                                                : "bg-gradient-to-r from-blue-600 to-blue-700 hover:from-blue-700 hover:to-blue-800 text-white shadow-lg hover:shadow-xl transition-all duration-200"
                                        }`}
                                    >
                                        {isPurchasing ? (
                                            <span className="flex items-center justify-center">
                                                <svg
                                                    className="animate-spin h-6 w-6 mr-3 text-white"
                                                    viewBox="0 0 24 24"
                                                >
                                                    <circle
                                                        className="opacity-25"
                                                        cx="12"
                                                        cy="12"
                                                        r="10"
                                                        stroke="currentColor"
                                                        strokeWidth="4"
                                                        fill="none"
                                                    />
                                                    <path
                                                        className="opacity-75"
                                                        fill="currentColor"
                                                        d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"
                                                    />
                                                </svg>
                                                Processing...
                                            </span>
                                        ) : (
                                            `BUY AIRTIME - ${formatPrice(
                                                finalAmount
                                            )}`
                                        )}
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        </AppLayout>
    );
}
