import { useState, useEffect } from "react";
import { Head, router, usePage } from "@inertiajs/react";
import AppLayout from "@/Layouts/AppLayout";
import { ToastContainer, toast } from "react-toastify";
import "react-toastify/dist/ReactToastify.css";

// Shadcn UI Components
import {
    Card,
    CardContent,
    CardDescription,
    CardFooter,
    CardHeader,
    CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
    Select,
    SelectContent,
    SelectItem,
    SelectTrigger,
    SelectValue,
} from "@/components/ui/select";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import {
    Dialog,
    DialogContent,
    DialogDescription,
    DialogFooter,
    DialogHeader,
    DialogTitle,
    DialogTrigger,
} from "@/components/ui/dialog";
import { Separator } from "@/components/ui/separator";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { Alert, AlertDescription, AlertTitle } from "@/components/ui/alert";
import { Loader2, Copy, AlertTriangle, CheckCircle } from "lucide-react";

interface UserShowData {
    id: number;
    name: string;
    email: string;
    kyc_status: string;
}

interface WalletShowData {
    id: number;
    balance: number;
    currency: string;
}

interface StaticAccountData {
    accountNumber?: string;
    bankName?: string;
    accountName?: string;
}

interface CompanyAccountDetails {
    account_number: string;
    bank_name: string;
    account_name: string;
    notes?: string;
    active?: boolean;
}

interface PageProps {
    user: UserShowData;
    wallet: WalletShowData;
    flash?: {
        success?: string;
        error?: string;
    };
    companyAccountDetails?: CompanyAccountDetails | null;
}

interface Bank {
    code: string;
    name: string;
}

interface WithdrawalData {
    amount: string;
    bank_code: string;
    account_number: string;
    account_name: string;
}

export default function Dashboard({
    user,
    wallet,
    companyAccountDetails = null,
}: PageProps) {
    const [showModal, setShowModal] = useState(false);
    const [showWithdrawModal, setShowWithdrawModal] = useState(false);
    const [loadingBanks, setLoadingBanks] = useState(false);
    const [banksError, setBanksError] = useState("");
    const [amount, setAmount] = useState<string>("");
    const [isLoading, setIsLoading] = useState(false);
    const [isWithdrawing, setIsWithdrawing] = useState(false);
    const [staticAccount, setStaticAccount] =
        useState<StaticAccountData | null>(null);
    const [banks, setBanks] = useState<Bank[]>([]);
    const [withdrawalData, setWithdrawalData] = useState<WithdrawalData>({
        amount: "",
        bank_code: "",
        account_number: "",
        account_name: "",
    });
    const [isValidatingAccount, setIsValidatingAccount] = useState(false);

    const { flash } = usePage<PageProps>().props;

    // Fetch static account data
    useEffect(() => {
        if (user.kyc_status === "admin_approved" && !staticAccount) {
            fetchVirtualAccount();
        }
    }, [user.kyc_status]);

    // Fetch banks on mount
    useEffect(() => {
        fetchBanks();
    }, []);

    // Show flash messages as toast notifications
    useEffect(() => {
        if (flash?.success) {
            toast.success(flash.success, {
                position: "top-right",
                autoClose: 5000,
            });
        }
        if (flash?.error) {
            toast.error(flash.error, {
                position: "top-right",
                autoClose: 5000,
            });
        }
    }, [flash]);

    const fetchVirtualAccount = async () => {
        try {
            const response = await fetch(route("wallet.virtual.account"));
            const data = await response.json();

            if (data.account) {
                setStaticAccount({
                    accountNumber: data.account.account_number,
                    bankName: data.account.bank_name,
                    accountName: data.account.account_name,
                });
            }
        } catch (error) {
            console.error("Failed to fetch virtual account:", error);
            toast.error("Failed to load virtual account details");
        }
    };

    const fetchBanks = async (retryCount = 0) => {
        setLoadingBanks(true);
        setBanksError("");

        try {
            const response = await fetch(route("wallet.banks"));

            if (!response.ok) {
                throw new Error(`Failed to fetch banks: ${response.status}`);
            }

            const data = await response.json();

            if (!data.banks || !Array.isArray(data.banks)) {
                throw new Error("Invalid bank data received");
            }

            setBanks(data.banks || []);
            setLoadingBanks(false);
        } catch (error) {
            console.error("Failed to fetch banks:", error);

            // Retry logic
            if (retryCount < 3) {
                setTimeout(() => {
                    fetchBanks(retryCount + 1);
                }, 1000 * (retryCount + 1));
            } else {
                setBanksError(
                    "Unable to load bank list. Please try again later."
                );
                setLoadingBanks(false);
                if (showWithdrawModal) {
                    toast.error("Unable to load bank list. Please try again.");
                }
            }
        }
    };

    const validateBankAccount = async () => {
        if (!withdrawalData.bank_code || !withdrawalData.account_number) {
            toast.warning("Please select bank and enter account number");
            return;
        }

        setIsValidatingAccount(true);
        try {
            const response = await fetch(route("wallet.validate.account"), {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                    "X-CSRF-TOKEN":
                        document
                            .querySelector('meta[name="csrf-token"]')
                            ?.getAttribute("content") || "",
                },
                body: JSON.stringify({
                    bank_code: withdrawalData.bank_code,
                    account_number: withdrawalData.account_number,
                }),
            });

            const data = await response.json();

            if (data.success) {
                setWithdrawalData((prev) => ({
                    ...prev,
                    account_name: data.account_name,
                }));
                toast.success(`Account validated: ${data.account_name}`);
            } else {
                toast.error(data.error || "Invalid account number");
            }
        } catch (error) {
            console.error("Validation error:", error);
            toast.error("Failed to validate account");
        } finally {
            setIsValidatingAccount(false);
        }
    };

    const handleDeposit = async (paymentMethod: string = "monnify") => {
        if (!amount) {
            toast.warning("Please enter an amount");
            return;
        }

        const depositAmount = parseFloat(amount);
        if (depositAmount < 100) {
            toast.warning("Minimum deposit amount is ₦100");
            return;
        }

        setIsLoading(true);

        try {
            const getCsrfToken = (): string => {
                const metaTag = document.querySelector(
                    'meta[name="csrf-token"]'
                );
                if (metaTag) {
                    return metaTag.getAttribute("content") || "";
                }

                if (window.Laravel?.csrfToken) {
                    return window.Laravel.csrfToken;
                }

                const cookieMatch = document.cookie.match(/XSRF-TOKEN=([^;]+)/);
                if (cookieMatch) {
                    return decodeURIComponent(cookieMatch[1]);
                }

                return "";
            };

            const csrfToken = getCsrfToken();

            if (!csrfToken) {
                throw new Error(
                    "CSRF token not found. Please refresh the page."
                );
            }

            const requestBody = {
                amount: depositAmount,
                payment_method: paymentMethod,
            };

            const response = await fetch(route("wallet.deposit"), {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                    Accept: "application/json",
                    "X-CSRF-TOKEN": csrfToken,
                    "X-Requested-With": "XMLHttpRequest",
                },
                credentials: "include",
                body: JSON.stringify(requestBody),
            });

            if (response.status === 419) {
                toast.error(
                    "Your session has expired. Please refresh the page."
                );
                window.location.reload();
                return;
            }

            if (response.status === 401 || response.status === 403) {
                toast.error("You need to be logged in to perform this action.");
                window.location.href = route("login");
                return;
            }

            if (response.status === 422) {
                const text = await response.text();
                try {
                    const data = JSON.parse(text);
                    const errors =
                        data.errors || data.message || "Validation failed";

                    if (typeof errors === "object") {
                        const errorMessages = Object.values(errors).flat();
                        toast.error(
                            `Validation Error: ${errorMessages.join(", ")}`
                        );
                    } else {
                        toast.error(`Validation Error: ${errors}`);
                    }
                } catch {
                    toast.error("Validation failed. Please check your input.");
                }
                return;
            }

            if (!response.ok) {
                throw new Error(`Server error: ${response.status}`);
            }

            const text = await response.text();
            let data;
            try {
                data = JSON.parse(text);
            } catch (parseError) {
                console.error("Failed to parse JSON:", parseError);
                throw new Error("Server returned an invalid response.");
            }

            if (data.success && data.paymentUrl) {
                window.location.href = data.paymentUrl;
            } else if (data.error) {
                toast.error(`Payment Error: ${data.error}`);
            } else {
                toast.error(
                    "Failed to generate payment link. Please try again."
                );
            }
        } catch (error: any) {
            console.error("Deposit error:", error);
            toast.error(
                error.message || "Failed to initiate payment. Please try again."
            );
        } finally {
            setIsLoading(false);
        }
    };

    const handleWithdraw = async () => {
        if (!withdrawalData.amount || parseFloat(withdrawalData.amount) < 100) {
            toast.warning("Minimum withdrawal amount is ₦100");
            return;
        }

        if (
            !withdrawalData.bank_code ||
            !withdrawalData.account_number ||
            !withdrawalData.account_name
        ) {
            toast.warning("Please fill all bank details");
            return;
        }

        setIsWithdrawing(true);

        try {
            const csrfToken =
                document
                    .querySelector('meta[name="csrf-token"]')
                    ?.getAttribute("content") || "";

            const response = await fetch(route("wallet.withdraw"), {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                    "X-CSRF-TOKEN": csrfToken,
                    "X-Requested-With": "XMLHttpRequest",
                },
                body: JSON.stringify(withdrawalData),
            });

            const data = await response.json();

            if (response.ok) {
                toast.success("Withdrawal request submitted successfully!");
                setShowWithdrawModal(false);
                window.location.reload();
            } else {
                toast.error(
                    data.error || "Withdrawal failed. Please try again."
                );
            }
        } catch (error) {
            console.error("Withdrawal error:", error);
            toast.error("An error occurred. Please try again.");
        } finally {
            setIsWithdrawing(false);
        }
    };

    const copyToClipboard = async (
        text: string | undefined,
        label: string = "Account number"
    ) => {
        if (!text) {
            toast.error("No text to copy");
            return;
        }

        try {
            await navigator.clipboard.writeText(text);
            toast.success(`${label} copied to clipboard!`, {
                icon: "📋",
            });
        } catch (err) {
            console.error("Failed to copy:", err);
            toast.error("Failed to copy to clipboard");
        }
    };

    const getKycBadge = () => {
        switch (user.kyc_status) {
            case "api_pending":
                return (
                    <Badge
                        variant="outline"
                        className="bg-yellow-50 text-yellow-700 border-yellow-200"
                    >
                        KYC Submitted
                    </Badge>
                );
            case "api_verified":
                return (
                    <Badge
                        variant="outline"
                        className="bg-blue-50 text-blue-700 border-blue-200"
                    >
                        Awaiting Admin Approval
                    </Badge>
                );
            case "admin_approved":
                return (
                    <Badge
                        variant="outline"
                        className="bg-green-50 text-green-700 border-green-200"
                    >
                        Fully Verified
                    </Badge>
                );
            default:
                return (
                    <Badge
                        variant="outline"
                        className="bg-red-50 text-red-700 border-red-200"
                    >
                        Not Verified
                    </Badge>
                );
        }
    };

    const products = [
        {
            name: "Buy Data",
            color: "blue",
            desc: "Mobile & Internet Data",
            route: "#",
        },
        {
            name: "Buy Airtime",
            color: "green",
            desc: "Top-up Recharge",
            route: "#",
        },
        {
            name: "Airtime 2 Cash",
            color: "purple",
            desc: "Convert Airtime",
            route: "#",
        },
        {
            name: "Electricity Bill",
            color: "yellow",
            desc: "Pay Utility Bills",
            route: "#",
        },
    ];

    return (
        <AppLayout pageTitle="Dashboard">
            <Head title="Dashboard" />

            <ToastContainer
                position="top-right"
                autoClose={5000}
                hideProgressBar={false}
                newestOnTop={false}
                closeOnClick
                rtl={false}
                pauseOnFocusLoss
                draggable
                pauseOnHover
                theme="light"
            />

            <div className="p-6 space-y-8">
                {/* Welcome Section */}
                <div className="space-y-4">
                    <div className="flex items-center justify-between">
                        <div>
                            <h1 className="text-3xl font-bold tracking-tight">
                                Welcome back, {user.name}!
                            </h1>
                            <p className="text-muted-foreground">
                                Here's what's happening with your account today.
                            </p>
                        </div>
                        {getKycBadge()}
                    </div>

                    {(!user.kyc_status || user.kyc_status === "unverified") && (
                        <Alert className="bg-amber-50 border-amber-200">
                            <AlertTriangle className="h-4 w-4 text-amber-600" />
                            <AlertTitle className="text-amber-800">
                                No Linked Account Found
                            </AlertTitle>
                            <AlertDescription className="text-amber-700">
                                Complete your KYC using BVN or NIN to generate
                                your virtual account.
                                <Button
                                    variant="link"
                                    className="p-0 h-auto"
                                    asChild
                                >
                                    <a href={route("profile.edit")}>
                                        Click here to get started.
                                    </a>
                                </Button>
                            </AlertDescription>
                        </Alert>
                    )}
                </div>

                {/* Stats Cards */}
                <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                    {/* Wallet Card */}
                    <Card className="border-blue-200 shadow-sm">
                        <CardHeader className="pb-3">
                            <CardTitle className="text-lg text-blue-700">
                                Wallet Balance
                            </CardTitle>
                            <CardDescription>
                                Your current available balance
                            </CardDescription>
                        </CardHeader>
                        <CardContent>
                            <div className="text-3xl font-bold text-blue-800">
                                ₦{wallet.balance.toLocaleString()}
                            </div>
                            <p className="text-sm text-muted-foreground mt-1">
                                {wallet.currency}
                            </p>
                        </CardContent>
                        <CardFooter className="flex space-x-2">
                            <Dialog
                                open={showModal}
                                onOpenChange={setShowModal}
                            >
                                <DialogTrigger asChild>
                                    <Button className="flex-1 bg-blue-600 hover:bg-blue-700">
                                        Fund Wallet
                                    </Button>
                                </DialogTrigger>
                                <DialogContent className="sm:max-w-[500px]">
                                    <DialogHeader>
                                        <DialogTitle>
                                            Wallet Deposit Methods
                                        </DialogTitle>
                                        <DialogDescription>
                                            Kindly select a method of your
                                            choice, it's all seamless to use.
                                        </DialogDescription>
                                    </DialogHeader>

                                    <Tabs
                                        defaultValue="static"
                                        className="w-full"
                                    >
                                        <TabsList className="grid w-full grid-cols-2">
                                            <TabsTrigger value="static">
                                                Static Account
                                            </TabsTrigger>
                                            <TabsTrigger value="others">
                                                Others
                                            </TabsTrigger>
                                        </TabsList>

                                        <TabsContent
                                            value="static"
                                            className="space-y-4"
                                        >
                                            <p className="text-sm text-muted-foreground">
                                                You can send money to any of the
                                                accounts below for instant
                                                wallet deposit.
                                            </p>

                                            {user.kyc_status ===
                                                "admin_approved" &&
                                            staticAccount ? (
                                                <Card>
                                                    <CardHeader className="pb-3">
                                                        <CardTitle className="text-sm">
                                                            Your Personal
                                                            Virtual Account
                                                        </CardTitle>
                                                    </CardHeader>
                                                    <CardContent className="space-y-2">
                                                        <div className="flex justify-between">
                                                            <span className="text-sm font-medium">
                                                                Account Number:
                                                            </span>
                                                            <span className="font-mono">
                                                                {
                                                                    staticAccount.accountNumber
                                                                }
                                                            </span>
                                                        </div>
                                                        <div className="flex justify-between">
                                                            <span className="text-sm font-medium">
                                                                Bank:
                                                            </span>
                                                            <span>
                                                                {
                                                                    staticAccount.bankName
                                                                }
                                                            </span>
                                                        </div>
                                                        <div className="flex justify-between">
                                                            <span className="text-sm font-medium">
                                                                Account Name:
                                                            </span>
                                                            <span>
                                                                {
                                                                    staticAccount.accountName
                                                                }
                                                            </span>
                                                        </div>
                                                    </CardContent>
                                                    <CardFooter>
                                                        <Button
                                                            className="w-full"
                                                            onClick={() =>
                                                                copyToClipboard(
                                                                    staticAccount.accountNumber,
                                                                    "Account number"
                                                                )
                                                            }
                                                        >
                                                            <Copy className="mr-2 h-4 w-4" />
                                                            Copy Account Number
                                                        </Button>
                                                    </CardFooter>
                                                </Card>
                                            ) : (
                                                <Alert>
                                                    <AlertTitle>
                                                        No Linked Account Found
                                                    </AlertTitle>
                                                    <AlertDescription>
                                                        Complete your KYC using
                                                        BVN or NIN to generate
                                                        your virtual account.
                                                    </AlertDescription>
                                                </Alert>
                                            )}
                                        </TabsContent>

                                        <TabsContent
                                            value="others"
                                            className="space-y-4"
                                        >
                                            <div className="space-y-2">
                                                <Label htmlFor="amount">
                                                    Enter amount (min: ₦100)
                                                </Label>
                                                <Input
                                                    id="amount"
                                                    type="number"
                                                    value={amount}
                                                    onChange={(e) =>
                                                        setAmount(
                                                            e.target.value
                                                        )
                                                    }
                                                    placeholder="₦0.00"
                                                    min="100"
                                                />
                                                <p className="text-xs text-muted-foreground">
                                                    You will be redirected to
                                                    the payment gateway
                                                </p>
                                            </div>

                                            <Button
                                                className="w-full bg-red-600 hover:bg-red-700"
                                                onClick={() =>
                                                    handleDeposit("monnify")
                                                }
                                                disabled={isLoading}
                                            >
                                                {isLoading && (
                                                    <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                                                )}
                                                Proceed to Payment
                                            </Button>

                                            <Separator />

                                            <div className="space-y-4">
                                                <h4 className="font-semibold">
                                                    FOR MANUAL FUNDING
                                                </h4>
                                                <p className="text-sm text-muted-foreground">
                                                    Pay directly into our
                                                    company account
                                                </p>

                                                {companyAccountDetails ? (
                                                    <Card>
                                                        <CardContent className="pt-6 space-y-2">
                                                            <div className="flex justify-between">
                                                                <span className="text-sm font-medium">
                                                                    Account:
                                                                </span>
                                                                <span className="font-mono">
                                                                    {
                                                                        companyAccountDetails.account_number
                                                                    }
                                                                </span>
                                                            </div>
                                                            <div className="flex justify-between">
                                                                <span className="text-sm font-medium">
                                                                    Bank:
                                                                </span>
                                                                <span>
                                                                    {
                                                                        companyAccountDetails.bank_name
                                                                    }
                                                                </span>
                                                            </div>
                                                            <div className="flex justify-between">
                                                                <span className="text-sm font-medium">
                                                                    Name:
                                                                </span>
                                                                <span>
                                                                    {
                                                                        companyAccountDetails.account_name
                                                                    }
                                                                </span>
                                                            </div>
                                                        </CardContent>
                                                        <CardFooter>
                                                            <Button
                                                                className="w-full"
                                                                onClick={() =>
                                                                    copyToClipboard(
                                                                        companyAccountDetails.account_number,
                                                                        "Company account number"
                                                                    )
                                                                }
                                                            >
                                                                <Copy className="mr-2 h-4 w-4" />
                                                                Copy Account
                                                                Number
                                                            </Button>
                                                        </CardFooter>
                                                    </Card>
                                                ) : (
                                                    <Alert variant="destructive">
                                                        <AlertTriangle className="h-4 w-4" />
                                                        <AlertTitle>
                                                            Details Unavailable
                                                        </AlertTitle>
                                                        <AlertDescription>
                                                            Company account
                                                            details are not
                                                            available at the
                                                            moment.
                                                        </AlertDescription>
                                                    </Alert>
                                                )}
                                            </div>
                                        </TabsContent>
                                    </Tabs>
                                </DialogContent>
                            </Dialog>

                            <Dialog
                                open={showWithdrawModal}
                                onOpenChange={setShowWithdrawModal}
                            >
                                <DialogTrigger asChild>
                                    <Button
                                        variant="outline"
                                        className="flex-1 border-green-600 text-green-700 hover:bg-green-50"
                                        disabled={wallet.balance < 100}
                                    >
                                        Withdraw
                                    </Button>
                                </DialogTrigger>
                                <DialogContent className="sm:max-w-[425px]">
                                    <DialogHeader>
                                        <DialogTitle>
                                            Withdraw Funds
                                        </DialogTitle>
                                        <DialogDescription>
                                            Transfer money from your wallet to
                                            your bank account.
                                        </DialogDescription>
                                    </DialogHeader>

                                    <div className="space-y-4">
                                        <div className="space-y-2">
                                            <Label htmlFor="withdraw-amount">
                                                Amount (₦)
                                            </Label>
                                            <Input
                                                id="withdraw-amount"
                                                type="number"
                                                min="100"
                                                value={withdrawalData.amount}
                                                onChange={(e) =>
                                                    setWithdrawalData(
                                                        (prev) => ({
                                                            ...prev,
                                                            amount: e.target
                                                                .value,
                                                        })
                                                    )
                                                }
                                                placeholder="Minimum ₦100"
                                            />
                                        </div>

                                        <div className="space-y-2">
                                            <Label htmlFor="bank">Bank</Label>
                                            <Select
                                                value={withdrawalData.bank_code}
                                                onValueChange={(value) =>
                                                    setWithdrawalData(
                                                        (prev) => ({
                                                            ...prev,
                                                            bank_code: value,
                                                        })
                                                    )
                                                }
                                                disabled={
                                                    loadingBanks ||
                                                    banks.length === 0
                                                }
                                            >
                                                <SelectTrigger>
                                                    <SelectValue
                                                        placeholder={
                                                            loadingBanks
                                                                ? "Loading banks..."
                                                                : "Select Bank"
                                                        }
                                                    />
                                                </SelectTrigger>
                                                <SelectContent>
                                                    {banks.map((bank) => (
                                                        <SelectItem
                                                            key={bank.code}
                                                            value={bank.code}
                                                        >
                                                            {bank.name}
                                                        </SelectItem>
                                                    ))}
                                                </SelectContent>
                                            </Select>
                                            {banksError && (
                                                <p className="text-sm text-red-600">
                                                    {banksError}
                                                </p>
                                            )}
                                        </div>

                                        <div className="space-y-2">
                                            <Label htmlFor="account-number">
                                                Account Number
                                            </Label>
                                            <div className="flex space-x-2">
                                                <Input
                                                    id="account-number"
                                                    value={
                                                        withdrawalData.account_number
                                                    }
                                                    onChange={(e) =>
                                                        setWithdrawalData(
                                                            (prev) => ({
                                                                ...prev,
                                                                account_number:
                                                                    e.target
                                                                        .value,
                                                            })
                                                        )
                                                    }
                                                    placeholder="Account Number"
                                                />
                                                <Button
                                                    onClick={
                                                        validateBankAccount
                                                    }
                                                    disabled={
                                                        isValidatingAccount ||
                                                        !withdrawalData.bank_code ||
                                                        !withdrawalData.account_number
                                                    }
                                                    size="sm"
                                                >
                                                    {isValidatingAccount
                                                        ? "Validating..."
                                                        : "Validate"}
                                                </Button>
                                            </div>
                                        </div>

                                        <div className="space-y-2">
                                            <Label htmlFor="account-name">
                                                Account Name
                                            </Label>
                                            <Input
                                                id="account-name"
                                                value={
                                                    withdrawalData.account_name
                                                }
                                                readOnly
                                                placeholder="Will auto-fill after validation"
                                                className="bg-muted"
                                            />
                                        </div>
                                    </div>

                                    <DialogFooter>
                                        <Button
                                            variant="outline"
                                            onClick={() =>
                                                setShowWithdrawModal(false)
                                            }
                                        >
                                            Cancel
                                        </Button>
                                        <Button
                                            onClick={handleWithdraw}
                                            disabled={
                                                isWithdrawing ||
                                                !withdrawalData.account_name
                                            }
                                            className="bg-green-600 hover:bg-green-700"
                                        >
                                            {isWithdrawing && (
                                                <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                                            )}
                                            {isWithdrawing
                                                ? "Processing..."
                                                : "Withdraw"}
                                        </Button>
                                    </DialogFooter>
                                </DialogContent>
                            </Dialog>
                        </CardFooter>
                    </Card>

                    {/* KYC Card */}
                    <Card className="border-amber-200 shadow-sm">
                        <CardHeader className="pb-3">
                            <CardTitle className="text-lg text-amber-700">
                                KYC Status
                            </CardTitle>
                            <CardDescription>
                                Complete your verification
                            </CardDescription>
                        </CardHeader>
                        <CardContent>
                            <div className="flex items-center space-x-2 mb-4">
                                {user.kyc_status === "admin_approved" ? (
                                    <CheckCircle className="h-5 w-5 text-green-600" />
                                ) : (
                                    <AlertTriangle className="h-5 w-5 text-amber-600" />
                                )}
                                <span className="font-medium">
                                    {user.kyc_status === "admin_approved"
                                        ? "Verified"
                                        : "Verification Required"}
                                </span>
                            </div>
                        </CardContent>
                        <CardFooter>
                            <Button
                                className="w-full bg-amber-600 hover:bg-amber-700"
                                asChild
                            >
                                <a href={route("profile.edit")}>Update KYC</a>
                            </Button>
                        </CardFooter>
                    </Card>

                    {/* Package Card */}
                    <Card className="border-purple-200 shadow-sm">
                        <CardHeader className="pb-3">
                            <CardTitle className="text-lg text-purple-700">
                                Current Package
                            </CardTitle>
                            <CardDescription>
                                SMART User Package
                            </CardDescription>
                        </CardHeader>
                        <CardContent>
                            <p className="text-sm text-muted-foreground mb-2">
                                Upgrade your account and enjoy more discounts
                            </p>
                        </CardContent>
                        <CardFooter>
                            <Button className="w-full bg-purple-600 hover:bg-purple-700">
                                Upgrade Package
                            </Button>
                        </CardFooter>
                    </Card>
                </div>

                {/* Products & Services */}
                <div>
                    <h2 className="text-2xl font-bold tracking-tight mb-6">
                        Our Products & Services
                    </h2>
                    <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4">
                        {products.map((product) => (
                            <Card
                                key={product.name}
                                className="hover:shadow-md transition-shadow"
                            >
                                <CardHeader className="pb-3">
                                    <CardTitle className="text-lg">
                                        {product.name}
                                    </CardTitle>
                                    <CardDescription>
                                        {product.desc}
                                    </CardDescription>
                                </CardHeader>
                                <CardFooter>
                                    <Button
                                        className="w-full"
                                        variant="outline"
                                        asChild
                                    >
                                        <a href={product.route}>Purchase</a>
                                    </Button>
                                </CardFooter>
                            </Card>
                        ))}
                    </div>
                </div>
            </div>
        </AppLayout>
    );
}
