import { AppSidebar } from "@/Layouts/Partials/AppSidebar";
import {
    Breadcrumb,
    BreadcrumbItem,
    BreadcrumbLink,
    BreadcrumbList,
    BreadcrumbPage,
    BreadcrumbSeparator,
} from "@/Components/ui/breadcrumb";
import { Separator } from "@/Components/ui/separator";
import {
    SidebarInset,
    SidebarProvider,
    SidebarTrigger,
} from "@/Components/ui/sidebar";
import { ReactNode, useEffect, useState } from "react";
import { Head, usePage, Link, router } from "@inertiajs/react";
import ApplicationLogo from "@/Components/ApplicationLogo";
import NotificationDropdown from "@/Components/NotificationDropdown";
import NotificationModal from "@/Components/NotificationModal";

// Import the icons and components you need
import { LogOut, User, MessageCircle } from "lucide-react";
import {
    DropdownMenu,
    DropdownMenuContent,
    DropdownMenuItem,
    DropdownMenuSeparator,
    DropdownMenuLabel,
    DropdownMenuGroup,
    DropdownMenuTrigger,
} from "@/Components/ui/dropdown-menu";
import { Button } from "@/Components/ui/button";
import { Avatar, AvatarFallback, AvatarImage } from "@/Components/ui/avatar";

interface PageProps {
    csrf_token?: string;
    flash?: {
        show_notifications?: boolean;
        has_unread_messages?: boolean;
        show_critical_notifications?: boolean;
        notification?: {
            title: string;
            message: string;
            type: string;
        };
    };
    unread_notifications_count?: number;
    auth?: {
        user?: any;
    };
    [key: string]: any;
}

interface AppLayoutProps {
    children: ReactNode;
    pageTitle?: string;
}

// Add this interface for NotificationDropdown props
interface NotificationDropdownProps {
    initialUnreadCount?: number;
}

declare global {
    interface Window {
        csrfToken?: string;
        showNotificationModal?: () => void;
        hideNotificationModal?: () => void;
    }
}

export default function AppLayout({ children, pageTitle }: AppLayoutProps) {
    const { props } = usePage<PageProps>();
    const [showModal, setShowModal] = useState(false);
    const [initialUnreadCount, setInitialUnreadCount] = useState(0);
    const [isHoveringWhatsapp, setIsHoveringWhatsapp] = useState(false);

    const unreadCount = props.unread_notifications_count || 0;
    const user = props.auth?.user || {};

    // WhatsApp configuration
    const whatsappPhone = "2349015452077";
    const whatsappMessage = encodeURIComponent(
        "Hello, I need help with Hopay Sub"
    );
    const whatsappUrl = `https://wa.me/${whatsappPhone}?text=${whatsappMessage}`;

    // Function to get user initials
    const getInitials = (name: string) => {
        if (!name) return "U";
        return (
            name
                .split(" ")
                .map((word) => word[0] || "")
                .join("")
                .toUpperCase()
                .slice(0, 2) || "US"
        );
    };

    useEffect(() => {
        setInitialUnreadCount(unreadCount);

        window.showNotificationModal = () => {
            setShowModal(true);
            document.body.style.overflow = "hidden";
        };

        window.hideNotificationModal = () => {
            setShowModal(false);
            document.body.style.overflow = "auto";
        };

        return () => {
            delete window.showNotificationModal;
            delete window.hideNotificationModal;
            document.body.style.overflow = "auto";
        };
    }, []);

    useEffect(() => {
        if (props.flash?.show_notifications && unreadCount > 0) {
            const timer = setTimeout(() => {
                setShowModal(true);
                document.body.style.overflow = "hidden";
            }, 1500);

            return () => clearTimeout(timer);
        }

        if (props.flash?.notification) {
            const timer = setTimeout(() => {
                console.log("Flash notification:", props.flash?.notification);
            }, 1000);

            return () => clearTimeout(timer);
        }
    }, [props.flash, unreadCount]);

    useEffect(() => {
        const handleEscapeKey = (event: KeyboardEvent) => {
            if (event.key === "Escape" && showModal) {
                setShowModal(false);
                document.body.style.overflow = "auto";
            }
        };

        document.addEventListener("keydown", handleEscapeKey);
        return () => document.removeEventListener("keydown", handleEscapeKey);
    }, [showModal]);

    return (
        <>
            <Head>
                <title>{pageTitle || "Dashboard"}</title>
                <meta name="csrf-token" content={props.csrf_token || ""} />
                <script
                    dangerouslySetInnerHTML={{
                        __html: `
                            window.csrfToken = "${props.csrf_token || ""}";
                            window.Laravel = window.Laravel || {};
                            window.Laravel.csrfToken = "${
                                props.csrf_token || ""
                            }";
                            window.initialUnreadCount = ${unreadCount};
                        `,
                    }}
                />
            </Head>

            {/* WhatsApp Floating Support Button */}
            <a
                href={whatsappUrl}
                target="_blank"
                rel="noopener noreferrer"
                className="fixed bottom-6 right-6 z-50 group"
                title="Chat with admin on WhatsApp"
                onMouseEnter={() => setIsHoveringWhatsapp(true)}
                onMouseLeave={() => setIsHoveringWhatsapp(false)}
            >
                {/* Animated pulse ring */}
                <div className="absolute inset-0 animate-ping rounded-full bg-green-500 opacity-30"></div>

                {/* Main button */}
                <div className="relative bg-[#25D366] w-14 h-14 rounded-full flex items-center justify-center shadow-lg group-hover:shadow-2xl transition-all duration-300 group-hover:scale-110">
                    <MessageCircle className="w-7 h-7 text-white" />

                    {/* 24/7 Badge */}
                    <span className="absolute -top-1 -right-1 bg-red-500 text-white text-xs font-bold rounded-full w-6 h-6 flex items-center justify-center border-2 border-white animate-pulse">
                        24/7
                    </span>
                </div>

                {/* Tooltip */}
                <div
                    className={`absolute right-16 bottom-1/2 translate-y-1/2 bg-gray-900 text-white text-sm rounded-lg py-2 px-3 whitespace-nowrap shadow-lg transition-all duration-300 ${
                        isHoveringWhatsapp
                            ? "opacity-100 visible translate-x-0"
                            : "opacity-0 invisible translate-x-2"
                    }`}
                >
                    <div className="flex items-center gap-2">
                        <MessageCircle className="w-4 h-4" />
                        <span>
                            Need help?
                            <br />
                            Chat with admin
                        </span>
                    </div>
                    <div className="absolute top-1/2 right-[-4px] transform -translate-y-1/2">
                        <div className="w-2 h-2 bg-gray-900 rotate-45"></div>
                    </div>
                </div>
            </a>

            <SidebarProvider>
                <AppSidebar />
                <SidebarInset>
                    <header className="sticky top-0 z-40 flex h-16 shrink-0 items-center gap-4 bg-white border-b shadow-sm px-4 md:px-6">
                        <div className="flex items-center gap-3 md:gap-2 flex-1">
                            <SidebarTrigger className="h-9 w-9" />

                            <div className="md:hidden flex items-center gap-2">
                                <ApplicationLogo className="size-7 text-blue-600 fill-current" />
                                <div>
                                    <div className="text-l text-black-500">
                                        {pageTitle || "Dashboard"}
                                    </div>
                                </div>
                            </div>

                            <Separator
                                orientation="vertical"
                                className="h-6 hidden md:block"
                            />
                            <Breadcrumb className="hidden md:block">
                                <BreadcrumbList>
                                    <BreadcrumbItem>
                                        <BreadcrumbPage className="text-lg font-semibold">
                                            {pageTitle || "Dashboard"}
                                        </BreadcrumbPage>
                                    </BreadcrumbItem>
                                </BreadcrumbList>
                            </Breadcrumb>
                        </div>

                        <div className="flex items-center gap-3">
                            <NotificationDropdown
                                initialUnreadCount={unreadCount}
                            />

                            {/* User Avatar Dropdown - Only shows avatar by default */}
                            <DropdownMenu>
                                <DropdownMenuTrigger asChild>
                                    <Button
                                        variant="ghost"
                                        className="p-1 rounded-lg hover:bg-gray-100 transition-colors"
                                    >
                                        <Avatar className="h-9 w-9 rounded-lg border border-gray-200">
                                            <AvatarImage
                                                src={
                                                    user.avatar_url ||
                                                    user.avatar
                                                }
                                                alt={user.name}
                                                onError={(e) => {
                                                    console.error(
                                                        "Avatar image failed to load:",
                                                        e.currentTarget.src
                                                    );
                                                    e.currentTarget.style.display =
                                                        "none";
                                                }}
                                            />
                                            <AvatarFallback className="rounded-lg bg-primary/10 text-primary font-semibold">
                                                {getInitials(user.name)}
                                            </AvatarFallback>
                                        </Avatar>
                                    </Button>
                                </DropdownMenuTrigger>
                                <DropdownMenuContent
                                    className="w-56 rounded-lg"
                                    align="end"
                                >
                                    <DropdownMenuLabel className="p-0 font-normal">
                                        <div className="flex items-center gap-3 px-2 py-3">
                                            <Avatar className="h-12 w-12 rounded-lg">
                                                <AvatarImage
                                                    src={
                                                        user.avatar_url ||
                                                        user.avatar
                                                    }
                                                    alt={user.name}
                                                    onError={(e) => {
                                                        console.error(
                                                            "Dropdown Avatar failed to load:",
                                                            e.currentTarget.src
                                                        );
                                                        e.currentTarget.style.display =
                                                            "none";
                                                    }}
                                                />
                                                <AvatarFallback className="rounded-lg bg-primary/10 text-primary font-semibold text-lg">
                                                    {getInitials(user.name)}
                                                </AvatarFallback>
                                            </Avatar>
                                            <div className="flex flex-col">
                                                <span className="font-medium text-sm">
                                                    {user.name}
                                                </span>
                                                <span className="text-xs text-gray-500">
                                                    {user.email}
                                                </span>
                                            </div>
                                        </div>
                                    </DropdownMenuLabel>
                                    <DropdownMenuSeparator />
                                    <DropdownMenuGroup>
                                        <Link href={route("profile.edit")}>
                                            <DropdownMenuItem>
                                                <User className="mr-2 h-4 w-4" />
                                                <span>Profile</span>
                                            </DropdownMenuItem>
                                        </Link>
                                    </DropdownMenuGroup>
                                    <DropdownMenuSeparator />
                                    <DropdownMenuItem
                                        onClick={() =>
                                            router.post(route("logout"))
                                        }
                                    >
                                        <LogOut className="mr-2 h-4 w-4" />
                                        <span>Log out</span>
                                    </DropdownMenuItem>
                                </DropdownMenuContent>
                            </DropdownMenu>

                            {unreadCount > 0 && (
                                <Link
                                    href="/notifications"
                                    className="hidden md:flex items-center gap-2 px-4 py-2 rounded-lg bg-blue-50 hover:bg-blue-100 text-blue-700 text-sm font-medium transition-colors"
                                >
                                    <svg
                                        className="w-5 h-5"
                                        fill="none"
                                        stroke="currentColor"
                                        viewBox="0 0 24 24"
                                    >
                                        <path
                                            strokeLinecap="round"
                                            strokeLinejoin="round"
                                            strokeWidth="2"
                                            d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2"
                                        />
                                    </svg>
                                    <span>Messages ({unreadCount})</span>
                                </Link>
                            )}
                        </div>
                    </header>

                    <div className="flex-1 overflow-auto">
                        <div className="flex flex-1 flex-col gap-6 p-4 md:p-6">
                            {children}
                        </div>
                    </div>

                    {showModal && (
                        <NotificationModal
                            onClose={() => {
                                setShowModal(false);
                                document.body.style.overflow = "auto";
                            }}
                            initialUnreadCount={unreadCount}
                        />
                    )}
                </SidebarInset>
            </SidebarProvider>
        </>
    );
}
