import { useState, useRef, useEffect } from "react";
import { router } from "@inertiajs/react";
import { Camera, Loader2, X } from "lucide-react";
import { Avatar, AvatarImage, AvatarFallback } from "@/Components/ui/avatar";
import { Button } from "@/Components/ui/button";

interface AvatarUploadProps {
    currentAvatar?: string;
    userName: string;
}

export default function AvatarUpload({
    currentAvatar,
    userName,
}: AvatarUploadProps) {
    const [preview, setPreview] = useState<string | null>(null);
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState<string | null>(null);
    const fileInputRef = useRef<HTMLInputElement>(null);

    // Store the current avatar in state to update it
    const [displayAvatar, setDisplayAvatar] = useState<string | undefined>(
        currentAvatar
    );

    // Update displayAvatar when currentAvatar prop changes
    useEffect(() => {
        setDisplayAvatar(currentAvatar);
    }, [currentAvatar]);

    const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
        const file = e.target.files?.[0];
        if (file) {
            // Validate file
            if (!file.type.startsWith("image/")) {
                setError("Please select an image file");
                return;
            }

            if (file.size > 2 * 1024 * 1024) {
                // 2MB
                setError("File size should be less than 2MB");
                return;
            }

            setError(null);
            const previewUrl = URL.createObjectURL(file);
            setPreview(previewUrl);

            // Upload immediately when file is selected
            uploadAvatar(file);
        }
    };

    const uploadAvatar = async (file: File) => {
        setLoading(true);
        setError(null);

        const formData = new FormData();
        formData.append("avatar", file);

        try {
            // Use axios directly
            const response = await window.axios.post(
                route("profile.avatar.update"),
                formData,
                {
                    headers: {
                        "Content-Type": "multipart/form-data",
                    },
                }
            );

            console.log("Upload response:", response.data);

            // Update the displayed avatar immediately
            if (response.data?.user?.avatar_url) {
                setDisplayAvatar(response.data.user.avatar_url);
            } else if (response.data?.avatar_url) {
                setDisplayAvatar(response.data.avatar_url);
            }

            // Clear preview
            setPreview(null);
            if (fileInputRef.current) {
                fileInputRef.current.value = "";
            }

            // Show success message
            alert("Avatar updated successfully!");

            // Optionally reload the page to get fresh data
            setTimeout(() => {
                router.reload({ only: ["auth.user"] });
            }, 1000);
        } catch (err: any) {
            setError(
                err.response?.data?.message ||
                    err.response?.data?.errors?.avatar?.[0] ||
                    "Failed to upload avatar"
            );
            console.error("Upload error:", err.response?.data || err);

            // Reset preview on error
            setPreview(null);
            if (fileInputRef.current) {
                fileInputRef.current.value = "";
            }
        } finally {
            setLoading(false);
        }
    };

    const handleDeleteAvatar = async () => {
        if (confirm("Are you sure you want to remove your avatar?")) {
            try {
                await window.axios.delete(route("profile.avatar.delete"));

                // Clear the displayed avatar
                setDisplayAvatar(undefined);

                // Reload the page to get fresh data
                setTimeout(() => {
                    router.reload({ only: ["auth.user"] });
                }, 500);
            } catch (err) {
                console.error("Delete error:", err);
                alert("Failed to remove avatar. Please try again.");
            }
        }
    };

    const getInitials = (name: string) => {
        return name
            .split(" ")
            .map((word) => word[0])
            .join("")
            .toUpperCase()
            .slice(0, 2);
    };

    return (
        <div className="space-y-4">
            <div className="flex items-center space-x-4">
                <div className="relative">
                    <Avatar className="h-24 w-24 border-2 border-gray-200">
                        <AvatarImage
                            src={preview || displayAvatar}
                            alt={userName}
                            className="object-cover"
                            onError={(e) => {
                                // If image fails to load, fallback to initials
                                console.log(
                                    "Image failed to load:",
                                    preview || displayAvatar
                                );
                                e.currentTarget.style.display = "none";
                            }}
                        />
                        <AvatarFallback className="text-lg bg-primary/10 text-primary font-semibold">
                            {getInitials(userName)}
                        </AvatarFallback>
                    </Avatar>

                    <button
                        type="button"
                        onClick={() => fileInputRef.current?.click()}
                        disabled={loading}
                        className="absolute bottom-0 right-0 rounded-full bg-primary p-2 text-white shadow-lg hover:bg-primary/90 transition-colors disabled:opacity-50"
                        title="Change avatar"
                    >
                        {loading ? (
                            <Loader2 className="h-4 w-4 animate-spin" />
                        ) : (
                            <Camera className="h-4 w-4" />
                        )}
                    </button>
                </div>

                <div className="space-y-2">
                    <h3 className="text-sm font-medium text-gray-900">
                        Profile Photo
                    </h3>
                    <p className="text-xs text-gray-500">
                        Click the camera icon to upload a new photo. JPG, PNG,
                        GIF or WebP, max 2MB.
                    </p>
                    {error && <p className="text-xs text-red-600">{error}</p>}
                    {loading && (
                        <p className="text-xs text-blue-600">Uploading...</p>
                    )}
                </div>
            </div>

            <input
                type="file"
                ref={fileInputRef}
                onChange={handleFileChange}
                accept="image/*"
                className="hidden"
                disabled={loading}
            />

            {displayAvatar && !preview && !loading && (
                <Button
                    type="button"
                    variant="destructive"
                    size="sm"
                    onClick={handleDeleteAvatar}
                    className="inline-flex items-center"
                >
                    <X className="h-4 w-4 mr-2" />
                    Remove Avatar
                </Button>
            )}
        </div>
    );
}
