<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class NetworkSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $this->command->info('📶 Seeding Nigerian telecom networks...');

        // Network data matching YOUR table structure
        $networks = [
            [
                'code' => '01',
                'name' => 'MTN',
                'logo' => 'mtn.png',
                'airtime_discount' => 3.00, // 3% discount from ClubKonnect
                'is_active' => true,
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'code' => '02',
                'name' => 'GLO',
                'logo' => 'glo.png',
                'airtime_discount' => 8.00, // 8% discount from ClubKonnect
                'is_active' => true,
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'code' => '04',
                'name' => 'AIRTEL',
                'logo' => 'airtel.png',
                'airtime_discount' => 3.00, // 3% discount from ClubKonnect
                'is_active' => true,
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'code' => '03',
                'name' => '9MOBILE',
                'logo' => '9mobile.png',
                'airtime_discount' => 7.00, // 7% discount from ClubKonnect
                'is_active' => true,
                'created_at' => now(),
                'updated_at' => now(),
            ],
        ];

        try {
            // Clear the table first (truncate removes all data)
            DB::table('networks')->truncate();

            // Insert networks into database
            DB::table('networks')->insert($networks);

            $this->command->info('✅ Networks seeded successfully!');

            // Display pretty table
            $this->displayNetworkTable($networks);

            // Log the seeding
            Log::info('NetworkSeeder executed successfully', [
                'networks_seeded' => count($networks),
                'timestamp' => now(),
            ]);
        } catch (\Exception $e) {
            $this->command->error('❌ Error seeding networks: ' . $e->getMessage());
            Log::error('NetworkSeeder failed: ' . $e->getMessage());

            // Show what columns your table actually has
            $this->command->info("\n💡 Your networks table has these columns:");
            $columns = DB::getSchemaBuilder()->getColumnListing('networks');
            $this->command->info('   ' . implode(', ', $columns));
        }
    }

    /**
     * Display networks in a pretty table format
     */
    private function displayNetworkTable(array $networks): void
    {
        $headers = ['Code', 'Name', 'Logo', 'Airtime Discount', 'Status'];
        $rows = [];

        foreach ($networks as $network) {
            $rows[] = [
                $network['code'],
                $network['name'],
                $network['logo'] ?: 'N/A',
                $network['airtime_discount'] . '%',
                $network['is_active'] ? '✅ Active' : '❌ Inactive',
            ];
        }

        $this->command->table($headers, $rows);

        // Additional info
        $this->command->info("\n📊 Network Information:");
        $this->command->info("   • Total Networks: " . count($networks));
        $this->command->info("   • Highest Airtime Discount: GLO (8%)");
        $this->command->info("   • ClubKonnect API Codes: 01=MTN, 02=GLO, 04=AIRTEL, 03=9MOBILE");
        $this->command->info("   • Table Structure: " . implode(', ', array_keys($networks[0])));
    }
}
