<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use App\Models\Product;
use App\Models\Network;

class ClubKonnectAirtimeSeeder extends Seeder
{
    public function run(): void
    {
        $this->command->info('📞 Seeding ClubKonnect Airtime Products...');

        // Get networks to map codes to names
        $networks = Network::all()->keyBy('code');

        $airtimeProducts = [
            // MTN Airtime
            [
                'name' => 'MTN Airtime',
                'type' => 'airtime',
                'provider' => 'clubkonnect',
                'network_code' => '01',
                'network_name' => 'MTN',
                'provider_network_code' => '01',
                'provider_plan_id' => 'airtime_mtn',
                'plan_code' => 'airtime_mtn',
                'description' => 'MTN Airtime Top-up',
                'price' => 100.00, // Minimum amount
                'validity' => null,
                'data_size' => null,
                'is_active' => true,
                'is_available' => true,
                'provider_data' => [
                    'discount' => '3%',
                    'min_amount' => 50,
                    'max_amount' => 200000,
                    'bonus_types' => [
                        '01' => 'MTN Awuf (400%)',
                        '02' => 'MTN Garabasa (1,000%)',
                    ],
                ],
            ],

            // GLO Airtime
            [
                'name' => 'GLO Airtime',
                'type' => 'airtime',
                'provider' => 'clubkonnect',
                'network_code' => '02',
                'network_name' => 'GLO',
                'provider_network_code' => '02',
                'provider_plan_id' => 'airtime_glo',
                'plan_code' => 'airtime_glo',
                'description' => 'GLO Airtime Top-up',
                'price' => 100.00,
                'validity' => null,
                'data_size' => null,
                'is_active' => true,
                'is_available' => true,
                'provider_data' => [
                    'discount' => '8%',
                    'min_amount' => 50,
                    'max_amount' => 200000,
                ],
            ],

            // Airtel Airtime
            [
                'name' => 'AIRTEL Airtime',
                'type' => 'airtime',
                'provider' => 'clubkonnect',
                'network_code' => '04',
                'network_name' => 'AIRTEL',
                'provider_network_code' => '04',
                'provider_plan_id' => 'airtime_airtel',
                'plan_code' => 'airtime_airtel',
                'description' => 'AIRTEL Airtime Top-up',
                'price' => 100.00,
                'validity' => null,
                'data_size' => null,
                'is_active' => true,
                'is_available' => true,
                'provider_data' => [
                    'discount' => '3%',
                    'min_amount' => 50,
                    'max_amount' => 200000,
                ],
            ],

            // 9mobile Airtime
            [
                'name' => '9MOBILE Airtime',
                'type' => 'airtime',
                'provider' => 'clubkonnect',
                'network_code' => '03',
                'network_name' => '9MOBILE',
                'provider_network_code' => '03',
                'provider_plan_id' => 'airtime_9mobile',
                'plan_code' => 'airtime_9mobile',
                'description' => '9MOBILE Airtime Top-up',
                'price' => 100.00,
                'validity' => null,
                'data_size' => null,
                'is_active' => true,
                'is_available' => true,
                'provider_data' => [
                    'discount' => '7%',
                    'min_amount' => 50,
                    'max_amount' => 200000,
                ],
            ],
        ];

        $createdCount = 0;
        $updatedCount = 0;

        foreach ($airtimeProducts as $productData) {
            $product = Product::updateOrCreate(
                [
                    'provider' => 'clubkonnect',
                    'provider_plan_id' => $productData['provider_plan_id'],
                    'network_code' => $productData['network_code'],
                ],
                $productData
            );

            if ($product->wasRecentlyCreated) {
                $createdCount++;
            } else {
                $updatedCount++;
            }
        }

        $this->command->info("✅ ClubKonnect Airtime Products Seeded!");
        $this->command->info("   📥 New: {$createdCount}");
        $this->command->info("   🔄 Updated: {$updatedCount}");

        // Display table
        $this->command->table(
            ['Network', 'Type', 'Min Amount', 'Discount'],
            array_map(function ($p) {
                return [
                    $p['network_name'],
                    'Airtime',
                    '₦' . $p['price'],
                    $p['provider_data']['discount'] ?? 'N/A',
                ];
            }, $airtimeProducts)
        );
    }
}
