<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up()
    {
        // Create wallet_transactions table
        Schema::create('wallet_transactions', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->onDelete('cascade');
            $table->foreignId('wallet_id')->constrained()->onDelete('cascade');
            $table->string('reference')->unique();
            $table->decimal('amount', 15, 2);
            $table->enum('type', ['deposit', 'withdrawal', 'transfer', 'bonus', 'refund']);
            $table->enum('status', ['pending', 'success', 'failed', 'cancelled']);
            $table->string('payment_method')->nullable();
            $table->json('payment_details')->nullable();
            $table->json('metadata')->nullable();
            $table->timestamp('completed_at')->nullable();
            $table->timestamps();

            $table->index(['user_id', 'status']);
            $table->index(['wallet_id']);
            $table->index(['reference']);
            $table->index(['type', 'created_at']);
        });

        // Migrate existing deposit transactions
        DB::statement("
            INSERT INTO wallet_transactions 
            (id, user_id, wallet_id, reference, amount, type, status, 
             payment_method, metadata, created_at, updated_at)
            SELECT 
                t.id,
                t.user_id,
                w.id as wallet_id,
                t.reference,
                t.amount,
                'deposit' as type,
                t.status,
                CASE 
                    WHEN t.reference LIKE 'DEP_%' THEN 'manual'
                    ELSE NULL 
                END as payment_method,
                JSON_OBJECT(
                    'original_type', t.type,
                    'network', t.network,
                    'phone', t.phone,
                    'plan_code', t.plan_code,
                    'plan_description', t.plan_description,
                    'order_id', t.order_id,
                    'request_id', t.request_id,
                    'api_response', t.api_response,
                    'remark', t.remark,
                    'amount_charged', t.amount_charged,
                    'original_status', t.original_status
                ) as metadata,
                t.created_at,
                t.updated_at
            FROM transactions t
            LEFT JOIN wallets w ON t.user_id = w.user_id
            WHERE t.type = 'deposit'
        ");

        // Delete deposit transactions from transactions table
        DB::table('transactions')->where('type', 'deposit')->delete();

        // Update transactions table to only store service purchases
        Schema::table('transactions', function (Blueprint $table) {
            // Rename type column to service_type for clarity
            $table->renameColumn('type', 'service_type');

            // Add service-specific columns
            $table->string('service_provider')->nullable()->after('service_type');
            $table->json('service_details')->nullable()->after('service_provider');
            $table->foreignId('wallet_transaction_id')->nullable()->after('user_id');

            // Remove wallet-specific columns if they exist
            $table->dropColumn(['original_status']); // Not needed for purchases

            // Add foreign key constraint
            $table->foreign('wallet_transaction_id')
                ->references('id')
                ->on('wallet_transactions')
                ->onDelete('set null');

            // Add indexes
            $table->index(['service_type', 'status']);
            $table->index(['network', 'service_type']);
            $table->index(['plan_code']);
        });

        // Create transaction_fees table for tracking commissions/fees
        Schema::create('transaction_fees', function (Blueprint $table) {
            $table->id();
            $table->morphs('transaction'); // For both wallet_transactions and transactions
            $table->string('fee_type'); // platform_fee, provider_fee, commission, tax
            $table->decimal('amount', 10, 2);
            $table->decimal('percentage', 5, 2)->nullable();
            $table->string('description')->nullable();
            $table->timestamps();

            $table->index(['transaction_type', 'transaction_id']);
            $table->index(['fee_type']);
        });
    }

    public function down()
    {
        // Reverse the changes in transactions table first
        Schema::table('transactions', function (Blueprint $table) {
            $table->dropForeign(['wallet_transaction_id']);
            $table->dropColumn('wallet_transaction_id');
            $table->dropColumn('service_provider');
            $table->dropColumn('service_details');
            $table->renameColumn('service_type', 'type');
        });

        // Restore deposit transactions
        DB::statement("
            INSERT INTO transactions 
            (id, user_id, reference, type, status, amount, network, phone, 
             plan_code, plan_description, order_id, request_id, api_response, 
             remark, created_at, updated_at)
            SELECT 
                wt.id,
                wt.user_id,
                wt.reference,
                'deposit' as type,
                wt.status,
                wt.amount,
                JSON_UNQUOTE(JSON_EXTRACT(wt.metadata, '$.network')) as network,
                JSON_UNQUOTE(JSON_EXTRACT(wt.metadata, '$.phone')) as phone,
                JSON_UNQUOTE(JSON_EXTRACT(wt.metadata, '$.plan_code')) as plan_code,
                JSON_UNQUOTE(JSON_EXTRACT(wt.metadata, '$.plan_description')) as plan_description,
                JSON_UNQUOTE(JSON_EXTRACT(wt.metadata, '$.order_id')) as order_id,
                JSON_UNQUOTE(JSON_EXTRACT(wt.metadata, '$.request_id')) as request_id,
                JSON_UNQUOTE(JSON_EXTRACT(wt.metadata, '$.api_response')) as api_response,
                JSON_UNQUOTE(JSON_EXTRACT(wt.metadata, '$.remark')) as remark,
                wt.created_at,
                wt.updated_at
            FROM wallet_transactions wt
            WHERE wt.type = 'deposit'
        ");

        // Drop created tables
        Schema::dropIfExists('transaction_fees');
        Schema::dropIfExists('wallet_transactions');
    }
};
