<?php

namespace App\Services;

use App\Models\Product;
use App\Models\Transaction;
use App\Models\Wallet;
use App\Models\ApiProvider;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class PurchaseService
{
    public static function handle($user, Product $product, array $details, $maxRetries = 3)
    {
        // 🔴 LOG 1: ENTRY POINT
        \Log::info('🟢🟢🟢 PURCHASE SERVICE STARTED', [
            'user_id' => $user->id,
            'product_id' => $product->id,
            'product_name' => $product->name,
            'product_type' => $product->type,
            'price' => $product->price,
            'details' => $details,
            'timestamp' => now()->toDateTimeString(),
            'memory' => memory_get_usage() . ' bytes',
            'max_retries' => $maxRetries
        ]);

        try {
            // 🔴 LOG 2: DATABASE CONNECTION CHECK
            \Log::info('🟢🟢🟢 CHECKING DATABASE CONNECTION');
            DB::connection()->getPdo();
            \Log::info('🟢🟢🟢 DATABASE CONNECTION OK');
        } catch (\Exception $e) {
            \Log::error('🔴🔴🔴 DATABASE CONNECTION FAILED', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            throw new \Exception('Database connection failed: ' . $e->getMessage());
        }

        // Set timeout for entire operation
        set_time_limit(120); // 2 minutes max

        return DB::transaction(function () use ($user, $product, $details, $maxRetries) {
            // 🔴 LOG 3: INSIDE TRANSACTION
            \Log::info('🟢🟢🟢 INSIDE DATABASE TRANSACTION', [
                'step' => 'transaction_started',
                'time' => now()->toDateTimeString(),
                'memory' => memory_get_usage() . ' bytes'
            ]);

            try {
                // 🔴 LOG 4: WALLET CHECK
                \Log::info('🟢🟢🟢 CHECKING/CREATING WALLET', ['user_id' => $user->id]);

                $wallet = Wallet::firstOrCreate(['user_id' => $user->id]);

                \Log::info('🟢🟢🟢 WALLET FOUND/CREATED', [
                    'wallet_id' => $wallet->id,
                    'current_balance' => $wallet->balance,
                    'created_at' => $wallet->created_at
                ]);

                // Calculate final price
                $finalPrice = $product->discount ?
                    $product->price - ($product->price * $product->discount / 100) :
                    $product->price;

                \Log::info('🟢🟢🟢 PRICE CALCULATION', [
                    'original_price' => $product->price,
                    'discount' => $product->discount . '%',
                    'final_price' => $finalPrice,
                    'wallet_balance' => $wallet->balance
                ]);

                // Check wallet balance
                if ($wallet->balance < $finalPrice) {
                    \Log::error('🔴🔴🔴 INSUFFICIENT BALANCE', [
                        'balance_needed' => $finalPrice,
                        'balance_available' => $wallet->balance,
                        'shortfall' => $finalPrice - $wallet->balance
                    ]);
                    throw new \Exception('Insufficient wallet balance. Need: ' . $finalPrice . ', Have: ' . $wallet->balance);
                }

                // Create transaction
                $reference = 'TXN_' . time() . '_' . Str::random(6);

                \Log::info('🟢🟢🟢 CREATING TRANSACTION RECORD', [
                    'reference' => $reference,
                    'amount' => $finalPrice,
                    'status' => 'pending'
                ]);

                $transaction = Transaction::create([
                    'user_id' => $user->id,
                    'product_id' => $product->id,
                    'amount' => $finalPrice,
                    'reference' => $reference,
                    'status' => 'pending',
                    'details' => $details,
                ]);

                \Log::info('🟢🟢🟢 TRANSACTION CREATED SUCCESSFULLY', [
                    'transaction_id' => $transaction->id,
                    'reference' => $transaction->reference,
                    'created_at' => $transaction->created_at
                ]);

                $retryCount = 0;
                $lastError = null;
                $lastResponse = null;

                while ($retryCount <= $maxRetries) {
                    \Log::info('🟡🟡🟡 PURCHASE ATTEMPT', [
                        'attempt' => $retryCount + 1,
                        'max_attempts' => $maxRetries + 1,
                        'transaction_id' => $transaction->id
                    ]);

                    try {
                        // 🔴 LOG 5: BEFORE EXTERNAL SERVICE CALL
                        \Log::info('🟡🟡🟡 CALLING EXTERNAL VTU SERVICE', [
                            'step' => 'before_external_call',
                            'product_type' => $product->type,
                            'network' => $details['network'] ?? 'N/A',
                            'phone' => $details['phone'] ?? 'N/A',
                            'reference' => $transaction->reference
                        ]);

                        // Call external provider
                        $response = ExternalVtuService::send(
                            $product,
                            $details,
                            $transaction->reference,
                            $transaction
                        );

                        // 🔴 LOG 6: EXTERNAL SERVICE RESPONSE
                        \Log::info('🟡🟡🟡 EXTERNAL VTU SERVICE RESPONSE', [
                            'step' => 'external_response_received',
                            'success' => $response['success'] ?? false,
                            'message' => $response['message'] ?? 'No message',
                            'should_retry' => $response['should_retry'] ?? false,
                            'provider_code' => $response['provider_code'] ?? 'N/A',
                            'provider_id' => $response['provider_id'] ?? 'N/A',
                            'attempt' => $retryCount + 1
                        ]);

                        $lastResponse = $response;

                        if (!$response['success']) {
                            \Log::warning('🟡🟡🟡 EXTERNAL SERVICE FAILED', [
                                'message' => $response['message'],
                                'should_retry' => $response['should_retry'] ?? false,
                                'retry_after' => $response['retry_after'] ?? 'N/A'
                            ]);

                            // Check if we should retry
                            if (($response['should_retry'] ?? false) && $retryCount < $maxRetries) {
                                $retryCount++;
                                $waitTime = $response['retry_after'] ?? 5;

                                \Log::info('🟡🟡🟡 RETRYING AFTER DELAY', [
                                    'wait_seconds' => $waitTime,
                                    'next_attempt' => $retryCount,
                                    'max_retries' => $maxRetries
                                ]);

                                sleep($waitTime);
                                continue;
                            }

                            // Max retries reached or no retry flag
                            \Log::error('🔴🔴🔴 FINAL FAILURE - NO MORE RETRIES', [
                                'retry_count' => $retryCount,
                                'max_retries' => $maxRetries,
                                'failure_message' => $response['message']
                            ]);

                            $transaction->update([
                                'status' => 'failed',
                                'details' => array_merge($transaction->details ?? [], [
                                    'api_response' => $response,
                                    'failure_reason' => $response['message'],
                                    'failed_at' => now()->toDateTimeString(),
                                    'retry_count' => $retryCount,
                                    'last_attempt' => now()->toDateTimeString()
                                ])
                            ]);

                            throw new \Exception('VTU Service failed: ' . $response['message']);
                        }

                        // 🔴 LOG 7: SUCCESS - PROCESSING PAYMENT
                        \Log::info('🟢🟢🟢 EXTERNAL SERVICE SUCCESS - PROCESSING PAYMENT', [
                            'step' => 'processing_payment',
                            'provider_code' => $response['provider_code'] ?? 'N/A',
                            'provider_ref' => $response['provider_ref'] ?? 'N/A'
                        ]);

                        // SUCCESS: Deduct from wallet
                        $oldBalance = $wallet->balance;
                        $wallet->decrement('balance', $finalPrice);
                        $newBalance = $wallet->fresh()->balance;

                        \Log::info('🟢🟢🟢 WALLET UPDATED', [
                            'old_balance' => $oldBalance,
                            'amount_deducted' => $finalPrice,
                            'new_balance' => $newBalance,
                            'wallet_id' => $wallet->id
                        ]);

                        // Update provider balance if we have provider info
                        if (isset($response['provider_code'])) {
                            $provider = ApiProvider::where('code', $response['provider_code'])->first();
                            if ($provider) {
                                $providerOldBalance = $provider->balance;
                                $provider->decrement('balance', $finalPrice);
                                $providerNewBalance = $provider->fresh()->balance;

                                \Log::info('🟢🟢🟢 PROVIDER BALANCE UPDATED', [
                                    'provider_code' => $provider->code,
                                    'provider_name' => $provider->name,
                                    'old_balance' => $providerOldBalance,
                                    'amount_deducted' => $finalPrice,
                                    'new_balance' => $providerNewBalance
                                ]);
                            }
                        }

                        // Update transaction
                        $transaction->update([
                            'status' => 'success',
                            'api_provider_id' => $response['provider_id'] ?? null,
                            'provider_reference' => $response['provider_ref'] ?? null,
                            'details' => array_merge($transaction->details ?? [], [
                                'api_response' => $response,
                                'provider_ref' => $response['provider_ref'] ?? null,
                                'provider_code' => $response['provider_code'] ?? null,
                                'provider_name' => $response['provider_name'] ?? null,
                                'response_time' => $response['response_time'] ?? null,
                                'completed_at' => now()->toDateTimeString(),
                                'original_price' => $product->price,
                                'discount_applied' => $product->discount,
                                'final_price' => $finalPrice,
                                'retry_count' => $retryCount,
                                'success_at' => now()->toDateTimeString()
                            ])
                        ]);

                        \Log::info('🟢🟢🟢 TRANSACTION UPDATED TO SUCCESS', [
                            'transaction_id' => $transaction->id,
                            'status' => 'success',
                            'provider_reference' => $response['provider_ref'] ?? 'N/A'
                        ]);

                        // Log wallet history
                        try {
                            WalletHistoryService::log([
                                'user_id' => $user->id,
                                'reference' => $transaction->reference,
                                'type' => 'purchase',
                                'direction' => 'out',
                                'amount' => $finalPrice,
                                'balance_before' => $oldBalance,
                                'balance_after' => $newBalance,
                                'source' => $response['provider_code'] ?? 'external_vtu',
                                'description' => "Purchase: {$product->name} for " . ($details['phone'] ?? 'N/A'),
                                'status' => 'success',
                                'meta' => [
                                    'transaction_id' => $transaction->id,
                                    'product_id' => $product->id,
                                    'product_name' => $product->name,
                                    'phone' => $details['phone'] ?? 'N/A',
                                    'network' => $details['network'] ?? 'N/A',
                                    'provider' => $response['provider_code'] ?? null,
                                    'provider_name' => $response['provider_name'] ?? null,
                                    'response_time' => $response['response_time'] ?? null,
                                    'retry_count' => $retryCount,
                                ],
                            ]);

                            \Log::info('🟢🟢🟢 WALLET HISTORY LOGGED');
                        } catch (\Exception $e) {
                            \Log::warning('🟡🟡🟡 FAILED TO LOG WALLET HISTORY', [
                                'error' => $e->getMessage(),
                                'transaction_id' => $transaction->id
                            ]);
                            // Continue even if wallet history fails
                        }

                        // 🔴 LOG 8: FINAL SUCCESS
                        \Log::info('🟢🟢🟢 PURCHASE COMPLETED SUCCESSFULLY', [
                            'transaction_id' => $transaction->id,
                            'reference' => $transaction->reference,
                            'total_time' => time() - strtotime($transaction->created_at) . ' seconds',
                            'total_attempts' => $retryCount + 1,
                            'final_status' => 'success'
                        ]);

                        return $transaction;
                    } catch (\Exception $e) {
                        $lastError = $e;

                        \Log::error('🔴🔴🔴 PURCHASE ATTEMPT FAILED', [
                            'attempt' => $retryCount + 1,
                            'error' => $e->getMessage(),
                            'error_type' => get_class($e),
                            'file' => $e->getFile(),
                            'line' => $e->getLine(),
                            'product_id' => $product->id,
                            'transaction_id' => $transaction->id ?? 'N/A'
                        ]);

                        if ($retryCount >= $maxRetries) {
                            \Log::error('🔴🔴🔴 MAX RETRIES REACHED - BREAKING');
                            break;
                        }

                        $retryCount++;
                        $waitTime = min(5 * $retryCount, 15); // Exponential backoff

                        \Log::info('🟡🟡🟡 WAITING BEFORE NEXT RETRY', [
                            'wait_seconds' => $waitTime,
                            'next_attempt' => $retryCount,
                            'max_retries' => $maxRetries
                        ]);

                        sleep($waitTime);
                    }
                }

                // All retries failed
                \Log::error('🔴🔴🔴 ALL RETRIES FAILED', [
                    'total_attempts' => $retryCount,
                    'max_allowed_attempts' => $maxRetries,
                    'last_error' => $lastError ? $lastError->getMessage() : 'No error recorded',
                    'last_response' => $lastResponse,
                    'transaction_id' => $transaction->id ?? 'N/A'
                ]);

                if (isset($transaction)) {
                    $transaction->update([
                        'status' => 'failed',
                        'details' => array_merge($transaction->details ?? [], [
                            'failure_reason' => $lastError ? $lastError->getMessage() : 'Unknown error',
                            'failed_at' => now()->toDateTimeString(),
                            'retry_count' => $retryCount,
                            'last_error' => $lastError ? $lastError->getMessage() : null,
                            'last_response' => $lastResponse
                        ])
                    ]);

                    \Log::info('🟡🟡🟡 TRANSACTION UPDATED TO FAILED', [
                        'transaction_id' => $transaction->id,
                        'status' => 'failed',
                        'failure_reason' => $lastError ? $lastError->getMessage() : 'Unknown'
                    ]);
                }

                throw new \Exception('Purchase failed after ' . ($retryCount) . ' attempts' .
                    ($lastError ? ': ' . $lastError->getMessage() : ''));
            } catch (\Exception $e) {
                // 🔴 LOG 9: TRANSACTION BLOCK ERROR
                \Log::error('🔴🔴🔴 DATABASE TRANSACTION BLOCK FAILED', [
                    'error' => $e->getMessage(),
                    'error_type' => get_class($e),
                    'file' => $e->getFile(),
                    'line' => $e->getLine(),
                    'user_id' => $user->id,
                    'product_id' => $product->id,
                    'step' => 'transaction_block',
                    'timestamp' => now()->toDateTimeString()
                ]);

                throw $e; // Re-throw to trigger transaction rollback
            }
        }, 5); // 5 attempts for deadlock
    }

    /**
     * Emergency diagnostic method
     */
    public static function diagnose($transactionId)
    {
        \Log::info('🔍🔍🔍 PURCHASE DIAGNOSTICS STARTED', ['transaction_id' => $transactionId]);

        try {
            $transaction = Transaction::find($transactionId);

            if (!$transaction) {
                return ['error' => 'Transaction not found'];
            }

            $wallet = Wallet::where('user_id', $transaction->user_id)->first();
            $product = Product::find($transaction->product_id);

            return [
                'transaction' => [
                    'id' => $transaction->id,
                    'status' => $transaction->status,
                    'reference' => $transaction->reference,
                    'amount' => $transaction->amount,
                    'created_at' => $transaction->created_at,
                    'details' => $transaction->details
                ],
                'wallet' => $wallet ? [
                    'id' => $wallet->id,
                    'balance' => $wallet->balance,
                    'updated_at' => $wallet->updated_at
                ] : null,
                'product' => $product ? [
                    'id' => $product->id,
                    'name' => $product->name,
                    'type' => $product->type,
                    'price' => $product->price
                ] : null,
                'diagnostic_time' => now()->toDateTimeString()
            ];
        } catch (\Exception $e) {
            \Log::error('🔍🔍🔍 DIAGNOSTIC FAILED', ['error' => $e->getMessage()]);
            return ['error' => $e->getMessage()];
        }
    }
}
