<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class MonnifyService
{
    private $baseUrl;
    private $apiKey;
    private $secretKey;
    private $contractCode;
    private $walletAccountNumber;

    public function __construct()
    {
        $this->baseUrl = config('services.monnify.base_url', 'https://sandbox.monnify.com');
        $this->apiKey = config('services.monnify.api_key');
        $this->secretKey = config('services.monnify.secret_key');
        $this->contractCode = config('services.monnify.contract_code');
        $this->walletAccountNumber = config('services.monnify.wallet_account_number');
    }

    private function getAccessToken()
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Basic ' . base64_encode($this->apiKey . ':' . $this->secretKey),
            ])->post($this->baseUrl . '/api/v1/auth/login');

            if ($response->successful()) {
                return $response->json()['responseBody']['accessToken'];
            }

            Log::error('Monnify Auth Failed', $response->json());
            throw new \Exception('Failed to authenticate with Monnify');
        } catch (\Exception $e) {
            Log::error('Monnify Auth Error', ['error' => $e->getMessage()]);
            throw $e;
        }
    }

    public function initializeTransaction(array $data)
    {
        \Log::info('=== MONNIFY INITIALIZE TRANSACTION START ===');
        \Log::info('Data:', $data);

        try {
            // Get access token
            \Log::info('Getting access token...');
            $token = $this->getAccessToken();

            if (!$token) {
                throw new \Exception('Failed to get access token');
            }

            \Log::info('Access token obtained, length: ' . strlen($token));

            // Prepare payload - USE YOUR ACTUAL ROUTE NAMES
            $payload = [
                'amount' => (float) $data['amount'],
                'customerName' => $data['customer_name'],
                'customerEmail' => $data['customer_email'],
                'paymentReference' => $data['reference'],
                'paymentDescription' => $data['description'] ?? 'Wallet Funding',
                'currencyCode' => 'NGN',
                'contractCode' => $this->contractCode,
                // Use your actual route names from routes/web.php
                'redirectUrl' => route('monnify.callback.get'),
                'webhookUrl' => route('monnify.webhook'),  // Use your webhook route
                'paymentMethods' => ['CARD', 'ACCOUNT_TRANSFER'],
                'metadata' => [
                    'user_id' => $data['user_id'],
                    'wallet_id' => $data['wallet_id'],
                    'purpose' => 'wallet_funding'
                ]
            ];

            \Log::info('Sending to Monnify:', $payload);

            // Make API call
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
                'Content-Type' => 'application/json',
            ])->timeout(30)->post($this->baseUrl . '/api/v1/merchant/transactions/init-transaction', $payload);

            \Log::info('Monnify API Response Status: ' . $response->status());
            \Log::info('Monnify API Response Body: ' . $response->body());

            if ($response->successful()) {
                $responseData = $response->json();

                if (isset($responseData['responseBody'])) {
                    \Log::info('Transaction initialized successfully');
                    return $responseData['responseBody'];
                } else {
                    \Log::error('Monnify response missing responseBody', $responseData);
                    throw new \Exception('Invalid response from Monnify: missing responseBody');
                }
            } else {
                \Log::error('Monnify API error', [
                    'status' => $response->status(),
                    'body' => $response->body()
                ]);

                $errorBody = $response->json();
                $errorMessage = $errorBody['responseMessage'] ?? $errorBody['message'] ?? 'Unknown error';

                throw new \Exception('Monnify API Error: ' . $errorMessage . ' (Status: ' . $response->status() . ')');
            }
        } catch (\Exception $e) {
            \Log::error('Monnify initializeTransaction failed:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            throw $e;
        }
    }

    
    public function createReservedAccount(array $userData)
    {
        try {
            $token = $this->getAccessToken();

            $payload = [
                'accountReference' => 'USER_' . $userData['user_id'] . '_' . time(),
                'accountName' => $userData['account_name'],
                'currencyCode' => 'NGN',
                'contractCode' => $this->contractCode,
                'customerEmail' => $userData['email'],
                'customerName' => $userData['full_name'],
                'getAllAvailableBanks' => false,
                'preferredBanks' => ['035', '232', '058', '070'], // Wema, Zenith, GTB, Fidelity
                'restrictPaymentSource' => false,
                'nin' => $userData['nin'] ?? null,
                'bvn' => $userData['bvn'] ?? null,
                'incomeSplitConfig' => [
                    [
                        'subAccountCode' => $this->walletAccountNumber,
                        'splitPercentage' => 100,
                        'feePercentage' => 100,
                        'feeBearer' => true
                    ]
                ]
            ];

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
                'Content-Type' => 'application/json',
            ])->post($this->baseUrl . '/api/v2/bank-transfer/reserved-accounts', $payload);

            if ($response->successful()) {
                return $response->json()['responseBody'];
            }

            Log::error('Monnify Reserved Account Failed', $response->json());
            throw new \Exception('Failed to create reserved account');
        } catch (\Exception $e) {
            Log::error('Monnify Account Creation Error', ['error' => $e->getMessage()]);
            throw $e;
        }
    }

    public function verifyTransaction($transactionReference)
    {
        try {
            \Log::info('Verifying Monnify transaction: ' . $transactionReference);

            $token = $this->getAccessToken();

            // Use the correct endpoint
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
            ])->get($this->baseUrl . '/api/v2/transactions/' . $transactionReference);

            \Log::info('Verification response status: ' . $response->status());

            if ($response->successful()) {
                $data = $response->json();
                \Log::info('Verification successful', $data);

                return [
                    'success' => true,
                    'data' => $data,
                    'paymentStatus' => $data['responseBody']['paymentStatus'] ?? null,
                    'amountPaid' => $data['responseBody']['amountPaid'] ?? null,
                ];
            } else {
                \Log::error('Monnify verification failed', $response->json());
                return [
                    'success' => false,
                    'error' => 'Verification failed',
                    'status' => $response->status(),
                    'response' => $response->json(),
                ];
            }
        } catch (\Exception $e) {
            \Log::error('Monnify Verify Error', ['error' => $e->getMessage()]);
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function initiateTransfer(array $transferData)
    {
        try {
            $token = $this->getAccessToken();

            $payload = [
                'amount' => $transferData['amount'],
                'reference' => $transferData['reference'],
                'narration' => $transferData['narration'] ?? 'Withdrawal',
                'destinationBankCode' => $transferData['bank_code'],
                'destinationAccountNumber' => $transferData['account_number'],
                'destinationAccountName' => $transferData['account_name'],
                'currency' => 'NGN',
                'sourceAccountNumber' => $this->walletAccountNumber,
            ];

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
                'Content-Type' => 'application/json',
            ])->post($this->baseUrl . '/api/v2/disbursements/single', $payload);

            if ($response->successful()) {
                return $response->json()['responseBody'];
            }

            Log::error('Monnify Transfer Failed', $response->json());
            throw new \Exception('Failed to initiate transfer');
        } catch (\Exception $e) {
            Log::error('Monnify Transfer Error', ['error' => $e->getMessage()]);
            throw $e;
        }
    }

    public function getBanks()
    {
        try {
            $token = $this->getAccessToken();

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
            ])->get($this->baseUrl . '/api/v1/banks');

            if ($response->successful()) {
                return $response->json()['responseBody'];
            }

            return [];
        } catch (\Exception $e) {
            Log::error('Monnify Banks Error', ['error' => $e->getMessage()]);
            return [];
        }
    }

    public function validateAccount($accountNumber, $bankCode)
    {
        try {
            $token = $this->getAccessToken();

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
            ])->get($this->baseUrl . "/api/v1/disbursements/account/validate?accountNumber={$accountNumber}&bankCode={$bankCode}");

            if ($response->successful()) {
                return $response->json()['responseBody'];
            }

            return null;
        } catch (\Exception $e) {
            Log::error('Monnify Validate Error', ['error' => $e->getMessage()]);
            return null;
        }
    }
}
