<?php

namespace App\Services;

use App\Models\Product;
use App\Models\ApiProvider;
use App\Models\Transaction;
use App\Services\VtuProviders\ProviderFactory;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class ExternalVtuService
{
    protected static $cacheTime = 30; // seconds
    protected static $maxRetries = 3;

    /**
     * Main entry point for sending VTU requests
     */
    public static function send(Product $product, array $details, string $reference, Transaction $transaction = null)
    {
        // Log the request
        Log::info('VTU Purchase Request', [
            'product_id' => $product->id,
            'product_type' => $product->type,
            'reference' => $reference,
            'details' => $details
        ]);

        // Route based on product type
        return match ($product->type) {
            'data' => self::buyData($product, $details, $reference, $transaction),
            'airtime' => self::buyAirtime($product, $details, $reference, $transaction),
            'electricity' => self::payElectricity($product, $details, $reference, $transaction),
            'cable' => self::subscribeCable($product, $details, $reference, $transaction),
            'education' => self::payEducation($product, $details, $reference, $transaction),
            'betting' => self::fundBetting($product, $details, $reference, $transaction),
            default => [
                'success' => false,
                'message' => 'Invalid product type: ' . $product->type,
                'should_retry' => false,
            ]
        };
    }

    /**
     * Handle data bundle purchase
     */
    private static function buyData(Product $product, array $details, string $reference, Transaction $transaction = null)
    {
        return self::executePurchase($product, $details, $reference, $transaction, 'data');
    }

    /**
     * Handle airtime purchase
     */
    private static function buyAirtime(Product $product, array $details, string $reference, Transaction $transaction = null)
    {
        return self::executePurchase($product, $details, $reference, $transaction, 'airtime');
    }

    /**
     * Handle electricity bill payment
     */
    private static function payElectricity(Product $product, array $details, string $reference, Transaction $transaction = null)
    {
        return self::executePurchase($product, $details, $reference, $transaction, 'electricity');
    }

    /**
     * Handle cable TV subscription
     */
    private static function subscribeCable(Product $product, array $details, string $reference, Transaction $transaction = null)
    {
        return self::executePurchase($product, $details, $reference, $transaction, 'cable');
    }

    /**
     * Handle education payment (WAEC, JAMB, etc.)
     */
    private static function payEducation(Product $product, array $details, string $reference, Transaction $transaction = null)
    {
        return self::executePurchase($product, $details, $reference, $transaction, 'education');
    }

    /**
     * Handle betting wallet funding
     */
    private static function fundBetting(Product $product, array $details, string $reference, Transaction $transaction = null)
    {
        return self::executePurchase($product, $details, $reference, $transaction, 'betting');
    }

    /**
     * Generic purchase execution with retry logic
     */
    private static function executePurchase(Product $product, array $details, string $reference, Transaction $transaction, string $serviceType)
    {
        $retryCount = 0;
        $lastError = null;
        $lastResponse = null;

        while ($retryCount <= self::$maxRetries) {
            try {
                // Select best provider for this attempt
                $provider = self::selectProvider($product, $details, $serviceType, $retryCount);
                
                if (!$provider) {
                    return [
                        'success' => false,
                        'message' => 'No provider available for ' . $serviceType,
                        'should_retry' => $retryCount < self::$maxRetries,
                        'retry_after' => 30,
                        'attempt' => $retryCount + 1,
                    ];
                }

                // Try to increment load
                if (!$provider->incrementLoad()) {
                    Log::warning('Provider at capacity', [
                        'provider' => $provider->code,
                        'current_load' => $provider->current_load,
                        'limit' => $provider->concurrent_limit
                    ]);
                    
                    if ($retryCount < self::$maxRetries) {
                        $retryCount++;
                        sleep(2);
                        continue;
                    }
                    
                    return [
                        'success' => false,
                        'message' => 'Provider at capacity: ' . $provider->name,
                        'should_retry' => true,
                        'retry_after' => 10,
                        'attempt' => $retryCount + 1,
                    ];
                }

                // Create provider service instance
                $providerService = ProviderFactory::make(
                    $provider->code,
                    $provider,
                    $product,
                    $details,
                    $transaction
                );

                // Record start time
                $startTime = microtime(true);

                // Execute purchase
                $result = match($serviceType) {
                    'data' => $providerService->purchase(),
                    'airtime' => method_exists($providerService, 'purchaseAirtime') 
                                ? $providerService->purchaseAirtime() 
                                : $providerService->purchase(),
                    'electricity' => method_exists($providerService, 'purchaseElectricity')
                                   ? $providerService->purchaseElectricity()
                                   : self::dummyElectricityResponse($product, $details),
                    'cable' => method_exists($providerService, 'purchaseCable')
                              ? $providerService->purchaseCable()
                              : self::dummyCableResponse($product, $details),
                    default => [
                        'success' => false,
                        'message' => 'Service type not implemented: ' . $serviceType
                    ]
                };

                // Calculate response time
                $responseTime = (int)((microtime(true) - $startTime) * 1000);

                // Record transaction result
                $provider->recordTransaction($result['success'] ?? false, $responseTime);

                // Add provider info to result
                $result['provider_code'] = $provider->code;
                $result['provider_id'] = $provider->id;
                $result['provider_name'] = $provider->name;
                $result['response_time'] = $responseTime;
                $result['attempt'] = $retryCount + 1;

                // Log successful transaction
                if ($result['success']) {
                    Log::info('VTU Purchase Success', [
                        'provider' => $provider->code,
                        'product' => $product->plan,
                        'reference' => $reference,
                        'response_time' => $responseTime,
                        'attempt' => $retryCount + 1
                    ]);
                } else {
                    Log::warning('VTU Purchase Failed', [
                        'provider' => $provider->code,
                        'product' => $product->plan,
                        'reference' => $reference,
                        'error' => $result['message'] ?? 'Unknown error',
                        'attempt' => $retryCount + 1
                    ]);
                }

                // Check if we should retry on failure
                if (!$result['success'] && ($result['should_retry'] ?? false) && $retryCount < self::$maxRetries) {
                    $retryCount++;
                    $waitTime = $result['retry_after'] ?? 5;
                    sleep($waitTime);
                    continue;
                }

                return $result;

            } catch (\Exception $e) {
                $lastError = $e;
                
                // Record failure if we have a provider
                if (isset($provider)) {
                    $provider->recordTransaction(false);
                }

                Log::error('VTU Purchase Exception', [
                    'error' => $e->getMessage(),
                    'file' => $e->getFile(),
                    'line' => $e->getLine(),
                    'attempt' => $retryCount + 1
                ]);

                if ($retryCount < self::$maxRetries) {
                    $retryCount++;
                    sleep(min(5 * $retryCount, 15)); // Exponential backoff
                    continue;
                }

                break;
            } finally {
                // Always decrement load if provider was set
                if (isset($provider)) {
                    $provider->decrementLoad();
                }
            }
        }

        // All retries failed
        return [
            'success' => false,
            'message' => 'Purchase failed after ' . self::$maxRetries . ' attempts' . 
                        ($lastError ? ': ' . $lastError->getMessage() : ''),
            'should_retry' => false,
            'last_error' => $lastError ? $lastError->getMessage() : null,
            'attempts' => $retryCount + 1,
        ];
    }

    /**
     * Smart provider selection with caching
     */
    private static function selectProvider(Product $product, array $details, string $serviceType, int $attempt = 0)
    {
        // TEMPORARY WORKAROUND: Use manual filtering
        $provider = ApiProvider::where('is_active', true)
            ->where('balance', '>=', $product->price)
            ->where('code', 'clubkonnect')
            ->first();

        if ($provider) {
            // Manual service check
            $services = $provider->supported_services;
            if (is_string($services)) $services = json_decode($services, true);

            // Manual network check
            $networks = $provider->supported_networks;
            if (is_string($networks)) $networks = json_decode($networks, true);

            if (
                is_array($services) && in_array($serviceType, $services) &&
                is_array($networks) && in_array($details['network'] ?? '', $networks)
            ) {

                Log::info('Using ClubKonnect (manual filter)', [
                    'provider' => $provider->code,
                    'balance' => $provider->balance,
                    'has_service' => in_array($serviceType, $services),
                    'has_network' => in_array($details['network'] ?? '', $networks)
                ]);

                return $provider;
            }
        }
    }

    /**
     * Calculate provider score (higher is better)
     */
    private static function calculateProviderScore(ApiProvider $provider, Product $product, int $attempt = 0)
    {
        $score = 0;
        
        // Success rate (0-40 points)
        $score += ($provider->success_rate * 0.4);
        
        // Response time (0-30 points) - faster is better
        $maxTime = 5000; // 5 seconds max
        $responseTime = max(1, $provider->avg_response_time);
        $responseScore = max(0, 30 * (1 - ($responseTime / $maxTime)));
        $score += $responseScore;
        
        // Load factor (0-20 points) - less load is better
        $loadFactor = 1 - ($provider->current_load / max(1, $provider->concurrent_limit));
        $score += ($loadFactor * 20);
        
        // Priority (0-10 points) - higher priority is better
        $score += (10 / max(1, $provider->priority));
        
        // Balance ratio (0-10 points) - more balance relative to price is better
        $balanceRatio = $provider->balance / max(1, $product->price);
        $balanceScore = min(10, $balanceRatio * 2);
        $score += $balanceScore;
        
        // Circuit status penalty for retries
        if ($attempt > 0 && $provider->circuit_status === 'half_open') {
            $score *= 0.8; // 20% penalty for half-open circuits
        } elseif ($provider->circuit_status === 'open') {
            $score *= 0.5; // 50% penalty for open circuits
        }
        
        // Recent failures penalty
        if ($provider->consecutive_failures > 0) {
            $failurePenalty = min(0.5, $provider->consecutive_failures * 0.1);
            $score *= (1 - $failurePenalty);
        }
        
        return round($score, 2);
    }

    /**
     * Dummy responses for testing/unimplemented services
     */
    private static function dummyElectricityResponse(Product $product, array $details)
    {
        return [
            'success' => true,
            'message' => 'Electricity payment successful',
            'token' => '1234-5678-9012-3456',
            'units' => rand(10, 100),
            'amount_paid' => $product->price,
            'provider' => 'dummy_electricity',
        ];
    }

    private static function dummyCableResponse(Product $product, array $details)
    {
        return [
            'success' => true,
            'message' => 'Cable subscription successful',
            'smartcard' => $details['smartcard'] ?? '1234567890',
            'package' => $product->plan,
            'expiry_date' => now()->addMonth()->format('Y-m-d'),
            'provider' => 'dummy_cable',
        ];
    }

    /**
     * Test provider connection (for admin panel)
     */
    public static function testProvider($providerCode)
    {
        try {
            $provider = ApiProvider::where('code', $providerCode)->first();
            
            if (!$provider) {
                return [
                    'success' => false,
                    'message' => 'Provider not found'
                ];
            }
            
            // Create dummy product and transaction for testing
            $product = new Product([
                'type' => 'data',
                'plan' => 'Test Plan',
                'price' => 100,
                'plan_code' => 'TEST-001'
            ]);
            
            $transaction = new Transaction([
                'reference' => 'TEST-' . time()
            ]);

            $providerService = ProviderFactory::make(
                $provider->code,
                $provider,
                $product,
                ['phone' => '08012345678', 'network' => 'MTN'],
                $transaction
            );
            
            // Try balance check if method exists
            if (method_exists($providerService, 'testConnection')) {
                return $providerService->testConnection();
            }
            
            // Fallback: try to get balance
            return [
                'success' => true,
                'message' => 'Provider is active',
                'balance' => $provider->balance,
                'status' => $provider->is_active ? 'active' : 'inactive',
                'success_rate' => $provider->success_rate,
                'response_time' => $provider->avg_response_time . 'ms'
            ];
            
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => 'Test failed: ' . $e->getMessage(),
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Get provider statistics (for dashboard)
     */
    public static function getProviderStats()
    {
        $providers = ApiProvider::all();
        
        return $providers->map(function ($provider) {
            return [
                'id' => $provider->id,
                'name' => $provider->name,
                'code' => $provider->code,
                'balance' => $provider->balance,
                'is_active' => $provider->is_active,
                'success_rate' => $provider->success_rate,
                'response_time' => $provider->avg_response_time,
                'current_load' => $provider->current_load,
                'concurrent_limit' => $provider->concurrent_limit,
                'circuit_status' => $provider->circuit_status,
                'success_count' => $provider->success_count,
                'failure_count' => $provider->failure_count,
                'load_percentage' => $provider->concurrent_limit > 0 
                    ? round(($provider->current_load / $provider->concurrent_limit) * 100, 2)
                    : 0,
                'health_score' => self::calculateProviderScore($provider, new Product(['price' => 100])),
            ];
        })->sortByDesc('health_score')->values();
    }

    /**
     * Reset provider circuit breaker (for admin)
     */
    public static function resetProviderCircuit($providerId)
    {
        $provider = ApiProvider::find($providerId);
        
        if (!$provider) {
            return [
                'success' => false,
                'message' => 'Provider not found'
            ];
        }
        
        $provider->update([
            'circuit_status' => 'closed',
            'circuit_opened_at' => null,
            'consecutive_failures' => 0
        ]);
        
        return [
            'success' => true,
            'message' => 'Circuit breaker reset for ' . $provider->name
        ];
    }
}