<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;

class ClubKonnectService
{
    protected $userId;
    protected $apiKey;
    protected $baseUrl;
    protected $testMode;
    protected $testSuccessRate;

    public function __construct()
    {
        $this->userId = config('services.clubkonnect.user_id');
        $this->apiKey = config('services.clubkonnect.api_key');
        $this->baseUrl = config('services.clubkonnect.base_url', 'https://www.nellobytesystems.com');
        $this->testMode = config('services.clubkonnect.test_mode', false);
        $this->testSuccessRate = config('services.clubkonnect.test_success_rate', 100);
    }

    /**
     * ==============================================
     * CABLE TV METHODS (NEW)
     * ==============================================
     */

    /**
     * Verify Cable TV Smartcard
     */
    public function verifyCableTV(string $provider, string $smartcardNumber): array
    {
        // TEST MODE: Return mock response
        if ($this->testMode) {
            Log::info('ClubKonnect TEST MODE: Simulating cable smartcard verification', [
                'provider' => $provider,
                'smartcard' => substr($smartcardNumber, 0, 6) . '...',
            ]);

            return $this->mockCableVerificationResponse($provider, $smartcardNumber);
        }

        // REAL MODE: Call actual API
        return $this->realCableVerification($provider, $smartcardNumber);
    }

    /**
     * Mock Cable Verification Response
     */
    protected function mockCableVerificationResponse(string $provider, string $smartcardNumber): array
    {
        $shouldSucceed = rand(1, 100) <= $this->testSuccessRate;

        if ($shouldSucceed) {
            $customerNames = [
                'JOHN DOE',
                'JANE SMITH',
                'CHUKWU EMELI',
                'AMINA BALOGUN',
                'SANI YUSUF',
            ];

            return [
                'success' => true,
                'customer_name' => $customerNames[array_rand($customerNames)],
                'message' => 'Test mode: Smartcard verified',
                'is_test' => true,
            ];
        } else {
            return [
                'success' => false,
                'customer_name' => 'INVALID_SMARTCARDNO',
                'message' => 'Test mode: Invalid smartcard',
                'is_test' => true,
            ];
        }
    }

    /**
     * Real Cable Verification
     */
    protected function realCableVerification(string $provider, string $smartcardNumber): array
    {
        $url = "{$this->baseUrl}/APIVerifyCableTVV1.0.asp";

        $params = [
            'UserID' => $this->userId,
            'APIKey' => $this->apiKey,
            'CableTV' => $provider,
            'SmartCardNo' => $smartcardNumber,
        ];

        try {
            Log::info('Verifying cable smartcard:', [
                'url' => $url,
                'params' => [
                    'provider' => $provider,
                    'smartcard' => substr($smartcardNumber, 0, 6) . '...'
                ]
            ]);

            $response = Http::timeout(30)->retry(2, 1000)->get($url, $params);

            if ($response->successful()) {
                $data = $response->json();

                Log::info('Cable verification API response:', $data);

                // Check if verification was successful
                if (isset($data['customer_name']) && $data['customer_name'] !== 'INVALID_SMARTCARDNO') {
                    return [
                        'success' => true,
                        'customer_name' => $data['customer_name'],
                        'message' => 'Smartcard verified successfully',
                        'data' => $data
                    ];
                } else {
                    return [
                        'success' => false,
                        'customer_name' => $data['customer_name'] ?? 'INVALID_SMARTCARDNO',
                        'message' => 'Invalid smartcard number',
                        'data' => $data
                    ];
                }
            }

            Log::error('Cable verification failed. Status: ' . $response->status());
            return [
                'success' => false,
                'customer_name' => 'VERIFICATION_FAILED',
                'message' => 'Verification service unavailable',
            ];
        } catch (\Exception $e) {
            Log::error('Cable verification error: ' . $e->getMessage());
            return [
                'success' => false,
                'customer_name' => 'NETWORK_ERROR',
                'message' => 'Verification service error',
            ];
        }
    }

    /**
     * Purchase Cable TV Subscription
     */
    public function buyCableTV(
        string $provider,
        string $packageCode,
        string $smartcardNumber,
        string $phoneNumber,
        string $requestId = null,
        string $callbackUrl = null
    ): array {
        // TEST MODE: Return mock response
        if ($this->testMode) {
            Log::info('ClubKonnect TEST MODE: Simulating cable TV purchase', [
                'provider' => $provider,
                'package' => $packageCode,
                'smartcard' => substr($smartcardNumber, 0, 6) . '...',
                'phone' => substr($phoneNumber, 0, 6) . '...',
            ]);

            return $this->mockCablePurchaseResponse($provider, $packageCode, $smartcardNumber, $phoneNumber, $requestId);
        }

        // REAL MODE: Call actual API
        return $this->realCablePurchase($provider, $packageCode, $smartcardNumber, $phoneNumber, $requestId, $callbackUrl);
    }

    /**
     * Mock Cable Purchase Response
     */
    protected function mockCablePurchaseResponse(
        string $provider,
        string $packageCode,
        string $smartcardNumber,
        string $phoneNumber,
        string $requestId = null
    ): array {
        $requestId = $requestId ?? 'CABLE-TEST-' . time();
        $shouldSucceed = rand(1, 100) <= $this->testSuccessRate;

        if ($shouldSucceed) {
            $orderId = 'CABLE-' . time();
            return [
                'success' => true,
                'order_id' => $orderId,
                'orderid' => $orderId,
                'status' => 'ORDER_RECEIVED',
                'status_code' => '100',
                'statuscode' => '100',
                'request_id' => $requestId,
                'message' => 'Test mode: Cable subscription order received',
                'data' => [
                    'orderid' => $orderId,
                    'order_id' => $orderId,
                    'statuscode' => '100',
                    'status' => 'ORDER_RECEIVED',
                    'orderstatus' => 'ORDER_RECEIVED',
                    'orderremark' => 'Dear Customer, You have successfully subscribed to ' . strtoupper($provider),
                    'productname' => 'Cable TV Subscription (TEST)',
                    'amount' => '10000.00',
                    'cabletv' => $provider,
                    'package' => $packageCode,
                    'smartcardno' => $smartcardNumber,
                    'phoneno' => $phoneNumber,
                    'is_test' => true,
                ]
            ];
        } else {
            return [
                'success' => false,
                'order_id' => null,
                'status' => 'TEST_FAILED',
                'status_code' => '400',
                'request_id' => $requestId,
                'message' => 'Test mode: Cable subscription failed',
                'data' => ['is_test' => true]
            ];
        }
    }

    /**
     * Real Cable Purchase
     */
    protected function realCablePurchase(
        string $provider,
        string $packageCode,
        string $smartcardNumber,
        string $phoneNumber,
        string $requestId = null,
        string $callbackUrl = null
    ): array {
        $requestId = $requestId ?? uniqid('cable_', true);
        $url = "{$this->baseUrl}/APICableTVV1.asp";

        $params = [
            'UserID' => $this->userId,
            'APIKey' => $this->apiKey,
            'CableTV' => $provider,
            'Package' => $packageCode,
            'SmartCardNo' => $smartcardNumber,
            'PhoneNo' => $phoneNumber,
            'RequestID' => $requestId,
        ];

        // Add callback URL if provided
        if ($callbackUrl) {
            $params['CallBackURL'] = $callbackUrl;
        } else {
            // Default callback URL
            $params['CallBackURL'] = route('cable.callback');
        }

        try {
            Log::info('Purchasing cable subscription:', [
                'url' => $url,
                'params' => [
                    'provider' => $provider,
                    'package' => $packageCode,
                    'smartcard' => substr($smartcardNumber, 0, 6) . '...',
                    'phone' => substr($phoneNumber, 0, 6) . '...',
                    'request_id' => $requestId
                ]
            ]);

            $response = Http::timeout(45)->retry(2, 1000)->get($url, $params);

            if ($response->successful()) {
                $data = $response->json();

                Log::info('Cable purchase API response:', $data);

                // Standardized response format
                $responseData = [
                    'success' => true,
                    'order_id' => $data['orderid'] ?? null,
                    'orderid' => $data['orderid'] ?? null,
                    'status' => $data['status'] ?? null,
                    'orderstatus' => $data['orderstatus'] ?? ($data['status'] ?? null),
                    'status_code' => $data['statuscode'] ?? null,
                    'statuscode' => $data['statuscode'] ?? null,
                    'request_id' => $requestId,
                    'message' => $data['orderremark'] ?? ($data['status'] ?? 'Transaction processed'),
                    'data' => $data
                ];

                // Check for errors
                if (isset($data['status']) && in_array($data['status'], ['MISSING_CREDENTIALS', 'INVALID_CREDENTIALS'])) {
                    $responseData['success'] = false;
                    $responseData['message'] = 'Invalid ClubKonnect credentials.';
                }

                return $responseData;
            }

            Log::error('Cable purchase failed. Status: ' . $response->status());
            return [
                'success' => false,
                'order_id' => null,
                'status' => 'API_ERROR',
                'status_code' => $response->status(),
                'request_id' => $requestId,
                'message' => 'API request failed with status: ' . $response->status()
            ];
        } catch (\Exception $e) {
            Log::error('Cable purchase error: ' . $e->getMessage());
            return [
                'success' => false,
                'order_id' => null,
                'status' => 'NETWORK_ERROR',
                'status_code' => 500,
                'request_id' => $requestId,
                'message' => 'Network error: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Fetch Available Cable Packages (with caching)
     */
    public function getCablePackages($provider = null)
    {
        $cacheKey = 'clubkonnect_cable_packages_v2';

        // Cache packages for 2 hours to reduce API calls
        $allPackages = Cache::remember($cacheKey, 7200, function () {
            // TEST MODE: Return mock packages
            if ($this->testMode) {
                Log::info('ClubKonnect TEST MODE: Returning mock cable packages');
                return $this->mockCablePackages();
            }

            // REAL MODE: Fetch from API
            return $this->realGetCablePackages();
        });

        // Filter by provider if specified
        if ($provider && isset($allPackages[$provider])) {
            return [$provider => $allPackages[$provider]];
        }

        return $allPackages;
    }

    /**
     * Mock Cable Packages for Testing
     */
    protected function mockCablePackages()
    {
        return [
            'dstv' => [
                'name' => 'DStv',
                'code' => 'dstv',
                'display_name' => 'DStv',
                'packages' => [
                    [
                        'package_code' => 'dstv-padi',
                        'name' => 'DStv Padi',
                        'description' => 'DStv Padi Monthly Subscription',
                        'price' => 4400.00,
                        'validity' => '1 Month',
                    ],
                    [
                        'package_code' => 'dstv-confam',
                        'name' => 'DStv Confam',
                        'description' => 'DStv Confam Monthly Subscription',
                        'price' => 11000.00,
                        'validity' => '1 Month',
                    ],
                    [
                        'package_code' => 'dstv79',
                        'name' => 'DStv Compact',
                        'description' => 'DStv Compact Monthly Subscription',
                        'price' => 19000.00,
                        'validity' => '1 Month',
                    ],
                ]
            ],
            'gotv' => [
                'name' => 'GOtv',
                'code' => 'gotv',
                'display_name' => 'GOtv',
                'packages' => [
                    [
                        'package_code' => 'gotv-max',
                        'name' => 'GOtv Max',
                        'description' => 'GOtv Max Monthly Subscription',
                        'price' => 8500.00,
                        'validity' => '1 Month',
                    ],
                    [
                        'package_code' => 'gotv-jolli',
                        'name' => 'GOtv Jolli',
                        'description' => 'GOtv Jolli Monthly Subscription',
                        'price' => 5800.00,
                        'validity' => '1 Month',
                    ],
                    [
                        'package_code' => 'gotv-jinja',
                        'name' => 'GOtv Jinja',
                        'description' => 'GOtv Jinja Monthly Subscription',
                        'price' => 3900.00,
                        'validity' => '1 Month',
                    ],
                ]
            ],
            'startimes' => [
                'name' => 'StarTimes',
                'code' => 'startimes',
                'display_name' => 'StarTimes',
                'packages' => [
                    [
                        'package_code' => 'nova',
                        'name' => 'StarTimes Nova (Dish)',
                        'description' => 'Nova (Dish) - Monthly',
                        'price' => 2100.00,
                        'validity' => '1 Month',
                    ],
                    [
                        'package_code' => 'basic',
                        'name' => 'StarTimes Basic (Antenna)',
                        'description' => 'Basic (Antenna) - Monthly',
                        'price' => 4000.00,
                        'validity' => '1 Month',
                    ],
                ]
            ],
            'showmax' => [
                'name' => 'Showmax',
                'code' => 'showmax',
                'display_name' => 'Showmax',
                'packages' => [
                    [
                        'package_code' => 'full',
                        'name' => 'Showmax Full',
                        'description' => 'Showmax Full Monthly Subscription',
                        'price' => 4500.00,
                        'validity' => '1 Month',
                    ],
                    [
                        'package_code' => 'mobile_only',
                        'name' => 'Showmax Mobile Only',
                        'description' => 'Showmax Mobile Only Monthly Subscription',
                        'price' => 2000.00,
                        'validity' => '1 Month',
                    ],
                ]
            ],
        ];
    }

    /**
     * Real Cable Packages from API
     */
    protected function realGetCablePackages()
    {
        $url = "{$this->baseUrl}/APICableTVPackagesV2.asp";
        $params = ['UserID' => $this->userId];

        try {
            Log::info('Fetching cable packages from ClubKonnect API');

            $response = Http::timeout(30)->get($url, $params);

            if ($response->successful()) {
                $data = $response->json();
                return $this->formatCablePackagesForFrontend($data);
            }

            Log::error('Failed to fetch cable packages. Status: ' . $response->status());
            return [];
        } catch (\Exception $e) {
            Log::error('ClubKonnect Error fetching cable packages: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Format cable packages for frontend use
     */
    private function formatCablePackagesForFrontend($apiData)
    {
        if (!is_array($apiData)) {
            return [];
        }

        $formattedPackages = [];
        $providers = ['dstv', 'gotv', 'startimes', 'showmax'];

        $displayNames = [
            'dstv' => 'DStv',
            'gotv' => 'GOtv',
            'startimes' => 'StarTimes',
            'showmax' => 'Showmax',
        ];

        foreach ($providers as $provider) {
            if (isset($apiData[$provider]) && is_array($apiData[$provider])) {
                $formattedPackages[$provider] = [
                    'name' => $displayNames[$provider] ?? ucfirst($provider),
                    'code' => $provider,
                    'display_name' => $displayNames[$provider] ?? ucfirst($provider),
                    'packages' => []
                ];

                foreach ($apiData[$provider] as $packageString) {
                    $package = $this->parseCablePackageString($packageString, $provider);
                    if ($package) {
                        $formattedPackages[$provider]['packages'][] = $package;
                    }
                }

                // Sort packages by price
                usort($formattedPackages[$provider]['packages'], function ($a, $b) {
                    return $a['price'] <=> $b['price'];
                });
            }
        }

        return $formattedPackages;
    }

    /**
     * Parse individual cable package string
     */
    private function parseCablePackageString($packageString, $provider)
    {
        // Format: "dstv-padi — DStv Padi N4,400"
        $parts = explode('—', $packageString);
        if (count($parts) !== 2) {
            return null;
        }

        $packageCode = trim($parts[0]);
        $descriptionPart = trim($parts[1]);

        // Extract name and price
        preg_match('/^(.*?)\s+N([\d,]+(?:\.\d{2})?)$/', $descriptionPart, $matches);

        if (empty($matches)) {
            return null;
        }

        $name = trim($matches[1]);
        $price = (float) str_replace(',', '', $matches[2]);

        return [
            'package_code' => $packageCode,
            'name' => $name,
            'description' => $name . ' Monthly Subscription',
            'price' => $price,
            'validity' => '1 Month',
            'raw' => $packageString,
        ];
    }

    /**
     * Sync cable packages to database (for admin use)
     */
    public function syncCablePackagesToDatabase()
    {
        try {
            $apiPackages = $this->getCablePackages();
            $syncedCount = 0;

            foreach ($apiPackages as $providerCode => $provider) {
                foreach ($provider['packages'] as $package) {
                    \App\Models\Product::updateOrCreate(
                        [
                            'provider_plan_id' => $package['package_code'],
                            'network_code' => $providerCode,
                            'type' => 'cable'
                        ],
                        [
                            'name' => $package['name'],
                            'description' => $package['description'],
                            'price' => $package['price'],
                            'network_name' => $provider['name'],
                            'validity' => $package['validity'],
                            'is_active' => true,
                            'is_available' => true,
                            'provider' => 'clubkonnect',
                            'metadata' => json_encode($package),
                        ]
                    );
                    $syncedCount++;
                }
            }

            return [
                'success' => true,
                'message' => "Synced $syncedCount cable packages",
                'count' => $syncedCount
            ];
        } catch (\Exception $e) {
            Log::error('Failed to sync cable packages: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to sync: ' . $e->getMessage()
            ];
        }
    }

    /**
     * ==============================================
     * EXISTING METHODS (UNCHANGED)
     * ==============================================
     */

    /**
     * Purchase Data Bundle (with test mode support)
     */
    public function buyDataBundle($network, $dataPlanId, $phone, $requestId = null)
    {
        // TEST MODE: Return mock response
        if ($this->testMode) {
            Log::info('ClubKonnect TEST MODE: Simulating data bundle purchase', [
                'network' => $network,
                'data_plan_id' => $dataPlanId,
                'phone' => $phone,
            ]);

            return $this->mockDataBundleResponse($network, $dataPlanId, $phone, $requestId);
        }

        // REAL MODE: Call actual API
        return $this->realDataBundlePurchase($network, $dataPlanId, $phone, $requestId);
    }

    /**
     * Mock Data Bundle Response for Testing
     */
    protected function mockDataBundleResponse($network, $dataPlanId, $phone, $requestId = null)
    {
        $requestId = $requestId ?? 'TEST-DATA-' . time();
        $shouldSucceed = rand(1, 100) <= $this->testSuccessRate;

        if ($shouldSucceed) {
            $orderId = 'TEST-DATA-' . time();
            return [
                'success' => true,
                'order_id' => $orderId,
                'orderid' => $orderId,
                'status' => 'ORDER_RECEIVED',
                'status_code' => '100',
                'statuscode' => '100',
                'request_id' => $requestId,
                'message' => 'Test mode: Data bundle order received',
                'data' => [
                    'orderid' => $orderId,
                    'order_id' => $orderId,
                    'statuscode' => '100',
                    'status' => 'ORDER_RECEIVED',
                    'orderstatus' => 'ORDER_RECEIVED',
                    'orderremark' => 'Dear Customer, You have successfully shared Data to ' . $phone,
                    'productname' => 'Data Bundle (TEST)',
                    'amount' => '100.00',
                    'mobilenetwork' => $this->getNetworkName($network),
                    'mobilenumber' => $phone,
                    'is_test' => true,
                ]
            ];
        } else {
            return [
                'success' => false,
                'order_id' => null,
                'status' => 'TEST_FAILED',
                'status_code' => '400',
                'request_id' => $requestId,
                'message' => 'Test mode: Data purchase failed',
                'data' => ['is_test' => true]
            ];
        }
    }

    /**
     * Real Data Bundle Purchase
     */
    protected function realDataBundlePurchase($network, $dataPlanId, $phone, $requestId = null)
    {
        $requestId = $requestId ?? uniqid('data_', true);
        $url = "{$this->baseUrl}/APIDatabundleV1.asp";

        $params = [
            'UserID' => $this->userId,
            'APIKey' => $this->apiKey,
            'MobileNetwork' => $network,
            'DataPlan' => $dataPlanId,
            'MobileNumber' => $phone,
            'RequestID' => $requestId,
            'CallBackURL' => route('clubkonnect.callback'),
        ];

        try {
            $response = Http::timeout(30)->get($url, $params);

            if ($response->successful()) {
                $data = $response->json();

                // Standardized response format
                $responseData = [
                    'success' => true,
                    'order_id' => $data['orderid'] ?? null,
                    'orderid' => $data['orderid'] ?? null,
                    'status' => $data['status'] ?? null,
                    'orderstatus' => $data['orderstatus'] ?? ($data['status'] ?? null),
                    'status_code' => $data['statuscode'] ?? null,
                    'statuscode' => $data['statuscode'] ?? null,
                    'request_id' => $requestId,
                    'message' => $data['orderremark'] ?? ($data['status'] ?? 'Transaction processed'),
                    'data' => $data
                ];

                // Check for errors
                if (isset($data['status']) && in_array($data['status'], ['MISSING_CREDENTIALS', 'INVALID_CREDENTIALS'])) {
                    $responseData['success'] = false;
                    $responseData['message'] = 'Invalid ClubKonnect credentials.';
                }

                Log::info('ClubKonnect Data API Response', $responseData);
                return $responseData;
            }

            return [
                'success' => false,
                'order_id' => null,
                'status' => 'API_ERROR',
                'status_code' => $response->status(),
                'request_id' => $requestId,
                'message' => 'API request failed with status: ' . $response->status()
            ];
        } catch (\Exception $e) {
            Log::error('ClubKonnect Data API Error: ' . $e->getMessage());
            return [
                'success' => false,
                'order_id' => null,
                'status' => 'NETWORK_ERROR',
                'status_code' => 500,
                'request_id' => $requestId,
                'message' => 'Network error: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Fetch Available Data Plans (with caching)
     */
    public function getDataPlans()
    {
        // Cache plans for 1 hour to reduce API calls
        return Cache::remember('clubkonnect_data_plans_v2', 3600, function () {
            // TEST MODE: Return mock plans
            if ($this->testMode) {
                Log::info('ClubKonnect TEST MODE: Returning mock data plans');
                return $this->mockDataPlans();
            }

            // REAL MODE: Fetch from API
            return $this->realGetDataPlans();
        });
    }

    /**
     * Mock Data Plans for Testing
     */
    protected function mockDataPlans()
    {
        $plans = [
            '01' => [ // MTN
                'name' => 'MTN',
                'code' => '01',
                'plans' => [
                    [
                        'id' => '500.0',
                        'description' => '500 MB - 7 days (SME)',
                        'price' => 404.00,
                        'raw' => '500.0 — 500 MB - 7 days (SME) @ ₦404.00',
                        'validity' => '7 days',
                        'data_size' => '500 MB',
                    ],
                    [
                        'id' => '1000.0',
                        'description' => '1 GB - 7 days (SME)',
                        'price' => 567.00,
                        'raw' => '1000.0 — 1 GB - 7 days (SME) @ ₦567.00',
                        'validity' => '7 days',
                        'data_size' => '1 GB',
                    ],
                    [
                        'id' => '2000.0',
                        'description' => '2 GB - 7 days (SME)',
                        'price' => 1134.00,
                        'raw' => '2000.0 — 2 GB - 7 days (SME) @ ₦1,134.00',
                        'validity' => '7 days',
                        'data_size' => '2 GB',
                    ],
                ]
            ],
            '04' => [ // Airtel
                'name' => 'Airtel',
                'code' => '04',
                'plans' => [
                    [
                        'id' => '499.91',
                        'description' => '1GB - 1 day (Awoof Data)',
                        'price' => 484.91,
                        'raw' => '499.91 — 1GB - 1 day (Awoof Data) @ ₦484.91',
                        'validity' => '1 day',
                        'data_size' => '1 GB',
                    ],
                    [
                        'id' => '599.91',
                        'description' => '1.5GB - 2 days (Awoof Data)',
                        'price' => 581.91,
                        'raw' => '599.91 — 1.5GB - 2 days (Awoof Data) @ ₦581.91',
                        'validity' => '2 days',
                        'data_size' => '1.5 GB',
                    ],
                ]
            ],
            '02' => [ // Glo
                'name' => 'Glo',
                'code' => '02',
                'plans' => [
                    [
                        'id' => '200',
                        'description' => '200 MB - 14 days (SME)',
                        'price' => 94.00,
                        'raw' => '200 — 200 MB - 14 days (SME) @ ₦94.00',
                        'validity' => '14 days',
                        'data_size' => '200 MB',
                    ],
                    [
                        'id' => '500',
                        'description' => '500 MB - 7 days (SME)',
                        'price' => 235.00,
                        'raw' => '500 — 500 MB - 7 days (SME) @ ₦235.00',
                        'validity' => '7 days',
                        'data_size' => '500 MB',
                    ],
                ]
            ],
            '03' => [ // 9mobile
                'name' => '9mobile',
                'code' => '03',
                'plans' => [
                    [
                        'id' => '50',
                        'description' => '50 MB - 30 days (SME)',
                        'price' => 23.00,
                        'raw' => '50 — 50 MB - 30 days (SME) @ ₦23.00',
                        'validity' => '30 days',
                        'data_size' => '50 MB',
                    ],
                    [
                        'id' => '100',
                        'description' => '100 MB - 30 days (SME)',
                        'price' => 46.00,
                        'raw' => '100 — 100 MB - 30 days (SME) @ ₦46.00',
                        'validity' => '30 days',
                        'data_size' => '100 MB',
                    ],
                ]
            ],
        ];

        return $plans;
    }

    /**
     * Real Data Plans from API
     */
    protected function realGetDataPlans()
    {
        $url = "{$this->baseUrl}/APIDatabundlePlansV2.asp";
        $params = ['UserID' => $this->userId];

        try {
            $response = Http::timeout(30)->get($url, $params);

            if ($response->successful()) {
                $data = $response->json();
                return $this->formatDataPlansForFrontend($data);
            }

            Log::error('Failed to fetch data plans. Status: ' . $response->status());
            return [];
        } catch (\Exception $e) {
            Log::error('ClubKonnect Error fetching data plans: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Format data plans for frontend use
     */
    private function formatDataPlansForFrontend($apiData)
    {
        if (!is_array($apiData)) {
            return [];
        }

        $formattedPlans = [];
        $networkMap = [
            '01' => 'MTN',
            '02' => 'Glo',
            '03' => '9mobile',
            '04' => 'Airtel',
        ];

        foreach ($networkMap as $networkCode => $networkName) {
            if (isset($apiData[$networkCode]) && is_array($apiData[$networkCode])) {
                $formattedPlans[$networkCode] = [
                    'name' => $networkName,
                    'code' => $networkCode,
                    'plans' => []
                ];

                foreach ($apiData[$networkCode] as $planString) {
                    $plan = $this->parseDataPlanString($planString);
                    if ($plan) {
                        $formattedPlans[$networkCode]['plans'][] = $plan;
                    }
                }

                // Sort plans by price
                usort($formattedPlans[$networkCode]['plans'], function ($a, $b) {
                    return $a['price'] <=> $b['price'];
                });
            }
        }

        return $formattedPlans;
    }

    /**
     * Parse individual plan string (data)
     */
    private function parseDataPlanString($planString)
    {
        // Format: "500.0 — 500 MB - 7 days (SME) @ ₦404.00"
        $parts = explode('@', $planString);
        if (count($parts) !== 2) {
            return null;
        }

        $descriptionPart = trim($parts[0]);
        $pricePart = trim($parts[1]);

        // Extract plan ID (first part before "—")
        $idParts = explode('—', $descriptionPart);
        if (count($idParts) < 2) {
            return null;
        }

        $planId = trim($idParts[0]);
        $description = trim($idParts[1]);

        // Extract price
        $price = (float) preg_replace('/[^0-9.]/', '', $pricePart);

        // Extract validity
        $validity = 'N/A';
        if (preg_match('/(\d+\s*(?:day|days|month|months|year|years))/i', $description, $matches)) {
            $validity = $matches[1];
        }

        // Extract data size
        $dataSize = 'N/A';
        if (preg_match('/(\d+(?:\.\d+)?)\s*(GB|MB)/i', $description, $matches)) {
            $dataSize = $matches[1] . ' ' . strtoupper($matches[2]);
        }

        return [
            'id' => $planId,
            'description' => $description,
            'price' => $price,
            'raw' => $planString,
            'validity' => $validity,
            'data_size' => $dataSize,
        ];
    }

    /**
     * Purchase Airtime (with test mode support)
     */
    public function buyAirtime($network, $amount, $phone, $requestId = null)
    {
        // TEST MODE: Return mock response
        if ($this->testMode) {
            Log::info('ClubKonnect TEST MODE: Simulating airtime purchase', [
                'network' => $network,
                'amount' => $amount,
                'phone' => $phone,
            ]);

            return $this->mockAirtimeResponse($network, $amount, $phone, $requestId);
        }

        // REAL MODE: Call actual API
        return $this->realAirtimePurchase($network, $amount, $phone, $requestId);
    }

    /**
     * Mock Airtime Response for Testing
     */
    protected function mockAirtimeResponse($network, $amount, $phone, $requestId = null)
    {
        $requestId = $requestId ?? 'TEST-' . time();
        $shouldSucceed = rand(1, 100) <= $this->testSuccessRate;

        if ($shouldSucceed) {
            $orderId = 'TEST-' . time();
            return [
                'success' => true,
                'order_id' => $orderId,
                'orderid' => $orderId,
                'status' => 'ORDER_RECEIVED',
                'status_code' => '100',
                'request_id' => $requestId,
                'message' => 'Test mode: Order received',
                'data' => [
                    'orderid' => $orderId,
                    'order_id' => $orderId,
                    'statuscode' => '100',
                    'status' => 'ORDER_RECEIVED',
                    'productname' => $amount . ' Credit (TEST)',
                    'amount' => $amount * 0.965,
                    'mobilenetwork' => $this->getNetworkName($network),
                    'mobilenumber' => $phone,
                    'walletbalance' => '999999.99',
                    'is_test' => true,
                ]
            ];
        } else {
            return [
                'success' => false,
                'order_id' => null,
                'status' => 'TEST_FAILED',
                'status_code' => '400',
                'request_id' => $requestId,
                'message' => 'Test mode: Transaction failed',
                'data' => ['is_test' => true]
            ];
        }
    }

    /**
     * Real Airtime Purchase
     */
    protected function realAirtimePurchase($network, $amount, $phone, $requestId = null)
    {
        $requestId = $requestId ?? uniqid('airtime_', true);
        $url = "{$this->baseUrl}/APIAirtimeV1.asp";

        $params = [
            'UserID' => $this->userId,
            'APIKey' => $this->apiKey,
            'MobileNetwork' => $network,
            'Amount' => $amount,
            'MobileNumber' => $phone,
            'RequestID' => $requestId,
            'CallBackURL' => route('clubkonnect.callback'),
        ];

        try {
            $response = Http::timeout(30)->get($url, $params);

            if ($response->successful()) {
                $data = $response->json();

                $responseData = [
                    'success' => true,
                    'order_id' => $data['orderid'] ?? null,
                    'orderid' => $data['orderid'] ?? null,
                    'status' => $data['status'] ?? null,
                    'status_code' => $data['statuscode'] ?? null,
                    'statuscode' => $data['statuscode'] ?? null,
                    'request_id' => $requestId,
                    'message' => $data['status'] ?? 'Transaction processed',
                    'data' => $data
                ];

                if (isset($data['status']) && in_array($data['status'], ['MISSING_CREDENTIALS', 'INVALID_CREDENTIALS'])) {
                    $responseData['success'] = false;
                    $responseData['message'] = 'Invalid ClubKonnect credentials.';
                }

                Log::info('ClubKonnect API Response', $responseData);
                return $responseData;
            }

            return [
                'success' => false,
                'order_id' => null,
                'status' => 'API_ERROR',
                'status_code' => $response->status(),
                'request_id' => $requestId,
                'message' => 'API request failed with status: ' . $response->status()
            ];
        } catch (\Exception $e) {
            Log::error('ClubKonnect API Error: ' . $e->getMessage());
            return [
                'success' => false,
                'order_id' => null,
                'status' => 'NETWORK_ERROR',
                'status_code' => 500,
                'request_id' => $requestId,
                'message' => 'Network error: ' . $e->getMessage()
            ];
        }
    }


    /**
     * Get network name from code
     */
    public function getNetworkName($code)
    {
        $networks = [
            '01' => 'MTN',
            '02' => 'GLO',
            '04' => 'Airtel',
            '03' => '9mobile',
        ];

        return $networks[$code] ?? 'Unknown';
    }

    /**
     * Check if in test mode
     */
    public function isTestMode()
    {
        return $this->testMode;
    }

    /**
     * Query Transaction Status
     */
    public function queryTransaction($orderId = null, $requestId = null)
    {
        $url = "{$this->baseUrl}/APIQueryV1.asp";

        $params = [
            'UserID' => $this->userId,
            'APIKey' => $this->apiKey,
        ];

        if ($orderId) {
            $params['OrderID'] = $orderId;
        } elseif ($requestId) {
            $params['RequestID'] = $requestId;
        } else {
            return ['success' => false, 'message' => 'Either OrderID or RequestID required'];
        }

        try {
            $response = Http::timeout(30)->get($url, $params);

            if ($response->successful()) {
                $data = $response->json();

                return [
                    'success' => true,
                    'order_id' => $data['orderid'] ?? null,
                    'request_id' => $data['requestid'] ?? null,
                    'status' => $data['status'] ?? null,
                    'status_code' => $data['statuscode'] ?? null,
                    'remark' => $data['remark'] ?? null,
                    'date' => $data['date'] ?? null,
                    'data' => $data,
                ];
            }

            return [
                'success' => false,
                'message' => 'API request failed',
                'status_code' => $response->status(),
            ];
        } catch (\Exception $e) {
            Log::error('ClubKonnect Query Error: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Network error: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Get Networks with Discounts
     */
    public function getNetworks()
    {
        $url = "{$this->baseUrl}/APIAirtimeDiscountV2.asp";
        $params = ['UserID' => $this->userId];

        $response = Http::get($url, $params);
        return $response->json();
    }

    /**
     * Check vendor account balance using the correct ClubKonnect endpoint
     * This is the main check that will block transactions if insufficient
     */
    public function checkBalance()
    {
        // TEST MODE: Return mock balance
        if ($this->testMode) {
            Log::info('ClubKonnect TEST MODE: Simulating balance check');
            return 100000.00; // Mock high balance for testing
        }

        // REAL MODE: Call ClubKonnect API with correct endpoint
        $url = "{$this->baseUrl}/APIWalletBalanceV1.asp";

        $params = [
            'UserID' => $this->userId,
            'APIKey' => $this->apiKey,
        ];

        try {
            Log::info('Checking ClubKonnect balance with params:', $params);

            $response = Http::timeout(30)->get($url, $params);

            Log::info('ClubKonnect balance check response status:', [
                'status' => $response->status(),
                'successful' => $response->successful()
            ]);

            if ($response->successful()) {
                $data = $response->json();

                // Log the raw response for debugging
                Log::info('ClubKonnect Balance Check Response:', $data);

                // Check for error responses first
                if (isset($data['status'])) {
                    $status = strtoupper($data['status']);
                    if (in_array($status, ['INVALID_CREDENTIALS', 'MISSING_CREDENTIALS', 'MISSING_USERID', 'MISSING_APIKEY'])) {
                        throw new \Exception('ClubKonnect API Error: ' . $status);
                    }
                }

                // Extract balance from response
                if (isset($data['balance'])) {
                    $balance = (float) $data['balance'];
                    Log::info("ClubKonnect balance retrieved: {$balance}");
                    return $balance;
                } else {
                    throw new \Exception('Balance field not found in response: ' . json_encode($data));
                }
            } else {
                $status = $response->status();
                $body = $response->body();
                Log::error("ClubKonnect balance check failed. Status: {$status}, Body: {$body}");
                throw new \Exception("API request failed with status: {$status}");
            }
        } catch (\Exception $e) {
            Log::error('ClubKonnect Balance Check Error: ' . $e->getMessage());
            throw new \Exception('Service temporarily unavailable. Unable to verify provider balance.');
        }
    }

    /**
     * Check if vendor has sufficient balance - STRICT MODE
     * Will throw exception if insufficient balance
     */
    public function ensureSufficientBalance($requiredAmount)
    {
        try {
            // Get ClubKonnect balance
            $balance = $this->checkBalance();

            Log::info('ClubKonnect balance check:', [
                'balance' => $balance,
                'required_amount' => $requiredAmount,
                'has_sufficient' => $balance >= $requiredAmount
            ]);

            if ($balance < $requiredAmount) {
                // Log detailed error internally
                Log::error('❌ ClubKonnect INSUFFICIENT BALANCE', [
                    'balance' => $balance,
                    'required' => $requiredAmount,
                    'deficit' => $requiredAmount - $balance
                ]);

                throw new \Exception(
                    'ClubKonnect insufficient balance: ' .
                        '₦' . number_format($balance, 2) . ' available, ' .
                        '₦' . number_format($requiredAmount, 2) . ' required'
                );
            }

            return true;
        } catch (\Exception $e) {
            Log::error('Vendor balance check failed: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Sync data plans to database (for admin use)
     */
    public function syncDataPlansToDatabase()
    {
        try {
            $apiPlans = $this->getDataPlans();
            $syncedCount = 0;

            foreach ($apiPlans as $networkCode => $network) {
                foreach ($network['plans'] as $plan) {
                    \App\Models\Product::updateOrCreate(
                        [
                            'provider_plan_id' => $plan['id'],
                            'network_code' => $networkCode,
                            'type' => 'data'
                        ],
                        [
                            'name' => $plan['description'],
                            'description' => $plan['description'],
                            'price' => $plan['price'],
                            'network_name' => $network['name'],
                            'validity' => $plan['validity'],
                            'data_size' => $plan['data_size'],
                            'is_active' => true,
                            'is_available' => true,
                            'provider' => 'clubkonnect',
                            'metadata' => json_encode($plan),
                        ]
                    );
                    $syncedCount++;
                }
            }

            return [
                'success' => true,
                'message' => "Synced $syncedCount data plans",
                'count' => $syncedCount
            ];
        } catch (\Exception $e) {
            Log::error('Failed to sync data plans: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to sync: ' . $e->getMessage()
            ];
        }
    }
}
