<?php

namespace App\Providers;

use Illuminate\Support\Facades\Vite;
use Illuminate\Support\ServiceProvider;
use App\Services\MonnifyService;
use App\Models\Setting;
use App\Observers\SettingObserver;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     */
    public function register(): void
    {
        // Register MonnifyService as a singleton
        $this->app->singleton(MonnifyService::class, function ($app) {
            return new MonnifyService();
        });
    }

    /**
     * Bootstrap any application services.
     */
    public function boot(): void
    {
        // Vite configuration
        Vite::prefetch(concurrency: 3);

        // Register Setting Observer
        Setting::observe(SettingObserver::class);

        // Override config from database
        $this->overrideConfigFromDatabase();
    }

    protected function overrideConfigFromDatabase(): void
    {
        // Don't run during migrations
        if ($this->app->runningInConsole() && !$this->app->runningUnitTests()) {
            return;
        }

        try {
            if (!\Schema::hasTable('settings')) {
                return;
            }

            // Load all settings from database
            $settings = Setting::all();

            foreach ($settings as $setting) {
                $value = $setting->is_encrypted
                    ? $setting->getDecryptedValue()
                    : $setting->value;

                // Cast value
                $value = $this->castValue($value, $setting->type);

                // Set in config
                config([$setting->key => $value]);
            }
        } catch (\Exception $e) {
            // Log but don't crash
            \Log::error('Failed to load settings: ' . $e->getMessage());
        }
    }

    protected function castValue($value, $type)
    {
        if ($value === null) return null;

        switch ($type) {
            case 'boolean':
            case 'bool':
                return filter_var($value, FILTER_VALIDATE_BOOLEAN);
            case 'integer':
            case 'int':
                return (int) $value;
            case 'float':
            case 'double':
                return (float) $value;
            case 'array':
            case 'json':
                return json_decode($value, true);
            default:
                return (string) $value;
        }
    }
}
