<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class WalletHistory extends Model
{
  protected $fillable = [
    'user_id',
    'reference',
    'type',
    'direction',
    'amount',
    'balance_before',
    'balance_after',
    'source',
    'description',
    'status',
    'meta'
  ];

  protected $casts = [
    'amount' => 'decimal:2',
    'balance_before' => 'decimal:2',
    'balance_after' => 'decimal:2',
    'meta' => 'array'
  ];

  public function user(): BelongsTo
  {
    return $this->belongsTo(User::class);
  }

  public function wallet(): BelongsTo
  {
    return $this->belongsTo(Wallet::class, 'user_id', 'user_id');
  }

  // Scopes for filtering
  public function scopeCredits($query)
  {
    return $query->where('direction', 'in');
  }

  public function scopeDebits($query)
  {
    return $query->where('direction', 'out');
  }

  public function scopeHolds($query)
  {
    return $query->where('direction', 'hold');
  }

  public function scopeReleases($query)
  {
    return $query->where('direction', 'release');
  }

  public function scopeCompleted($query)
  {
    return $query->where('status', 'completed');
  }

  public function scopeFailed($query)
  {
    return $query->where('status', 'failed');
  }

  public function scopePending($query)
  {
    return $query->where('status', 'pending');
  }

  // Helper methods
  public function getFormattedAmountAttribute(): string
  {
    $prefix = $this->direction === 'in' ? '+' : '-';
    return $prefix . '₦' . number_format($this->amount, 2);
  }

  public function getFormattedBalanceBeforeAttribute(): string
  {
    return '₦' . number_format($this->balance_before, 2);
  }

  public function getFormattedBalanceAfterAttribute(): string
  {
    return '₦' . number_format($this->balance_after, 2);
  }

  public function getReadableTypeAttribute(): string
  {
    $types = [
      'credit' => 'Credit',
      'debit' => 'Debit',
      'hold' => 'Hold',
      'release' => 'Release'
    ];

    return $types[$this->type] ?? ucfirst($this->type);
  }

  public function getReadableDirectionAttribute(): string
  {
    $directions = [
      'in' => 'Incoming',
      'out' => 'Outgoing',
      'hold' => 'Hold',
      'release' => 'Release'
    ];

    return $directions[$this->direction] ?? ucfirst($this->direction);
  }

  public function getIconAttribute(): string
  {
    switch ($this->direction) {
      case 'in':
        return '⬇️';
      case 'out':
        return '⬆️';
      case 'hold':
        return '⏸️';
      case 'release':
        return '▶️';
      default:
        return '📝';
    }
  }

  public function getColorAttribute(): string
  {
    switch ($this->direction) {
      case 'in':
        return 'text-green-600';
      case 'out':
        return 'text-red-600';
      case 'hold':
        return 'text-yellow-600';
      case 'release':
        return 'text-blue-600';
      default:
        return 'text-gray-600';
    }
  }
}
