<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Str;

class Wallet extends Model
{
    protected $fillable = [
        'user_id',
        'balance',
        'available_balance',
        'ledger_balance',
        'currency',
        'status',
    ];

    protected $casts = [
        'balance' => 'decimal:2',
        'available_balance' => 'decimal:2',
        'ledger_balance' => 'decimal:2',
    ];

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    // Check if wallet has sufficient funds
    public function hasSufficientFunds(float $amount): bool
    {
        return $this->available_balance >= $amount;
    }

    // Credit wallet with history logging
    public function credit(float $amount, string $narration = '', array $meta = []): bool
    {
        $previousBalance = $this->balance;
        $previousAvailable = $this->available_balance;

        $this->available_balance += $amount;
        $this->balance = $this->available_balance + $this->ledger_balance;

        if ($this->save()) {
            // Log to wallet history
            WalletHistory::create([
                'user_id' => $this->user_id,
                'reference' => 'CR-' . time() . '-' . strtoupper(Str::random(6)),
                'type' => 'credit',
                'direction' => 'in',
                'amount' => $amount,
                'balance_before' => $previousBalance,
                'balance_after' => $this->balance,
                'source' => 'wallet_credit',
                'description' => $narration ?: 'Wallet credit',
                'status' => 'completed',
                'meta' => array_merge($meta, [
                    'previous_available' => $previousAvailable,
                    'new_available' => $this->available_balance,
                ]),
            ]);

            return true;
        }

        return false;
    }

    // Debit wallet with history logging
    public function debit(float $amount, string $narration = '', array $meta = []): bool
    {
        if (!$this->hasSufficientFunds($amount)) {
            return false;
        }

        $previousBalance = $this->balance;
        $previousAvailable = $this->available_balance;

        $this->available_balance -= $amount;
        $this->balance = $this->available_balance + $this->ledger_balance;

        if ($this->save()) {
            // Log to wallet history
            WalletHistory::create([
                'user_id' => $this->user_id,
                'reference' => 'DR-' . time() . '-' . strtoupper(Str::random(6)),
                'type' => 'debit',
                'direction' => 'out',
                'amount' => $amount,
                'balance_before' => $previousBalance,
                'balance_after' => $this->balance,
                'source' => 'wallet_debit',
                'description' => $narration ?: 'Wallet debit',
                'status' => 'completed',
                'meta' => array_merge($meta, [
                    'previous_available' => $previousAvailable,
                    'new_available' => $this->available_balance,
                ]),
            ]);

            return true;
        }

        return false;
    }

    // Hold funds (move from available to ledger)
    public function holdFunds(float $amount, string $narration = '', array $meta = []): bool
    {
        if (!$this->hasSufficientFunds($amount)) {
            return false;
        }

        $previousBalance = $this->balance;
        $previousAvailable = $this->available_balance;
        $previousLedger = $this->ledger_balance;

        $this->available_balance -= $amount;
        $this->ledger_balance += $amount;
        $this->balance = $this->available_balance + $this->ledger_balance;

        if ($this->save()) {
            WalletHistory::create([
                'user_id' => $this->user_id,
                'reference' => 'HLD-' . time() . '-' . strtoupper(Str::random(6)),
                'type' => 'hold',
                'direction' => 'hold',
                'amount' => $amount,
                'balance_before' => $previousBalance,
                'balance_after' => $this->balance,
                'source' => 'funds_hold',
                'description' => $narration ?: 'Funds held for transaction',
                'status' => 'completed',
                'meta' => array_merge($meta, [
                    'previous_available' => $previousAvailable,
                    'new_available' => $this->available_balance,
                    'previous_ledger' => $previousLedger,
                    'new_ledger' => $this->ledger_balance,
                ]),
            ]);

            return true;
        }

        return false;
    }

    // Release funds (move from ledger to available)
    public function releaseFunds(float $amount, string $narration = '', array $meta = []): bool
    {
        if ($this->ledger_balance < $amount) {
            return false;
        }

        $previousBalance = $this->balance;
        $previousAvailable = $this->available_balance;
        $previousLedger = $this->ledger_balance;

        $this->ledger_balance -= $amount;
        $this->available_balance += $amount;
        $this->balance = $this->available_balance + $this->ledger_balance;

        if ($this->save()) {
            WalletHistory::create([
                'user_id' => $this->user_id,
                'reference' => 'RLS-' . time() . '-' . strtoupper(Str::random(6)),
                'type' => 'release',
                'direction' => 'release',
                'amount' => $amount,
                'balance_before' => $previousBalance,
                'balance_after' => $this->balance,
                'source' => 'funds_release',
                'description' => $narration ?: 'Funds released',
                'status' => 'completed',
                'meta' => array_merge($meta, [
                    'previous_available' => $previousAvailable,
                    'new_available' => $this->available_balance,
                    'previous_ledger' => $previousLedger,
                    'new_ledger' => $this->ledger_balance,
                ]),
            ]);

            return true;
        }

        return false;
    }

    // Get formatted balance
    public function formattedBalance(): string
    {
        return '₦' . number_format($this->balance, 2);
    }

    // Get formatted available balance
    public function formattedAvailableBalance(): string
    {
        return '₦' . number_format($this->available_balance, 2);
    }
}
