<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class VirtualAccount extends Model
{
    protected $fillable = [
        'user_id',
        'account_reference',
        'account_number',
        'bank_name',
        'bank_code',
        'account_name',
        'bvn',
        'currency',
        'account_details',
        'status',
        'bvn_verified_at',
    ];

    protected $casts = [
        'account_details' => 'array',
        'bvn_verified_at' => 'datetime',
    ];

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    // Check if BVN is verified
    public function isBvnVerified(): bool
    {
        return !is_null($this->bvn_verified_at);
    }

    // Get formatted account details for display
    public function getFormattedDetails(): array
    {
        return [
            'account_number' => $this->account_number,
            'bank_name' => $this->bank_name,
            'account_name' => $this->account_name,
            'bvn_verified' => $this->isBvnVerified(),
            'status' => $this->status,
            'created_at' => $this->created_at->format('M d, Y'),
        ];
    }

    // Get masked account number for display
    public function getMaskedAccountNumber(): string
    {
        $length = strlen($this->account_number);
        if ($length <= 4) {
            return $this->account_number;
        }

        $visible = 4;
        $masked = str_repeat('*', $length - $visible);
        return substr($this->account_number, 0, $length - $visible) . $masked;
    }

    // Activate account
    public function activate(): bool
    {
        return $this->update(['status' => 'active']);
    }

    // Deactivate account
    public function deactivate(): bool
    {
        return $this->update(['status' => 'inactive']);
    }

    // Mark BVN as verified
    public function markBvnAsVerified(): bool
    {
        return $this->update(['bvn_verified_at' => now()]);
    }

    // Get bank logo URL (you can add bank logos later)
    public function getBankLogo(): string
    {
        $logos = [
            '058' => 'gtb.png',
            '033' => 'uba.png',
            '035' => 'wema.png',
            '232' => 'zenith.png',
            '214' => 'firstbank.png',
            '057' => 'stanbic.png',
            '076' => 'polaris.png',
            '011' => 'firstbank.png',
        ];

        $logo = $logos[$this->bank_code] ?? 'default-bank.png';
        return asset('images/banks/' . $logo);
    }
}
