<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class UserMessage extends Model
{
    protected $fillable = [
        'user_id',
        'sender_id', // Add this if you need it
        'title',
        'message',
        'type',
        'is_read',
        'actions',
        'expires_at'
    ];

    protected $casts = [
        'is_read' => 'boolean',
        'actions' => 'array',
        'expires_at' => 'datetime'
    ];

    // For personal messages (direct relationship)
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    // For sender (if you have admins sending messages)
    public function sender()
    {
        return $this->belongsTo(User::class, 'sender_id');
    }

    // For system announcements (many-to-many)
    public function assignedUsers()
    {
        return $this->belongsToMany(User::class, 'user_message_assignments')
            ->withPivot('is_read')
            ->withTimestamps();
    }

    public function scopeUnread($query)
    {
        return $query->where('is_read', false);
    }

    public function scopeActive($query)
    {
        return $query->where(function ($q) {
            $q->whereNull('expires_at')
                ->orWhere('expires_at', '>', now());
        });
    }

    // Get messages for a specific user
    public function scopeForUser($query, $userId)
    {
        return $query->where(function ($q) use ($userId) {
            $q->where('user_id', $userId)
                ->orWhereHas('assignedUsers', function ($q) use ($userId) {
                    $q->where('user_id', $userId);
                });
        });
    }
}
