<?php

namespace App\Models;

use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class User extends Authenticatable
{
    use Notifiable;

    protected $fillable = [
        'name',
        'email',
        'phone',
        'nin',
        'bvn',
        'bvn_verified_at',
        'kyc_verified_at',
        'is_verified',
        'password',
        'kyc_status',
        'last_login_at',
        'last_login_ip',
        'avatar',
    ];

    protected $hidden = [
        'password',
        'remember_token',
        'bvn',
        'nin',
    ];

    protected $appends = [
        'avatar_url',
    ];

    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'is_verified' => 'boolean',
            'bvn_verified_at' => 'datetime',
            'kyc_verified_at' => 'datetime',
            'last_login_at' => 'datetime',
        ];
    }

    // ========== RELATIONSHIPS ==========

    public function wallet(): HasOne
    {
        return $this->hasOne(Wallet::class);
    }

    public function virtualAccount(): HasOne
    {
        return $this->hasOne(VirtualAccount::class);
    }

    public function walletHistories(): HasMany
    {
        return $this->hasMany(WalletHistory::class);
    }

    public function transactions(): HasMany
    {
        return $this->hasMany(Transaction::class);
    }

    // ========== BUSINESS LOGIC METHODS ==========

    public function getWallet()
    {
        if (!$this->wallet) {
            $this->wallet()->create([
                'user_id' => $this->id,
                'balance' => 0,
                'available_balance' => 0,
                'ledger_balance' => 0,
                'currency' => 'NGN',
                'status' => 'active',
            ]);
        }
        return $this->wallet;
    }

    public function hasVirtualAccount(): bool
    {
        return $this->virtualAccount()->exists();
    }

    public function isVerified(): bool
    {
        return $this->kyc_status === 'verified' || $this->is_verified === true;
    }

    public function hasSubmittedKyc(): bool
    {
        return !empty($this->bvn) || !empty($this->nin);
    }

    public function isBvnVerified(): bool
    {
        return !is_null($this->bvn_verified_at);
    }

    public function isKycVerified(): bool
    {
        return !is_null($this->kyc_verified_at);
    }

    public function getKycStatus(): array
    {
        if ($this->isKycVerified()) {
            return ['status' => 'verified', 'label' => 'Verified', 'color' => 'success'];
        }

        if ($this->hasSubmittedKyc()) {
            return ['status' => 'pending', 'label' => 'Under Review', 'color' => 'warning'];
        }

        return ['status' => 'not_submitted', 'label' => 'Not Submitted', 'color' => 'danger'];
    }

    public function canTransact(float $amount): bool
    {
        if (!$this->isVerified()) {
            return false;
        }

        $wallet = $this->getWallet();
        return $wallet->hasSufficientFunds($amount);
    }

    public function getFormattedBalance(): string
    {
        $wallet = $this->getWallet();
        return '₦' . number_format($wallet->balance, 2);
    }

    public function getAvailableBalance(): float
    {
        return $this->getWallet()->available_balance;
    }

    public function markBvnAsVerified(): bool
    {
        return $this->update([
            'bvn_verified_at' => now(),
            'kyc_status' => 'verified',
            'is_verified' => true,
            'kyc_verified_at' => now(),
        ]);
    }

    public function monnifyAccount(): HasOne
    {
        return $this->hasOne(VirtualAccount::class);
    }

    public function approveKyc(): bool
    {
        return $this->update([
            'kyc_status' => 'verified',
            'is_verified' => true,
            'kyc_verified_at' => now(),
        ]);
    }

    public function rejectKyc(string $reason = null): bool
    {
        return $this->update([
            'kyc_status' => 'rejected',
            'is_verified' => false,
            'kyc_verified_at' => null,
        ]);
    }

    public function submitBvn(string $bvn): bool
    {
        return $this->update([
            'bvn' => $bvn,
            'kyc_status' => 'pending',
        ]);
    }

    public function submitNin(string $nin): bool
    {
        return $this->update([
            'nin' => $nin,
            'kyc_status' => 'pending',
        ]);
    }

    public function getAccountName(): string
    {
        $name = preg_replace('/[^A-Za-z\s]/', '', $this->name);
        return substr(trim($name), 0, 100);
    }

    public function getBalanceAttribute()
    {
        $wallet = $this->getWallet();
        return $wallet->balance;
    }

    public function decrementBalance($amount)
    {
        $wallet = $this->getWallet();
        return $wallet->decrement('balance', $amount);
    }

    public function incrementBalance($amount)
    {
        $wallet = $this->getWallet();
        return $wallet->increment('balance', $amount);
    }

    public function getMaskedBvn(): ?string
    {
        if (empty($this->bvn)) {
            return null;
        }

        return '*****' . substr($this->bvn, -4);
    }

    public function getMaskedNin(): ?string
    {
        if (empty($this->nin)) {
            return null;
        }

        return '*****' . substr($this->nin, -4);
    }

    public function logWalletHistory(array $data): WalletHistory
    {
        $wallet = $this->getWallet();

        $historyData = array_merge([
            'user_id' => $this->id,
            'reference' => 'TRX-' . time() . '-' . strtoupper(Str::random(6)),
            'balance_before' => $wallet->balance,
            'balance_after' => $wallet->balance,
            'status' => 'completed',
        ], $data);

        return $this->walletHistories()->create($historyData);
    }

    public function getRecentWalletHistory(int $limit = 10)
    {
        return $this->walletHistories()
            ->latest()
            ->limit($limit)
            ->get();
    }

    public function updateLastLogin(): void
    {
        $this->update([
            'last_login_at' => now(),
            'last_login_ip' => request()->ip(),
        ]);
    }

    public function needsKyc(): bool
    {
        return !$this->isVerified() && !$this->hasSubmittedKyc();
    }

    public function hasPendingKyc(): bool
    {
        return $this->kyc_status === 'pending' && !$this->isVerified();
    }

    public function getVerificationLevel(): string
    {
        if ($this->isKycVerified() && $this->isBvnVerified()) {
            return 'full';
        }

        if ($this->hasSubmittedKyc()) {
            return 'pending';
        }

        return 'none';
    }

    // ========== AVATAR METHODS ==========

    public function getAvatarUrlAttribute(): string
    {
        if ($this->avatar) {
            return Storage::url($this->avatar);
        }

        // Generate a default avatar based on name
        $name = urlencode($this->name);
        return "https://ui-avatars.com/api/?name={$name}&color=7F9CF5&background=EBF4FF";
    }

    public function deleteAvatar(): bool
    {
        if ($this->avatar && Storage::disk('public')->exists($this->avatar)) {
            Storage::disk('public')->delete($this->avatar);
        }

        return $this->update(['avatar' => null]);
    }

    public function updateAvatar(string $path): bool
    {
        // Delete old avatar if exists
        $this->deleteAvatar();

        return $this->update(['avatar' => $path]);
    }
    // app/Models/User.php
    // Add these to your RELATIONSHIPS section (around line 50)

    public function messages(): HasMany
    {
        return $this->hasMany(UserMessage::class);
    }

    public function unreadMessages(): HasMany
    {
        return $this->hasMany(UserMessage::class)
            ->where('is_read', false)
            ->where(function ($query) {
                $query->whereNull('expires_at')
                    ->orWhere('expires_at', '>', now());
            });
    }

    // For system announcements (many-to-many)
    public function systemAnnouncements()
    {
        return $this->belongsToMany(UserMessage::class, 'user_message_assignments')
            ->withPivot('is_read')
            ->withTimestamps();
    }
    
    public function messageAssignments()
    {
        return $this->belongsToMany(UserMessage::class, 'user_message_assignments')
            ->withPivot('is_read')
            ->withTimestamps();
    }
}
