<?php
// app/Models/Transaction.php
namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo; // ADD THIS LINE

class Transaction extends Model
{
    protected $fillable = [
        'user_id',
        'wallet_transaction_id',
        'reference',
        'type',
        'status',
        'amount',
        'amount_charged',
        'network',
        'phone',
        'plan_code',
        'plan_description',
        'order_id',
        'request_id',
        'api_response',
        'remark',
        'service_provider',
        'service_details',
        'service_type', // ADD THIS IF YOU'RE USING IT
    ];

    protected $casts = [
        'api_response' => 'array',
        'amount' => 'decimal:2',
        'amount_charged' => 'decimal:2',
        'service_details' => 'array',
    ];

    // Remove or fix these methods - they're causing issues
    // public function getTypeAttribute()
    // {
    //     return $this->attributes['type'] ?? null;
    // }

    // Optional: Alias for service_type if you prefer that naming
    // public function getServiceTypeAttribute()
    // {
    //     return $this->attributes['type'] ?? null;
    // }

    // public function setServiceTypeAttribute($value)
    // {
    //     $this->attributes['type'] = $value;
    // }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    // NEW: Relationship to wallet transaction
    public function walletTransaction(): BelongsTo
    {
        return $this->belongsTo(WalletTransaction::class, 'wallet_transaction_id');
    }

    // NEW: Optional fee relationship (remove if not using fees table)
    public function fees()
    {
        return $this->morphMany(TransactionFee::class, 'transaction');
    }

    // OPTIONAL: Product relationship
    public function product()
    {
        return $this->belongsTo(Product::class, 'plan_code', 'plan_code')
            ->where('network_code', $this->network);
    }

    // Helper methods for better clarity
    public function isPurchase()
    {
        $type = $this->attributes['type'] ?? null;
        return in_array($type, ['airtime', 'data', 'electricity', 'cable']);
    }

    public function isSuccessful()
    {
        return $this->status === 'success';
    }

    public function isPending()
    {
        return $this->status === 'pending';
    }

    // Method to update with service details
    public function updateServiceDetails(array $details)
    {
        $this->update([
            'service_details' => array_merge($this->service_details ?? [], $details)
        ]);
    }
}
