<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Str;

class Setting extends Model
{
    protected $fillable = [
        'group',
        'category',
        'key',
        'value',
        'type',
        'is_encrypted',
        'is_sensitive',
        'input_type',
        'file_path',
        'file_disk',
        'description',
        'help_text',
        'validation_rules'
    ];

    protected $casts = [
        'is_encrypted' => 'boolean',
        'is_sensitive' => 'boolean',
        'validation_rules' => 'array',
    ];

    /**
     * Get the decrypted value of this setting
     */
    public function getDecryptedValue()
    {
        if (!$this->is_encrypted || !$this->value) {
            return $this->value;
        }

        try {
            return Crypt::decryptString($this->value);
        } catch (\Exception $e) {
            \Log::error('Failed to decrypt setting: ' . $this->key);
            return null;
        }
    }

    /**
     * Get the setting value with automatic decryption and casting
     */
    public function getValue()
    {
        $value = $this->is_encrypted ? $this->getDecryptedValue() : $this->value;
        return $this->castValue($value, $this->type);
    }

    /**
     * Cast a value based on its type
     */
    public static function castValue($value, $type)
    {
        if ($value === null) return null;

        return match ($type) {
            'boolean', 'bool' => filter_var($value, FILTER_VALIDATE_BOOLEAN),
            'integer', 'int' => (int) $value,
            'float', 'double' => (float) $value,
            'array', 'json' => json_decode($value, true),
            default => (string) $value,
        };
    }

    /**
     * Save an encrypted value
     */
    public function setEncryptedValue($value)
    {
        $this->is_encrypted = true;
        $this->is_sensitive = true;
        $this->value = Crypt::encryptString($value);
        $this->save();

        Cache::forget('app_settings');

        return $this;
    }

    /**
     * Set a setting value or create it if it doesn't exist
     */
    public static function setValue($key, $value, $type = 'string', $options = [])
    {
        $setting = self::updateOrCreate(
            ['key' => $key],
            [
                'type' => $type,
                'value' => $value,
                'is_encrypted' => $options['encrypt'] ?? false,
                'is_sensitive' => $options['sensitive'] ?? false,
                'group' => $options['group'] ?? 'general',
                'category' => $options['category'] ?? 'general',
                'input_type' => $options['input_type'] ?? 'text',
                'description' => $options['description'] ?? null,
                'help_text' => $options['help_text'] ?? null,
            ]
        );

        Cache::forget('app_settings');

        return $setting;
    }

    /**
     * Upload a file for this setting
     */
    public function uploadFile($file, $directory = 'settings', $disk = 'public')
    {
        // Delete old file if exists
        if ($this->file_path && Storage::disk($this->file_disk)->exists($this->file_path)) {
            Storage::disk($this->file_disk)->delete($this->file_path);
        }

        $path = $file->store($directory, ['disk' => $disk]);

        $this->file_path = $path;
        $this->file_disk = $disk;
        $this->value = $path;
        $this->save();

        Cache::forget('app_settings');

        return $path;
    }

    /**
     * Get the public URL of the file
     */
    public function getFileUrl()
    {
        if (!$this->file_path || !$this->file_disk) return null;

        return Storage::disk($this->file_disk)->url($this->file_path);
    }

    /**
     * Get the file content
     */
    public function getFileContent()
    {
        if (!$this->file_path || !$this->file_disk || !Storage::disk($this->file_disk)->exists($this->file_path)) {
            return null;
        }

        return Storage::disk($this->file_disk)->get($this->file_path);
    }

    /**
     * Get all settings as key => value pairs
     */
    public static function allValues(): array
    {
        try {
            return self::all()->mapWithKeys(function ($setting) {
                return [$setting->key => $setting->getValue()];
            })->toArray();
        } catch (\Exception $e) {
            \Log::error('Failed to load settings: ' . $e->getMessage());
            return [];
        }
    }
}
