<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Product extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'type',
        'network_code',
        'network_name',
        'plan_code',
        'description',
        'price',
        'discounted_price',
        'validity',
        'data_size',
        'is_active',
        'api_data',
    ];

    protected $casts = [
        'price' => 'decimal:2',
        'discounted_price' => 'decimal:2',
        'is_active' => 'boolean',
        'api_data' => 'array',
    ];

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeDataPlans($query)
    {
        return $query->where('type', 'data');
    }

    public function scopeAirtime($query)
    {
        return $query->where('type', 'airtime');
    }

    public function scopeByNetwork($query, $networkCode)
    {
        return $query->where('network_code', $networkCode);
    }

    public function getFormattedPriceAttribute()
    {
        return '₦' . number_format($this->price, 2);
    }

    public function getDisplayPriceAttribute()
    {
        if ($this->discounted_price) {
            return '₦' . number_format($this->discounted_price, 2);
        }
        return $this->formatted_price;
    }

    public function getHasDiscountAttribute()
    {
        return !is_null($this->discounted_price);
    }

    public function getDiscountPercentageAttribute()
    {
        if (!$this->has_discount) return 0;

        $discount = (($this->price - $this->discounted_price) / $this->price) * 100;
        return round($discount, 2);
    }
}
