<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\Cache;

class ApiProvider extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'code',
        'api_url',
        'api_key',
        'secret_key',
        'username',
        'password',
        'balance',
        'is_active',
        'priority',
        'supported_services',
        'supported_networks',
        'rates',
        'config',

        // VTU-specific fields
        'success_rate',
        'avg_response_time',
        'success_count',
        'failure_count',
        'circuit_status',
        'circuit_opened_at',
        'consecutive_failures',
        'circuit_failure_threshold',
        'concurrent_limit',
        'current_load',
        'service_configs',
        'last_activity_at',
        'timeout_seconds',
        'retry_attempts',
        'is_default',
        'min_amount',
        'max_amount',
        'supported_countries',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'is_default' => 'boolean',
        'balance' => 'decimal:2',
        'success_rate' => 'decimal:2',
        'avg_response_time' => 'integer',
        'success_count' => 'integer',
        'failure_count' => 'integer',
        'consecutive_failures' => 'integer',
        'circuit_failure_threshold' => 'integer',
        'concurrent_limit' => 'integer',
        'current_load' => 'integer',
        'priority' => 'integer',
        'timeout_seconds' => 'integer',
        'retry_attempts' => 'integer',
        'min_amount' => 'decimal:2',
        'max_amount' => 'decimal:2',

        // JSON casts
        'supported_services' => 'array',
        'supported_networks' => 'array',
        'rates' => 'array',
        'config' => 'array',
        'service_configs' => 'array',
        'supported_countries' => 'array',

        // Date casts
        'circuit_opened_at' => 'datetime',
        'last_activity_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    protected $attributes = [
        'balance' => 0,
        'is_active' => true,
        'priority' => 1,
        'success_rate' => 100.00,
        'avg_response_time' => 0,
        'success_count' => 0,
        'failure_count' => 0,
        'circuit_status' => 'closed',
        'consecutive_failures' => 0,
        'circuit_failure_threshold' => 5,
        'concurrent_limit' => 10,
        'current_load' => 0,
        'timeout_seconds' => 30,
        'retry_attempts' => 3,
        'is_default' => false,
        'min_amount' => 0,
        'max_amount' => 100000,
        'supported_services' => '["airtime", "data"]',
        'supported_networks' => '["MTN", "AIRTEL", "GLO", "9MOBILE"]',
        'supported_countries' => '["NG"]',
    ];

    /**
     * RELATIONSHIPS
     */
    public function products(): HasMany
    {
        return $this->hasMany(Product::class, 'api_provider_id');
    }

    public function transactions(): HasMany
    {
        return $this->hasMany(Transaction::class, 'api_provider_id');
    }

    /**
     * SCOPES
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeAvailable($query)
    {
        return $query->where('is_active', true)
            ->where('circuit_status', 'closed')
            ->whereRaw('current_load < concurrent_limit');
    }

    public function scopeForService($query, string $service)
    {
        return $query->whereJsonContains('supported_services', $service);
    }

    public function scopeForNetwork($query, string $network)
    {
        return $query->whereJsonContains('supported_networks', $network);
    }

    public function scopeForCountry($query, string $country)
    {
        return $query->whereJsonContains('supported_countries', $country);
    }

    public function scopeWithBalance($query, float $amount)
    {
        return $query->where('balance', '>=', $amount);
    }

    public function scopeOrderByPerformance($query)
    {
        return $query->orderByDesc('success_rate')
            ->orderBy('avg_response_time')
            ->orderByDesc('priority');
    }

    /**
     * VTU-SPECIFIC METHODS
     */

    /**
     * Check if provider can handle a purchase
     */
    public function canHandlePurchase(float $amount, string $service, ?string $network = null, ?string $country = 'NG'): bool
    {
        // Basic checks
        if (!$this->is_active) {
            return false;
        }

        if ($this->circuit_status === 'open') {
            // Check if circuit should be half-open
            if (!$this->shouldAttemptHalfOpen()) {
                return false;
            }
        }

        // Check load
        if ($this->current_load >= $this->concurrent_limit) {
            return false;
        }

        // Check balance
        if ($this->balance < $amount) {
            return false;
        }

        // Check amount limits
        if ($amount < $this->min_amount || $amount > $this->max_amount) {
            return false;
        }

        // Check service support
        if (!in_array($service, $this->supported_services ?? [])) {
            return false;
        }

        // Check network support for network-specific services
        if (in_array($service, ['airtime', 'data']) && $network) {
            if (!in_array($network, $this->supported_networks ?? [])) {
                return false;
            }
        }

        // Check country support
        if ($country && !in_array($country, $this->supported_countries ?? [])) {
            return false;
        }

        return true;
    }

    /**
     * Increment current load
     */
    public function incrementLoad(): bool
    {
        if ($this->current_load < $this->concurrent_limit) {
            $this->current_load++;
            $this->last_activity_at = now();
            $this->save();
            return true;
        }
        return false;
    }

    /**
     * Decrement current load
     */
    public function decrementLoad(): void
    {
        if ($this->current_load > 0) {
            $this->current_load--;
            $this->save();
        }
    }

    /**
     * Record transaction result
     */
    public function recordTransaction(bool $success, ?int $responseTime = null): void
    {
        if ($success) {
            $this->success_count++;
            $this->consecutive_failures = 0;

            // Update average response time
            if ($responseTime !== null) {
                $this->updateAverageResponseTime($responseTime);
            }

            // Close circuit if half-open
            if ($this->circuit_status === 'half_open') {
                $this->circuit_status = 'closed';
                $this->circuit_opened_at = null;
            }
        } else {
            $this->failure_count++;
            $this->consecutive_failures++;

            // Update circuit breaker
            $this->updateCircuitBreaker();
        }

        // Update success rate
        $this->updateSuccessRate();

        $this->last_activity_at = now();
        $this->save();

        // Clear cache
        $this->clearPerformanceCache();
    }

    /**
     * Update average response time
     */
    private function updateAverageResponseTime(int $newResponseTime): void
    {
        $totalTransactions = $this->success_count + $this->failure_count;

        if ($totalTransactions <= 1) {
            $this->avg_response_time = $newResponseTime;
        } else {
            // Weighted average (newer responses have more weight)
            $weight = 0.3; // 30% weight for new response
            $this->avg_response_time = (int)(
                ($this->avg_response_time * (1 - $weight)) +
                ($newResponseTime * $weight)
            );
        }
    }

    /**
     * Update success rate
     */
    private function updateSuccessRate(): void
    {
        $totalTransactions = $this->success_count + $this->failure_count;

        if ($totalTransactions > 0) {
            $this->success_rate = round(($this->success_count / $totalTransactions) * 100, 2);
        } else {
            $this->success_rate = 100.00;
        }
    }

    /**
     * Update circuit breaker status
     */
    private function updateCircuitBreaker(): void
    {
        // Check if we should open the circuit
        if ($this->consecutive_failures >= $this->circuit_failure_threshold) {
            $this->circuit_status = 'open';
            $this->circuit_opened_at = now();
        }
        // Check if we should go to half-open from open
        elseif ($this->circuit_status === 'open' && $this->shouldAttemptHalfOpen()) {
            $this->circuit_status = 'half_open';
        }
    }

    /**
     * Check if circuit should attempt half-open state
     */
    public function shouldAttemptHalfOpen(): bool
    {
        if ($this->circuit_status !== 'open') {
            return false;
        }

        $retryAfter = $this->config['circuit_retry_seconds'] ?? 60;
        $openedAt = $this->circuit_opened_at ?? now()->subHours(1);

        return now()->diffInSeconds($openedAt) >= $retryAfter;
    }

    /**
     * Get service-specific configuration
     */
    public function getServiceConfig(string $service): array
    {
        return $this->service_configs[$service] ?? $this->config ?? [];
    }

    /**
     * Check if provider has specific capability
     */
    public function hasCapability(string $capability, ?string $service = null): bool
    {
        if ($service) {
            $serviceConfig = $this->getServiceConfig($service);
            return in_array($capability, $serviceConfig['capabilities'] ?? []);
        }

        return in_array($capability, $this->config['capabilities'] ?? []);
    }

    /**
     * Get provider health score (0-100)
     */
    public function getHealthScore(): float
    {
        $cacheKey = "provider_health_{$this->id}";

        return Cache::remember($cacheKey, 60, function () {
            $score = 0;

            // Success rate (0-40 points)
            $score += ($this->success_rate * 0.4);

            // Response time (0-30 points)
            $maxTime = 5000; // 5 seconds
            $responseScore = max(0, 30 * (1 - ($this->avg_response_time / $maxTime)));
            $score += $responseScore;

            // Load factor (0-20 points)
            if ($this->concurrent_limit > 0) {
                $loadFactor = 1 - ($this->current_load / $this->concurrent_limit);
                $score += ($loadFactor * 20);
            } else {
                $score += 20; // Full points if no limit
            }

            // Recent activity (0-10 points)
            $lastActivity = $this->last_activity_at;
            if ($lastActivity) {
                $hoursSinceActivity = now()->diffInHours($lastActivity);
                $activityScore = max(0, 10 - ($hoursSinceActivity * 0.5));
                $score += $activityScore;
            }

            return min(100, round($score, 2));
        });
    }

    /**
     * Get provider status badge
     */
    public function getStatusBadge(): array
    {
        $healthScore = $this->getHealthScore();

        if (!$this->is_active) {
            return ['color' => 'danger', 'text' => 'Disabled', 'icon' => '✗'];
        }

        if ($this->circuit_status === 'open') {
            return ['color' => 'danger', 'text' => 'Circuit Open', 'icon' => '⚡'];
        }

        if ($this->circuit_status === 'half_open') {
            return ['color' => 'warning', 'text' => 'Testing', 'icon' => '⚠️'];
        }

        if ($healthScore >= 80) {
            return ['color' => 'success', 'text' => 'Healthy', 'icon' => '✓'];
        }

        if ($healthScore >= 60) {
            return ['color' => 'warning', 'text' => 'Degraded', 'icon' => '⚠️'];
        }

        return ['color' => 'danger', 'text' => 'Unhealthy', 'icon' => '✗'];
    }

    /**
     * Get load percentage
     */
    public function getLoadPercentage(): float
    {
        if ($this->concurrent_limit === 0) {
            return 0;
        }

        return round(($this->current_load / $this->concurrent_limit) * 100, 2);
    }

    /**
     * Check if provider is overloaded
     */
    public function isOverloaded(): bool
    {
        $threshold = $this->config['overload_threshold'] ?? 0.8; // 80%
        return $this->getLoadPercentage() >= ($threshold * 100);
    }

    /**
     * Get estimated wait time in seconds
     */
    public function getEstimatedWaitTime(): int
    {
        if ($this->current_load === 0) {
            return 0;
        }

        $avgProcessingTime = $this->avg_response_time / 1000; // Convert to seconds
        $queueSize = $this->current_load;

        return (int)($queueSize * $avgProcessingTime);
    }

    /**
     * Reset provider statistics
     */
    public function resetStatistics(): void
    {
        $this->update([
            'success_count' => 0,
            'failure_count' => 0,
            'success_rate' => 100.00,
            'avg_response_time' => 0,
            'consecutive_failures' => 0,
            'circuit_status' => 'closed',
            'circuit_opened_at' => null,
        ]);

        $this->clearPerformanceCache();
    }

    /**
     * Reset circuit breaker
     */
    public function resetCircuitBreaker(): void
    {
        $this->update([
            'circuit_status' => 'closed',
            'circuit_opened_at' => null,
            'consecutive_failures' => 0,
        ]);

        $this->clearPerformanceCache();
    }

    /**
     * Clear performance cache
     */
    private function clearPerformanceCache(): void
    {
        Cache::forget("provider_health_{$this->id}");

        // Clear all provider selection caches
        $keys = Cache::get('provider_cache_keys', []);
        foreach ($keys as $key) {
            if (str_contains($key, 'vtu_provider_')) {
                Cache::forget($key);
            }
        }
    }

    /**
     * Get formatted balance
     */
    public function getFormattedBalance(): string
    {
        $currency = $this->config['currency'] ?? 'NGN';
        $symbol = $this->getCurrencySymbol($currency);

        return $symbol . number_format($this->balance, 2);
    }

    /**
     * Get currency symbol
     */
    private function getCurrencySymbol(string $currency): string
    {
        return match ($currency) {
            'NGN' => '₦',
            default => $currency . ' ',
        };
    }

    /**
     * Get provider metrics for dashboard
     */
    public function getMetrics(): array
    {
        return [
            'id' => $this->id,
            'name' => $this->name,
            'code' => $this->code,
            'balance' => $this->balance,
            'formatted_balance' => $this->getFormattedBalance(),
            'is_active' => $this->is_active,
            'status' => $this->getStatusBadge(),
            'success_rate' => $this->success_rate,
            'response_time' => $this->avg_response_time,
            'response_time_formatted' => $this->avg_response_time . 'ms',
            'current_load' => $this->current_load,
            'concurrent_limit' => $this->concurrent_limit,
            'load_percentage' => $this->getLoadPercentage(),
            'load_status' => $this->isOverloaded() ? 'overloaded' : 'normal',
            'circuit_status' => $this->circuit_status,
            'health_score' => $this->getHealthScore(),
            'success_count' => $this->success_count,
            'failure_count' => $this->failure_count,
            'total_transactions' => $this->success_count + $this->failure_count,
            'estimated_wait_time' => $this->getEstimatedWaitTime(),
            'last_activity' => $this->last_activity_at ? $this->last_activity_at->diffForHumans() : 'Never',
            'supported_services' => $this->supported_services,
            'supported_networks' => $this->supported_networks,
            'priority' => $this->priority,
            'is_default' => $this->is_default,
        ];
    }

    /**
     * Get best provider for a specific request
     */
    public static function getBestForRequest(float $amount, string $service, ?string $network = null, ?string $country = 'NG'): ?self
    {
        $cacheKey = "best_provider_{$service}_{$network}_{$country}_{$amount}";

        return Cache::remember($cacheKey, 30, function () use ($amount, $service, $network, $country) {
            $providers = self::active()
                ->available()
                ->withBalance($amount)
                ->forService($service)
                ->when($network, function ($query) use ($network) {
                    return $query->forNetwork($network);
                })
                ->forCountry($country)
                ->get();

            if ($providers->isEmpty()) {
                return null;
            }

            // Score providers
            return $providers->sortByDesc(function ($provider) {
                return $provider->getHealthScore();
            })->first();
        });
    }

    /**
     * Update provider balance
     */
    public function updateBalance(float $amount, string $type = 'debit'): bool
    {
        if ($type === 'debit' && $this->balance < $amount) {
            return false;
        }

        $newBalance = $type === 'debit'
            ? $this->balance - $amount
            : $this->balance + $amount;

        $this->balance = $newBalance;
        return $this->save();
    }

    /**
     * Check if provider needs funding
     */
    public function needsFunding(): bool
    {
        $threshold = $this->config['low_balance_threshold'] ?? 1000;
        return $this->balance < $threshold;
    }

    /**
     * Get funding recommendation
     */
    public function getFundingRecommendation(): float
    {
        $recommended = $this->config['recommended_balance'] ?? 5000;
        return max(0, $recommended - $this->balance);
    }
}
