<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Wallet;
use App\Models\Transaction;
use Inertia\Inertia;
use App\Models\WalletTransaction;
use App\Models\MonnifyAccount;
use App\Models\VirtualAccount;
use App\Services\MonnifyService;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class WalletController extends Controller
{
    protected $monnifyService;

    public function __construct(MonnifyService $monnifyService)
    {
        $this->monnifyService = $monnifyService;
    }

    /**
     * Handle wallet deposits (NEW: Uses WalletTransaction instead of Transaction)
     */
    public function deposit(Request $request)
    {
        \Log::info('=== DEPOSIT REQUEST START (CUSTOM SERVICE) ===');
        \Log::info('Request:', $request->all());

        $validator = Validator::make($request->all(), [
            'amount' => 'required|numeric|min:100',
            'payment_method' => 'required|in:monnify,bank_transfer'
        ]);

        if ($validator->fails()) {
            \Log::error('Validation failed', $validator->errors()->toArray());
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $user = Auth::user();
        $wallet = $user->wallet;

        if (!$wallet) {
            return response()->json(['error' => 'Wallet not found'], 404);
        }

        try {
            DB::beginTransaction();

            // Create WALLET transaction (not regular transaction)
            $reference = 'DEP_' . Str::random(12);

            $walletTransaction = WalletTransaction::create([
                'user_id' => $user->id,
                'wallet_id' => $wallet->id,
                'reference' => $reference,
                'amount' => $request->amount,
                'type' => 'deposit',
                'status' => 'pending',
                'payment_method' => $request->payment_method,
                'payment_details' => []
            ]);

            \Log::info('WalletTransaction created with ID: ' . $walletTransaction->id . ' and reference: ' . $reference);

            if ($request->payment_method === 'monnify') {
                // USE YOUR CUSTOM SERVICE HERE
                $monnifyData = [
                    'amount' => $request->amount,
                    'customer_name' => $user->name,
                    'customer_email' => $user->email,
                    'reference' => $reference,
                    'description' => 'Wallet Funding',
                    'user_id' => $user->id,
                    'wallet_id' => $wallet->id
                ];

                \Log::info('Calling custom MonnifyService::initializeTransaction', $monnifyData);

                $response = $this->monnifyService->initializeTransaction($monnifyData);

                \Log::info('Custom service response:', $response);

                // Update wallet transaction payment_details
                $walletTransaction->payment_details = [
                    'monnify_transaction_reference' => $response['transactionReference'],
                    'monnify_payment_reference' => $response['paymentReference'],
                ];
                $walletTransaction->save();

                \Log::info('WalletTransaction payment_details updated:', $walletTransaction->payment_details);

                DB::commit();

                return response()->json([
                    'success' => true,
                    'paymentUrl' => $response['checkoutUrl'],
                    'reference' => $reference,
                    'monnifyTransactionReference' => $response['transactionReference'],
                    'transaction_id' => $walletTransaction->id
                ]);
            } else {
                // For bank transfer, you might add bank details
                $walletTransaction->payment_details = [
                    'bank_name' => env('COMPANY_BANK_NAME', 'Your Bank'),
                    'account_number' => env('COMPANY_BANK_ACCOUNT_NUMBER', '1234567890'),
                    'account_name' => env('COMPANY_BANK_ACCOUNT_NAME', 'Your Company'),
                ];
                $walletTransaction->save();

                DB::commit();
                return response()->json([
                    'success' => true,
                    'message' => 'Manual transfer required',
                    'reference' => $reference,
                    'bank_details' => $walletTransaction->payment_details
                ]);
            }
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Deposit Error (Custom Service):', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'error' => 'Deposit failed: ' . $e->getMessage()
            ], 500);
        }
    }


    /**
     * Initialize deposit (for form submissions)
     */
    public function initializeDeposit(Request $request)
    {
        // Call the existing deposit method
        return $this->deposit($request);
    }
    public function monnifyCallback(Request $request)
    {
        $paymentReference = $request->input('paymentReference');
        $status = $request->input('status');

        try {
            // Verify transaction with Monnify
            $monnifyTransaction = $this->monnifyService->verifyTransaction($paymentReference);

            if (!$monnifyTransaction || $monnifyTransaction['paymentStatus'] !== 'PAID') {
                return redirect()->route('dashboard')->with('error', 'Payment verification failed');
            }

            // Find transaction
            $transaction = Transaction::where('metadata->monnify_reference', $paymentReference)->first();

            if (!$transaction) {
                return redirect()->route('dashboard')->with('error', 'Transaction not found');
            }

            if ($transaction->status === 'success') {
                return redirect()->route('dashboard')->with('success', 'Payment already processed');
            }

            DB::beginTransaction();

            // Update transaction
            $transaction->update([
                'status' => 'success',
                'completed_at' => now(),
                'metadata' => array_merge($transaction->metadata ?? [], [
                    'monnify_details' => $monnifyTransaction
                ])
            ]);

            // Update wallet balance
            $wallet = $transaction->wallet;
            $wallet->increment('balance', $transaction->amount);

            DB::commit();

            return redirect()->route('dashboard')->with('success', 'Payment successful! Wallet funded.');
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Monnify Callback Error: ' . $e->getMessage());

            return redirect()->route('dashboard')->with('error', 'Payment processing failed');
        }
    }

    public function createVirtualAccount(Request $request)
    {
        $user = Auth::user();

        // Check if user already has a virtual account
        $existingAccount = MonnifyAccount::where('user_id', $user->id)->first();
        if ($existingAccount) {
            return response()->json([
                'success' => false,
                'message' => 'Virtual account already exists'
            ]);
        }

        // Validate KYC data
        $validator = Validator::make($request->all(), [
            'bvn' => 'required_without:nin|string|size:11',
            'nin' => 'required_without:bvn|string|size:11',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        try {
            DB::beginTransaction();

            // Update user KYC status
            $user->update([
                'kyc_status' => 'pending',
                'bvn' => $request->bvn ?? null,
                'nin' => $request->nin ?? null
            ]);

            // Create virtual account with Monnify
            $accountData = [
                'user_id' => $user->id,
                'full_name' => $user->name,
                'email' => $user->email,
                'bvn' => $request->bvn ?? null,
                'nin' => $request->nin ?? null,
                'account_name' => $user->name . ' - Virtual Account'
            ];

            $monnifyResponse = $this->monnifyService->createReservedAccount($accountData);

            // Save account details
            $monnifyAccount = MonnifyAccount::create([
                'user_id' => $user->id,
                'account_reference' => $monnifyResponse['accountReference'],
                'account_number' => $monnifyResponse['accounts'][0]['accountNumber'],
                'bank_name' => $monnifyResponse['accounts'][0]['bankName'],
                'account_name' => $monnifyResponse['accountName'],
                'bank_details' => $monnifyResponse['accounts'],
                'customer_email' => $monnifyResponse['customerEmail'],
                'customer_name' => $monnifyResponse['customerName'],
                'bvn' => $request->bvn ?? null,
                'nin' => $request->nin ?? null,
                'status' => 'active'
            ]);

            // Update user KYC status
            $user->update(['kyc_status' => 'admin_approved']);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Virtual account created successfully',
                'account' => $monnifyAccount
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Virtual Account Creation Error: ' . $e->getMessage());

            return response()->json([
                'error' => 'Failed to create virtual account. Please try again.'
            ], 500);
        }
    }
    
    /**
     * Show the full wallet funding page
     * UPDATED: Returns wallet transactions instead of all transactions
     */
    public function fund(Request $request)
    {
        $user = Auth::user();
        $wallet = $user->wallet;

        // Ensure wallet exists
        if (!$wallet) {
            $wallet = Wallet::create([
                'user_id' => $user->id,
                'balance' => 0,
                'status' => 'active'
            ]);
        }

        // Get static account if exists
        $staticAccount = VirtualAccount::where('user_id', $user->id)->first();

        // Get company account details from config or .env
        $companyAccountDetails = [
            'account_number' => env('COMPANY_BANK_ACCOUNT_NUMBER', '1234567890'),
            'bank_name' => env('COMPANY_BANK_NAME', 'Your Bank Name'),
            'account_name' => env('COMPANY_BANK_ACCOUNT_NAME', 'Your Company Name'),
            'notes' => env('COMPANY_BANK_TRANSFER_NOTES', 'Use your registered name/email as depositor. Funds reflect in 1-2 hours after confirmation.'),
            'active' => env('COMPANY_BANK_ACTIVE', true),
        ];

        // Get WALLET transaction history (deposits & withdrawals only) - UPDATED
        $walletHistory = WalletTransaction::where('user_id', $user->id)
            ->whereIn('type', ['deposit', 'withdrawal'])
            ->latest()
            ->take(5)
            ->get()
            ->map(function ($transaction) {
                return [
                    'id' => $transaction->id,
                    'amount' => $transaction->amount,
                    'type' => $transaction->type,
                    'status' => $transaction->status,
                    'description' => $this->getWalletTransactionDescription($transaction),
                    'date' => $transaction->created_at->format('M d, Y h:i A'),
                    'payment_method' => $transaction->payment_method,
                ];
            });

        // Get purchase transaction history (airtime, data purchases) - OPTIONAL
        $purchaseHistory = Transaction::where('user_id', $user->id)
            ->latest()
            ->take(5)
            ->get()
            ->map(function ($transaction) {
                return [
                    'id' => $transaction->id,
                    'amount' => $transaction->amount,
                    'type' => $transaction->type, // This is still 'type' column in the database
                    'status' => $transaction->status,
                    'description' => $transaction->plan_description ?? ucfirst($transaction->type) . ' Purchase',
                    'date' => $transaction->created_at->format('M d, Y h:i A'),
                    'network' => $transaction->network,
                    'phone' => $transaction->phone,
                ];
            });

        return Inertia::render('Wallet/Fund', [
            'user' => [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'kyc_status' => $user->kyc_status ?? 'unverified',
            ],
            'wallet' => [
                'balance' => $wallet->balance,
                'currency' => 'NGN',
            ],
            'paymentMethods' => [
                [
                    'id' => 'monnify',
                    'name' => 'Bank Transfer/Card',
                    'icon' => 'Bank',
                    'requires_account_details' => false,
                ],
                [
                    'id' => 'bank_transfer',
                    'name' => 'Direct Bank Transfer',
                    'icon' => 'BuildingBank',
                    'requires_account_details' => true,
                ],
                [
                    'id' => 'virtual_account',
                    'name' => 'Virtual Account',
                    'icon' => 'CreditCard',
                    'requires_account_details' => false,
                ],
                [
                    'id' => 'ussd',
                    'name' => 'USSD',
                    'icon' => 'Smartphone',
                    'requires_account_details' => false,
                ],
            ],
            'quickAmounts' => [1000, 2000, 5000, 10000, 20000, 50000],
            'hasVirtualAccount' => $staticAccount ? true : false,
            'staticAccount' => $staticAccount ? [
                'account_number' => $staticAccount->account_number,
                'bank_name' => $staticAccount->bank_name,
                'account_name' => $staticAccount->account_name,
            ] : null,
            'companyAccountDetails' => $companyAccountDetails,
            'walletHistory' => $walletHistory, // UPDATED: Use wallet transactions
            'purchaseHistory' => $purchaseHistory, // OPTIONAL: If you want to show purchases too
        ]);
    }

    /**
     * Helper method to get wallet transaction description
     */
    private function getWalletTransactionDescription($transaction)
    {
        if ($transaction->type === 'deposit') {
            $method = $transaction->payment_method ?? 'manual';
            return ucfirst($method) . ' Deposit';
        } elseif ($transaction->type === 'withdrawal') {
            $method = $transaction->payment_method ?? 'bank';
            return ucfirst($method) . ' Withdrawal';
        }

        return ucfirst($transaction->type);
    }

    
    /**
     * Handle withdrawals (UPDATED: Uses WalletTransaction)
     */
    public function withdraw(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'amount' => 'required|numeric|min:100',
            'bank_code' => 'required|string',
            'account_number' => 'required|string',
            'account_name' => 'required|string'
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $user = Auth::user();
        $wallet = $user->wallet;

        // Check balance
        if ($wallet->balance < $request->amount) {
            return response()->json(['error' => 'Insufficient balance'], 400);
        }

        // Validate bank account
        $accountValidation = $this->monnifyService->validateAccount(
            $request->account_number,
            $request->bank_code
        );

        if (!$accountValidation) {
            return response()->json(['error' => 'Invalid bank account details'], 400);
        }

        try {
            DB::beginTransaction();

            // Create WALLET transaction for withdrawal
            $reference = 'WTH_' . Str::random(12);
            $walletTransaction = WalletTransaction::create([
                'user_id' => $user->id,
                'wallet_id' => $wallet->id,
                'reference' => $reference,
                'amount' => $request->amount,
                'type' => 'withdrawal',
                'status' => 'pending',
                'payment_method' => 'bank_transfer',
                'payment_details' => [
                    'bank_code' => $request->bank_code,
                    'account_number' => $request->account_number,
                    'account_name' => $request->account_name,
                    'bank_name' => $accountValidation['bankName']
                ]
            ]);

            // Debit wallet
            $wallet->decrement('balance', $request->amount);

            // Initiate transfer via Monnify
            $transferData = [
                'amount' => $request->amount,
                'reference' => $reference,
                'narration' => 'Withdrawal from VTU Wallet',
                'bank_code' => $request->bank_code,
                'account_number' => $request->account_number,
                'account_name' => $request->account_name
            ];

            $transferResponse = $this->monnifyService->initiateTransfer($transferData);

            // Update wallet transaction with Monnify reference
            $walletTransaction->update([
                'status' => 'success',
                'completed_at' => now(),
                'payment_details' => array_merge($walletTransaction->payment_details, [
                    'monnify_transfer_reference' => $transferResponse['reference'],
                    'transfer_response' => $transferResponse
                ]),
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Withdrawal initiated successfully',
                'reference' => $reference
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Withdrawal Error: ' . $e->getMessage());

            // Refund wallet if error occurred
            if (isset($wallet)) {
                $wallet->increment('balance', $request->amount);
            }

            // Mark wallet transaction as failed
            if (isset($walletTransaction)) {
                $walletTransaction->update([
                    'status' => 'failed',
                    'payment_details' => array_merge($walletTransaction->payment_details ?? [], [
                        'error' => $e->getMessage()
                    ]),
                ]);
            }

            return response()->json([
                'error' => 'Failed to process withdrawal. Please try again.'
            ], 500);
        }
    }


    public function getBanks()
    {
        try {
            $banks = $this->monnifyService->getBanks();
            return response()->json(['banks' => $banks]);
        } catch (\Exception $e) {
            \Log::error('Get Banks Error: ' . $e->getMessage());
            return response()->json(['banks' => []]);
        }
    }
    /**
     * Get transaction history
     */
    public function walletHistory(Request $request)
    {
        $user = Auth::user();

        // Get WALLET transactions (deposits, withdrawals)
        $walletTransactions = WalletTransaction::where('user_id', $user->id)
            ->with('wallet')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        // Get PURCHASE transactions (airtime, data)
        $purchaseTransactions = Transaction::where('user_id', $user->id)
            ->whereIn('type', ['airtime', 'data']) // Only purchase types
            ->orderBy('created_at', 'desc')
            ->paginate(20, ['*'], 'purchase_page');

        if (class_exists(\Inertia\Inertia::class)) {
            return Inertia::render('Wallet/History', [
                'walletTransactions' => $walletTransactions,
                'purchaseTransactions' => $purchaseTransactions,
                'wallet' => $user->wallet
            ]);
        }

        return view('wallet.history', [
            'walletTransactions' => $walletTransactions,
            'purchaseTransactions' => $purchaseTransactions,
            'wallet' => $user->wallet
        ]);
    }

    /**
     * API endpoint for transaction history
     */
    public function transactionHistory(Request $request)
    {
        $user = Auth::user();

        // Get wallet transactions
        $walletTransactions = WalletTransaction::where('user_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        // Get purchase transactions  
        $purchaseTransactions = Transaction::where('user_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->paginate(20, ['*'], 'purchase_page');

        return response()->json([
            'wallet_transactions' => $walletTransactions,
            'purchase_transactions' => $purchaseTransactions
        ]);
    }


    public function getVirtualAccount()
    {
        $user = Auth::user();
        $account = MonnifyAccount::where('user_id', $user->id)->first();

        if (!$account) {
            return response()->json(['error' => 'No virtual account found'], 404);
        }

        return response()->json([
            'account' => $account
        ]);
    }

    /**
     * Handle Monnify GET callback (user browser redirect)
     * UPDATED: Now works with WalletTransaction
     */
    public function handleMonnifyRedirect(Request $request)
    {
        \Log::info('=== MONNIFY GET REDIRECT CALLBACK START ===');
        \Log::info('Full Request URL: ' . $request->fullUrl());

        $queryParams = $request->query();
        \Log::info('GET Parameters:', is_array($queryParams) ? $queryParams : $queryParams->all());

        // Get the payment reference from query
        $paymentReference = $request->query('paymentReference');

        \Log::info('Payment Reference from callback: ' . ($paymentReference ?? 'NULL'));

        if (!$paymentReference) {
            \Log::error('No paymentReference found in callback');
            return redirect()->route('wallet.deposit')->with('error', 'No transaction reference found.');
        }

        try {
            // Find WALLET transaction by reference (DEP_xxx)
            $walletTransaction = WalletTransaction::where('reference', $paymentReference)->first();

            if (!$walletTransaction) {
                \Log::error('WalletTransaction not found for reference: ' . $paymentReference);
                return redirect()->route('wallet.deposit')->with('error', 'Transaction not found.');
            }

            // Get the Monnify transaction reference from payment_details
            $monnifyTransactionReference = $walletTransaction->payment_details['monnify_transaction_reference'] ?? null;

            if (!$monnifyTransactionReference) {
                \Log::error('No Monnify transaction reference stored for: ' . $paymentReference);
                return redirect()->route('wallet.deposit')->with('error', 'Cannot verify payment.');
            }

            \Log::info('Found wallet transaction. Monnify reference: ' . $monnifyTransactionReference);

            // Verify with Monnify using transactionReference (MNFY|xx)
            $verification = $this->monnifyService->verifyTransaction($monnifyTransactionReference);

            if (!$verification) {
                \Log::error('Monnify service returned null for verification');
                return redirect()->route('wallet.deposit')->with('error', 'Payment verification failed.');
            }

            // Check if verification was successful
            if (!isset($verification['success']) || $verification['success'] !== true) {
                \Log::error('Payment verification failed:', $verification);
                return redirect()->route('wallet.deposit')->with('error', 'Payment verification failed: ' . ($verification['error'] ?? 'Unknown error'));
            }

            $paymentStatus = $verification['paymentStatus'] ?? null;
            \Log::info('Payment Status from verification: ' . ($paymentStatus ?? 'NULL'));

            if ($walletTransaction->status === 'success') {
                \Log::info('Wallet transaction already processed');
                return redirect()->route('wallet.deposit')->with('info', 'Payment already processed.');
            }

            if ($paymentStatus === 'PAID' || $paymentStatus === 'SUCCESS') {
                DB::beginTransaction();

                // Update wallet transaction
                $walletTransaction->update([
                    'status' => 'success',
                    'completed_at' => now(),
                    'payment_details' => array_merge($walletTransaction->payment_details ?? [], [
                        'monnify_verification' => $verification,
                        'callback_data' => $request->query(),
                    ]),
                ]);

                // Credit wallet
                $wallet = $walletTransaction->wallet;
                $wallet->increment('balance', $walletTransaction->amount);

                DB::commit();

                \Log::info('Payment successful! Wallet credited with: ' . $walletTransaction->amount);
                \Log::info('New wallet balance: ' . $wallet->balance);

                return redirect()->route('wallet.deposit')->with([
                    'success' => 'Payment successful! Your wallet has been credited with ₦' . number_format($walletTransaction->amount) . '.',
                    'amount' => $walletTransaction->amount,
                    'new_balance' => $wallet->balance,
                ]);
            } else {
                \Log::warning('Payment not successful. Status: ' . $paymentStatus);

                // Update wallet transaction as failed
                $walletTransaction->update([
                    'status' => 'failed',
                    'payment_details' => array_merge($walletTransaction->payment_details ?? [], [
                        'verification_error' => $verification,
                    ]),
                ]);

                return redirect()->route('wallet.deposit')->with('error', 'Payment was not successful. Status: ' . ($paymentStatus ?? 'unknown'));
            }
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Monnify callback error: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            return redirect()->route('wallet.deposit')->with('error', 'Payment processing failed: ' . $e->getMessage());
        }
    }


    /**
     * Handle Monnify POST callback (server-to-server)
     * UPDATED: Now works with WalletTransaction
     */
    public function handleMonnifyCallback(Request $request)
    {
        Log::info('=== MONNIFY POST CALLBACK (WEBHOOK) ===');
        Log::info('POST Data:', $request->all());

        // Get the payment reference
        $paymentReference = $request->input('paymentReference');

        if (!$paymentReference) {
            Log::error('No paymentReference in POST callback');
            return response()->json(['status' => 'error', 'message' => 'Invalid callback']);
        }

        try {
            // Find WALLET transaction by reference
            $walletTransaction = WalletTransaction::where('reference', $paymentReference)->first();

            if (!$walletTransaction) {
                Log::error('WalletTransaction not found for reference: ' . $paymentReference);
                return response()->json(['status' => 'error', 'message' => 'Transaction not found']);
            }

            // Verify the transaction with Monnify
            $verification = $this->monnifyService->verifyTransaction(
                $walletTransaction->payment_details['monnify_transaction_reference'] ?? $paymentReference
            );

            if ($verification['success'] && ($verification['paymentStatus'] === 'PAID' || $verification['paymentStatus'] === 'SUCCESS')) {
                if ($walletTransaction->status !== 'success') {
                    DB::beginTransaction();

                    // Update wallet transaction
                    $walletTransaction->update([
                        'status' => 'success',
                        'completed_at' => now(),
                        'payment_details' => array_merge($walletTransaction->payment_details ?? [], [
                            'webhook_verification' => $verification['data'],
                            'webhook_callback_data' => $request->all(),
                        ]),
                    ]);

                    // Credit wallet
                    $wallet = $walletTransaction->wallet;
                    $wallet->increment('balance', $walletTransaction->amount);

                    DB::commit();

                    Log::info('Deposit completed via webhook: ' . $paymentReference);
                }
            }

            Log::info('=== MONNIFY POST CALLBACK PROCESSED ===');
            return response()->json(['status' => 'success']);
        } catch (\Exception $e) {
                Log::error('Webhook processing error: ' . $e->getMessage());
                return response()->json(['status' => 'error', 'message' => 'Processing failed'], 500);
            }
        }
    }