<?php
// app/Http/Controllers/TransactionController.php
namespace App\Http\Controllers;

use App\Models\Transaction;
use Inertia\Inertia;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class TransactionController extends Controller
{
    public function index()
    {
        $transactions = Transaction::where('user_id', Auth::id())
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        // Transform for frontend
        $transformedTransactions = $transactions->through(function ($transaction) {
            return [
                'id' => $transaction->id,
                'reference' => $transaction->reference,
                'type' => $transaction->type,
                'status' => $transaction->status,
                'amount' => (float) $transaction->amount,
                'amount_charged' => $transaction->amount_charged ? (float) $transaction->amount_charged : null,
                'network' => $transaction->network,
                'phone' => $transaction->phone,
                'plan_description' => $transaction->plan_description,
                'plan_code' => $transaction->plan_code,
                'order_id' => $transaction->order_id,
                'request_id' => $transaction->request_id,
                'created_at' => $transaction->created_at->format('Y-m-d H:i:s'),
                'updated_at' => $transaction->updated_at->format('Y-m-d H:i:s'),
                'api_response' => $transaction->api_response,
                'remark' => $transaction->remark,
                'is_test' => (bool) $transaction->is_test,
            ];
        });

        return Inertia::render('Transactions/Index', [
            'transactions' => $transformedTransactions
        ]);
    }
    public function downloadReceipt(Request $request, $id)
    {
        $transaction = Transaction::where('user_id', Auth::id())
            ->findOrFail($id);

        $user = Auth::user();

        // Generate PDF
        $pdf = $this->generateReceiptPdf($transaction, $user);

        // Return PDF as download response
        $filename = "receipt-{$transaction->reference}.pdf";

        return response($pdf)
            ->header('Content-Type', 'application/pdf')
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '"')
            ->header('Content-Length', strlen($pdf));
    }

    private function generateReceiptPdf($transaction, $user)
    {
        // Configure dompdf
        $options = new Options();
        $options->set('defaultFont', 'DejaVu Sans');
        $options->set('isHtml5ParserEnabled', true);
        $options->set('isRemoteEnabled', true);

        $dompdf = new Dompdf($options);

        // Generate HTML
        $html = $this->getReceiptHtml($transaction, $user);

        // Load HTML
        $dompdf->loadHtml($html);

        // Set paper size and orientation
        $dompdf->setPaper('A4', 'portrait');

        // Render PDF
        $dompdf->render();

        // Output PDF
        return $dompdf->output();
    }

    private function getReceiptHtml($transaction, $user)
    {
        $date = now()->format('d/m/Y H:i:s');
        $transactionType = $transaction->type === 'deposit' ? 'Credit' : 'Debit';
        $paymentMethod = $transaction->type === 'deposit' ? 'Bank Deposit' : 'Wallet';
        $statusColor = $transaction->status === 'success' || $transaction->status === 'completed' ? '#059669' : ($transaction->status === 'failed' ? '#dc2626' : '#d97706');

        return <<<HTML
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <title>Receipt - {$transaction->reference}</title>
            <style>
                body {
                    font-family: 'DejaVu Sans', Arial, sans-serif;
                    max-width: 800px;
                    margin: 0 auto;
                    padding: 20px;
                    color: #333;
                }
                .receipt-header {
                    text-align: center;
                    border-bottom: 2px solid #4f46e5;
                    padding-bottom: 20px;
                    margin-bottom: 30px;
                }
                .receipt-header h1 {
                    color: #4f46e5;
                    margin: 0;
                    font-size: 24px;
                }
                .receipt-header .subtitle {
                    color: #666;
                    font-size: 14px;
                    margin-top: 5px;
                }
                .info-grid {
                    display: table;
                    width: 100%;
                    margin-bottom: 20px;
                    border-collapse: collapse;
                }
                .info-row {
                    display: table-row;
                }
                .info-label, .info-value {
                    display: table-cell;
                    padding: 8px 0;
                    border-bottom: 1px solid #eee;
                }
                .info-label {
                    width: 40%;
                    font-weight: bold;
                    color: #555;
                }
                .info-value {
                    width: 60%;
                    color: #222;
                }
                .amount-section {
                    text-align: center;
                    padding: 20px;
                    background: #f8fafc;
                    border-radius: 8px;
                    margin: 25px 0;
                    border: 1px solid #e5e7eb;
                }
                .amount-label {
                    font-size: 14px;
                    color: #4b5563;
                    margin-bottom: 8px;
                }
                .amount {
                    font-size: 36px;
                    font-weight: bold;
                    color: #111827;
                    margin: 10px 0;
                }
                .transaction-type {
                    display: inline-block;
                    padding: 6px 12px;
                    border-radius: 20px;
                    font-weight: 600;
                    font-size: 14px;
                }
                .transaction-type.credit {
                    background: #d1fae5;
                    color: #065f46;
                }
                .transaction-type.debit {
                    background: #fee2e2;
                    color: #991b1b;
                }
                .section-title {
                    font-size: 16px;
                    font-weight: 600;
                    color: #374151;
                    margin: 25px 0 15px 0;
                    padding-bottom: 8px;
                    border-bottom: 1px solid #e5e7eb;
                }
                .footer {
                    text-align: center;
                    margin-top: 40px;
                    padding-top: 20px;
                    border-top: 1px solid #e5e7eb;
                    color: #6b7280;
                    font-size: 12px;
                    line-height: 1.5;
                }
                .status-badge {
                    display: inline-block;
                    padding: 4px 12px;
                    border-radius: 12px;
                    font-size: 12px;
                    font-weight: 600;
                }
                .company-info {
                    text-align: center;
                    margin-bottom: 30px;
                    padding-bottom: 20px;
                    border-bottom: 1px solid #e5e7eb;
                }
                .company-name {
                    font-size: 20px;
                    font-weight: bold;
                    color: #4f46e5;
                    margin-bottom: 5px;
                }
                
                .company-tagline {
                    font-size: 12px;
                    color: #6b7280;
                }
            </style>
        </head>
        <body>
            <div class="company-info">
                <div class="company-name">Your Company Name</div>
                <div class="company-tagline">Transaction Receipt • Official Document</div>
            </div>
            
            <div class="receipt-header">
                <h1>TRANSACTION RECEIPT</h1>
                <div class="subtitle">Receipt #{$transaction->reference}</div>
            </div>
            
            <div class="info-grid">
                <div class="info-row">
                    <div class="info-label">Transaction ID:</div>
                    <div class="info-value">{$transaction->reference}</div>
                </div>
                <div class="info-row">
                    <div class="info-label">Date & Time:</div>
                    <div class="info-value">{$transaction->created_at->format('d/m/Y H:i:s')}</div>
                </div>
                <div class="info-row">
                    <div class="info-label">Status:</div>
                    <div class="info-value">
                        <span class="status-badge" style="background-color: {$statusColor}; color: white;">
                            " . strtoupper($transaction->status) . "
                        </span>
                    </div>
                </div>
                <div class="info-row">
                    <div class="info-label">Payment Method:</div>
                    <div class="info-value">{$paymentMethod}</div>
                </div>
            </div>
            
            <div class="amount-section">
                <div class="amount-label">TRANSACTION AMOUNT</div>
                <div class="amount">₦" . number_format($transaction->amount_charged ?: $transaction->amount, 2) . "</div>
                <div class="transaction-type " . ($transactionType === 'Credit' ? 'credit' : 'debit') . ">
                    {$transactionType}
                </div>
            </div>
            
            <div class="section-title">CUSTOMER INFORMATION</div>
            <div class="info-grid">
                <div class="info-row">
                    <div class="info-label">Customer Name:</div>
                    <div class="info-value">{$user->name}</div>
                </div>
                <div class="info-row">
                    <div class="info-label">Customer Email:</div>
                    <div class="info-value">{$user->email}</div>
                </div>
                " . ($transaction->phone ? "
                <div class="info-row">
                    <div class="info-label">Phone Number:</div>
                    <div class="info-value">{$transaction->phone}</div>
                </div>
                " : "") . "
            </div>
            
            <div class="section-title">TRANSACTION DETAILS</div>
            <div class="info-grid">
                <div class="info-row">
                    <div class="info-label">Product Description:</div>
                    <div class="info-value">" . ($transaction->plan_description ?: 'Purchase') . "</div>
                </div>
                " . ($transaction->network ? "
                <div class="info-row">
                    <div class="info-label">Network:</div>
                    <div class="info-value">{$transaction->network}</div>
                </div>
                " : "") . "
                <div class="info-row">
                    <div class="info-label">Transaction Type:</div>
                    <div class="info-value">" . ucfirst($transaction->type) . "</div>
                </div>
                <div class="info-row">
                    <div class="info-label">Original Amount:</div>
                    <div class="info-value">₦" . number_format($transaction->amount, 2) . "</div>
                </div>
                " . ($transaction->amount_charged ? "
                <div class="info-row">
                    <div class="info-label">Amount Charged:</div>
                    <div class="info-value">₦" . number_format($transaction->amount_charged, 2) . "</div>
                </div>
                " : "") . "
            </div>
            
            " . ($transaction->remark ? "
            <div class="section-title">REMARKS</div>
            <div style="padding: 15px; background: #f9fafb; border-radius: 6px; margin-bottom: 20px;">
                {$transaction->remark}
            </div>
            " : "") . "
            
            <div class="section-title">AUTHORIZATION</div>
            <div style="padding: 15px; border: 1px dashed #d1d5db; border-radius: 6px; margin-bottom: 20px;">
                <div style="display: flex; justify-content: space-between; align-items: center;">
                    <div>
                        <div style="font-size: 12px; color: #6b7280; margin-bottom: 5px;">Authorized Signature</div>
                        <div style="height: 40px; border-bottom: 1px solid #d1d5db; width: 200px;"></div>
                    </div>
                    <div style="text-align: right;">
                        <div style="font-size: 12px; color: #6b7280; margin-bottom: 5px;">Date</div>
                        <div style="font-weight: 600;">{$date}</div>
                    </div>
                </div>
            </div>
            
            <div class="footer">
                <p>This is an official receipt. Please keep it for your records.</p>
                <p>Generated on {$date} • Receipt ID: {$transaction->reference}</p>
                <p>For any queries, please contact our support team.</p>
                <p style="margin-top: 10px; font-size: 10px; color: #9ca3af;">
                    Document generated electronically • Valid without signature
                </p>
            </div>
        </body>
        </html>
        HTML;
    }
}