<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Auth;
use App\Models\Transaction;
use App\Models\User;

class PaymentController extends Controller
{
    // Get Momnify Access Token
    private function getAccessToken()
    {
        // Check if token exists and is still valid
        if (session()->has('monnify_access_token') && session()->has('monnify_token_expiry')) {
            if (time() < session('monnify_token_expiry')) {
                return session('monnify_access_token');
            }
        }
        
        // Get new token
        $apiKey = config('services.monnify.api_key');
        $secretKey = config('services.monnify.secret_key');
        
        $credentials = base64_encode($apiKey . ':' . $secretKey);
        
        $response = Http::withHeaders([
            'Authorization' => 'Basic ' . $credentials,
        ])->post('https://api.monnify.com/api/v1/auth/login');
        
        if ($response->successful()) {
            $data = $response->json();
            $token = $data['responseBody']['accessToken'];
            $expiresIn = $data['responseBody']['expiresIn']; // usually in seconds
            
            // Store token in session
            session()->put('monnify_access_token', $token);
            session()->put('monnify_token_expiry', time() + $expiresIn - 60); // Subtract 60 seconds for safety
            
            return $token;
        }
        
        throw new \Exception('Failed to get Momnify access token');
    }
    
    // Initialize Payment
    public function initializePayment(Request $request)
    {
        $request->validate([
            'amount' => 'required|numeric|min:100', // Minimum amount e.g., 100 Naira
        ]);
        
        $amount = $request->amount;
        $reference = uniqid('txn_');
        
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->getAccessToken(),
            ])->post('https://api.monnify.com/api/v1/merchant/transactions/initiate', [
                'amount' => $amount,
                'customerName' => Auth::user()->name,
                'customerEmail' => Auth::user()->email,
                'paymentReference' => $reference,
                'paymentDescription' => 'Wallet Top-up',
                'currencyCode' => 'NGN',
                'contractCode' => config('services.monnify.contract_code'),
                'redirectUrl' => route('payment.callback'), // Your callback URL
                'paymentMethods' => ['CARD', 'ACCOUNT_TRANSFER'],
            ]);
            
            if ($response->successful()) {
                $data = $response->json();
                
                // Store transaction in database
                Transaction::create([
                    'user_id' => Auth::id(),
                    'reference' => $reference,
                    'amount' => $amount,
                    'status' => 'pending',
                    'type' => 'credit',
                    'gateway' => 'monnify',
                    'metadata' => json_encode($data),
                ]);
                
                // Store reference in session
                session()->put('transaction_reference', $reference);
                
                // Redirect to Momnify checkout
                return redirect($data['responseBody']['checkoutUrl']);
            }
            
            return back()->with('error', 'Failed to initialize payment. Please try again.');
            
        } catch (\Exception $e) {
            return back()->with('error', 'Payment initialization failed: ' . $e->getMessage());
        }
    }
    
    // Handle Payment Callback
    public function handleCallback(Request $request)
    {
        $paymentReference = $request->query('paymentReference');
        $transactionReference = $request->query('transactionReference');
        
        if (!$paymentReference) {
            return redirect()->route('wallet.history')->with('error', 'Invalid payment reference');
        }
        
        try {
            // Verify the transaction
            $transaction = $this->verifyTransaction($paymentReference);
            
            if ($transaction && $transaction['paymentStatus'] === 'PAID') {
                // Update transaction in database
                $dbTransaction = Transaction::where('reference', $paymentReference)->first();
                
                if ($dbTransaction) {
                    $dbTransaction->update([
                        'status' => 'completed',
                        'gateway_reference' => $transactionReference,
                        'metadata' => json_encode($transaction),
                    ]);
                    
                    // Credit user's wallet
                    $user = User::find($dbTransaction->user_id);
                    $user->wallet()->increment('balance', $transaction['amountPaid']);
                    
                    // Clear session
                    session()->forget('transaction_reference');
                    
                    return redirect()->route('wallet.history')->with('success', 'Payment successful! Your wallet has been credited with ₦' . number_format($transaction['amountPaid'], 2));
                }
            }
            
            return redirect()->route('wallet.history')->with('error', 'Payment verification failed or payment not completed');
            
        } catch (\Exception $e) {
            return redirect()->route('wallet.history')->with('error', 'Payment verification error: ' . $e->getMessage());
        }
    }
    
    // Verify Transaction
    private function verifyTransaction($paymentReference)
    {
        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $this->getAccessToken(),
        ])->get("https://api.monnify.com/api/v2/transactions/{$paymentReference}");
        
        if ($response->successful()) {
            return $response->json()['responseBody'];
        }
        
        return null;
    }
    
    // Webhook Handler (for background notifications)
    public function handleWebhook(Request $request)
    {
        // Verify webhook signature
        $signature = $request->header('monnify-signature');
        $payload = $request->getContent();
        
        if (!$this->verifyWebhookSignature($signature, $payload)) {
            return response()->json(['error' => 'Invalid signature'], 401);
        }
        
        $event = $request->input('eventType');
        $data = $request->input('eventData');
        
        switch ($event) {
            case 'SUCCESSFUL_TRANSACTION':
                $this->processSuccessfulTransaction($data);
                break;
                
            case 'FAILED_TRANSACTION':
                $this->processFailedTransaction($data);
                break;
        }
        
        return response()->json(['status' => 'success']);
    }
    
    // Verify webhook signature
    private function verifyWebhookSignature($signature, $payload)
    {
        $secretKey = config('services.monnify.webhook_secret');
        $computedSignature = hash_hmac('sha512', $payload, $secretKey);
        
        return hash_equals($signature, $computedSignature);
    }
    
    // Process successful transaction from webhook
    private function processSuccessfulTransaction($data)
    {
        $transaction = Transaction::where('reference', $data['paymentReference'])->first();
        
        if ($transaction && $transaction->status === 'pending') {
            $transaction->update([
                'status' => 'completed',
                'gateway_reference' => $data['transactionReference'],
                'metadata' => json_encode($data),
            ]);
            
            // Credit user's wallet
            $user = User::find($transaction->user_id);
            $user->wallet()->increment('balance', $data['amountPaid']);
        }
    }
    
    // Process failed transaction from webhook
    private function processFailedTransaction($data)
    {
        Transaction::where('reference', $data['paymentReference'])
            ->update(['status' => 'failed']);
    }
    
    // Payment Status Check (for AJAX polling)
    public function checkStatus($reference)
    {
        try {
            $transaction = Transaction::where('reference', $reference)->first();
            
            if (!$transaction) {
                return response()->json(['status' => 'not_found']);
            }
            
            return response()->json([
                'status' => $transaction->status,
                'amount' => $transaction->amount,
            ]);
            
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }
}