<?php

namespace App\Http\Controllers;

use App\Models\Transaction;
use App\Models\Wallet;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class HistoryController extends Controller
{
    /**
     * Display wallet transaction history page
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        $query = $this->buildTransactionQuery($user, $request);

        return Inertia::render('Wallet/History', [
            'transactions' => $query->paginate($request->per_page ?? 20),
            'wallet' => $user->wallet,
            'user' => $user,
            'summary' => $this->getSummaryStatistics($user),
            'filters' => $request->only(['type', 'status', 'date_from', 'date_to', 'per_page'])
        ]);
    }

    /**
     * Get transaction history for API
     */
    public function getHistory(Request $request)
    {
        $user = Auth::user();
        $query = $this->buildTransactionQuery($user, $request);

        return response()->json([
            'transactions' => $query->paginate($request->per_page ?? 20),
            'wallet' => $user->wallet,
            'summary' => $this->getSummaryStatistics($user)
        ]);
    }

    /**
     * Get detailed transaction view
     */
    public function show($id)
    {
        $transaction = $this->getUserTransaction($id);

        return Inertia::render('Wallet/TransactionDetail', [
            'transaction' => $transaction,
            'wallet' => Auth::user()->wallet,
            'user' => Auth::user()
        ]);
    }

    /**
     * Get transaction details for API
     */
    public function getTransaction($id)
    {
        return response()->json([
            'transaction' => $this->getUserTransaction($id)
        ]);
    }

    /**
     * Export transactions to CSV
     */
    public function export(Request $request)
    {
        $user = Auth::user();
        $transactions = $this->buildTransactionQuery($user, $request)->get();

        $csvData = $this->formatCsvData($transactions);
        $filename = 'transactions-' . $user->id . '-' . date('Y-m-d-H-i') . '.csv';

        return response()->streamDownload(
            function () use ($csvData) {
                $this->generateCsvOutput($csvData);
            },
            $filename,
            ['Content-Type' => 'text/csv']
        );
    }

    /**
     * Search transactions
     */
    public function search(Request $request)
    {
        $user = Auth::user();
        $query = $request->input('query');
        $field = $request->input('field', 'reference');

        $searchQuery = Transaction::where('user_id', $user->id);

        if ($field === 'reference') {
            $searchQuery->where('reference', 'like', "%{$query}%");
        } elseif ($field === 'payment_reference') {
            $searchQuery->where('payment_reference', 'like', "%{$query}%");
        } elseif ($field === 'transaction_type') {
            $searchQuery->where('transaction_type', 'like', "%{$query}%");
        } elseif ($field === 'status') {
            $searchQuery->where('status', 'like', "%{$query}%");
        } elseif ($field === 'payment_method') {
            $searchQuery->where('payment_method', 'like', "%{$query}%");
        } else {
            $searchQuery->where(function ($q) use ($query) {
                $q->where('reference', 'like', "%{$query}%")
                    ->orWhere('payment_reference', 'like', "%{$query}%")
                    ->orWhere('transaction_type', 'like', "%{$query}%")
                    ->orWhere('status', 'like', "%{$query}%");
            });
        }

        $transactions = $searchQuery->orderBy('created_at', 'desc')->paginate(20);

        return response()->json([
            'transactions' => $transactions,
            'search_query' => $query,
            'search_field' => $field
        ]);
    }

    /**
     * Get transaction statistics for charts
     */
    public function statistics(Request $request)
    {
        $user = Auth::user();
        $period = $request->input('period', 'month');

        $data = $this->generateStatisticsData($user, $period);

        return response()->json([
            'period' => $period,
            'data' => $data,
            'summary' => $this->calculateStatisticsSummary($data)
        ]);
    }

    /**
     * Get transaction types count
     */
    public function transactionTypes()
    {
        $user = Auth::user();

        $types = Transaction::where('user_id', $user->id)
            ->selectRaw('transaction_type as type, COUNT(*) as count, SUM(amount) as total_amount')
            ->where('status', 'success')
            ->groupBy('transaction_type')
            ->get();

        return response()->json(['types' => $types]);
    }

    /**
     * Get recent transactions for dashboard widget
     */
    public function recent(Request $request)
    {
        $limit = $request->input('limit', 10);

        $transactions = Transaction::where('user_id', Auth::id())
            ->orderBy('created_at', 'desc')
            ->limit($limit)
            ->get();

        return response()->json(['transactions' => $transactions]);
    }

    /**
     * ============================================
     * PRIVATE HELPER METHODS
     * ============================================
     */

    /**
     * Build base transaction query with filters
     */
    private function buildTransactionQuery($user, Request $request)
    {
        $query = Transaction::where('user_id', $user->id)
            ->with(['wallet', 'product'])
            ->orderBy('created_at', 'desc');

        $this->applyCommonFilters($query, $request);

        return $query;
    }

    /**
     * Apply common filters to query
     */
    private function applyCommonFilters($query, Request $request)
    {
        if ($request->has('type') && $request->type !== 'all') {
            $query->where('transaction_type', $request->type);
        }

        if ($request->has('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        if ($request->has('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->has('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }
    }

    /**
     * Get user's transaction by ID with authorization
     */
    private function getUserTransaction($id)
    {
        return Transaction::where('user_id', Auth::id())
            ->with(['wallet', 'product'])
            ->findOrFail($id);
    }

    /**
     * Get summary statistics
     */
    private function getSummaryStatistics($user)
    {
        return [
            'current_balance' => $user->wallet->balance ?? 0,

            // Wallet funding (deposits)
            'total_deposits' => Transaction::where('user_id', $user->id)
                ->where('transaction_type', 'wallet_funding')
                ->where('status', 'success')
                ->sum('amount'),

            // Wallet withdrawals
            'total_withdrawals' => Transaction::where('user_id', $user->id)
                ->where('transaction_type', 'wallet_withdrawal')
                ->where('status', 'success')
                ->sum('amount'),

            // Commissions (bonuses)
            'total_bonus' => Transaction::where('user_id', $user->id)
                ->where('transaction_type', 'commission')
                ->where('status', 'success')
                ->sum('amount'),

            // Total purchases (airtime + data)
            'total_purchases' => Transaction::where('user_id', $user->id)
                ->whereIn('transaction_type', ['airtime_purchase', 'data_purchase'])
                ->where('status', 'success')
                ->sum('amount'),

            // Airtime purchases only
            'total_airtime' => Transaction::where('user_id', $user->id)
                ->where('transaction_type', 'airtime_purchase')
                ->where('status', 'success')
                ->sum('amount'),

            // Data purchases only
            'total_data' => Transaction::where('user_id', $user->id)
                ->where('transaction_type', 'data_purchase')
                ->where('status', 'success')
                ->sum('amount'),

            // Refunds
            'total_refunds' => Transaction::where('user_id', $user->id)
                ->where('transaction_type', 'refund')
                ->where('status', 'success')
                ->sum('amount'),

            // Pending transactions
            'pending_deposits' => Transaction::where('user_id', $user->id)
                ->where('transaction_type', 'wallet_funding')
                ->where('status', 'pending')
                ->sum('amount'),

            'pending_withdrawals' => Transaction::where('user_id', $user->id)
                ->where('transaction_type', 'wallet_withdrawal')
                ->where('status', 'pending')
                ->sum('amount'),

            // Failed transactions
            'failed_transactions' => Transaction::where('user_id', $user->id)
                ->where('status', 'failed')
                ->count(),

            // Today's successful transactions
            'today_transactions' => Transaction::where('user_id', $user->id)
                ->whereDate('created_at', today())
                ->where('status', 'success')
                ->count(),

            // Weekly total amount
            'weekly_transactions' => Transaction::where('user_id', $user->id)
                ->whereBetween('created_at', [now()->startOfWeek(), now()->endOfWeek()])
                ->where('status', 'success')
                ->sum('amount'),
        ];
    }

    /**
     * Format CSV data
     */
    private function formatCsvData($transactions)
    {
        $csvData = [];
        $csvData[] = ['VTU Service - Transaction History', date('F j, Y')];
        $csvData[] = []; // Empty row
        $csvData[] = [
            'Date',
            'Reference',
            'Payment Reference',
            'Transaction Type',
            'Amount (₦)',
            'Status',
            'Payment Method',
            'Details',
            'Created At'
        ];

        foreach ($transactions as $transaction) {
            // Parse details if it's JSON
            $details = $transaction->details;
            if (is_string($details) && json_decode($details)) {
                $details = json_decode($details, true);
                $details = is_array($details) ? json_encode($details, JSON_PRETTY_PRINT) : $details;
            }

            $csvData[] = [
                $transaction->created_at->format('Y-m-d H:i:s'),
                $transaction->reference,
                $transaction->payment_reference ?? 'N/A',
                $this->formatTransactionType($transaction->transaction_type),
                number_format($transaction->amount / 100, 2), // Assuming amount is in kobo
                ucfirst($transaction->status),
                $this->formatPaymentMethod($transaction->payment_method),
                $details ?? 'N/A',
                $transaction->created_at->format('Y-m-d H:i:s')
            ];
        }

        return $csvData;
    }

    /**
     * Format transaction type for display
     */
    private function formatTransactionType($type)
    {
        $formatted = str_replace('_', ' ', $type);
        return ucwords($formatted);
    }

    /**
     * Format payment method for display
     */
    private function formatPaymentMethod($method)
    {
        if (!$method) return 'N/A';

        $formatted = str_replace('_', ' ', $method);
        return ucwords($formatted);
    }

    /**
     * Generate CSV output
     */
    private function generateCsvOutput($csvData)
    {
        $file = fopen('php://output', 'w');
        foreach ($csvData as $row) {
            fputcsv($file, $row);
        }
        fclose($file);
    }

    /**
     * Generate statistics data based on period
     */
    private function generateStatisticsData($user, $period)
    {
        $data = [];

        switch ($period) {
            case 'day':
                $data = $this->generateHourlyData($user);
                break;
            case 'week':
                $data = $this->generateDailyData($user, 7);
                break;
            case 'month':
                $data = $this->generateDailyData($user, 30);
                break;
            case 'year':
                $data = $this->generateMonthlyData($user, 12);
                break;
        }

        return $data;
    }

    /**
     * Generate hourly data for last 24 hours
     */
    private function generateHourlyData($user)
    {
        $data = [];

        for ($i = 23; $i >= 0; $i--) {
            $hour = now()->subHours($i);
            $nextHour = $hour->copy()->addHour();

            $funding = $this->sumTransactionsBetween($user, 'wallet_funding', $hour, $nextHour);
            $withdrawals = $this->sumTransactionsBetween($user, 'wallet_withdrawal', $hour, $nextHour);
            $purchases = $this->sumTransactionsBetweenTypes($user, ['airtime_purchase', 'data_purchase'], $hour, $nextHour);

            $data[] = [
                'label' => $hour->format('H:00'),
                'funding' => $funding,
                'withdrawals' => $withdrawals,
                'purchases' => $purchases,
                'total' => $funding - $withdrawals - $purchases
            ];
        }

        return $data;
    }

    /**
     * Generate daily data for last N days
     */
    private function generateDailyData($user, $days)
    {
        $data = [];

        for ($i = $days - 1; $i >= 0; $i--) {
            $date = now()->subDays($i);

            $funding = $this->sumTransactionsOnDate($user, 'wallet_funding', $date);
            $withdrawals = $this->sumTransactionsOnDate($user, 'wallet_withdrawal', $date);
            $purchases = $this->sumTransactionsOnDateTypes($user, ['airtime_purchase', 'data_purchase'], $date);

            $data[] = [
                'label' => $date->format($days <= 14 ? 'D' : 'd M'),
                'date' => $date->format('Y-m-d'),
                'funding' => $funding,
                'withdrawals' => $withdrawals,
                'purchases' => $purchases,
                'total' => $funding - $withdrawals - $purchases
            ];
        }

        return $data;
    }

    /**
     * Generate monthly data for last N months
     */
    private function generateMonthlyData($user, $months)
    {
        $data = [];

        for ($i = $months - 1; $i >= 0; $i--) {
            $date = now()->subMonths($i);

            $funding = $this->sumTransactionsInMonth($user, 'wallet_funding', $date);
            $withdrawals = $this->sumTransactionsInMonth($user, 'wallet_withdrawal', $date);
            $purchases = $this->sumTransactionsInMonthTypes($user, ['airtime_purchase', 'data_purchase'], $date);

            $data[] = [
                'label' => $date->format('M Y'),
                'month' => $date->format('Y-m'),
                'funding' => $funding,
                'withdrawals' => $withdrawals,
                'purchases' => $purchases,
                'total' => $funding - $withdrawals - $purchases
            ];
        }

        return $data;
    }

    /**
     * Helper to sum transactions between dates
     */
    private function sumTransactionsBetween($user, $type, $start, $end)
    {
        return Transaction::where('user_id', $user->id)
            ->where('transaction_type', $type)
            ->where('status', 'success')
            ->whereBetween('created_at', [$start, $end])
            ->sum('amount');
    }

    /**
     * Helper to sum multiple transaction types between dates
     */
    private function sumTransactionsBetweenTypes($user, $types, $start, $end)
    {
        return Transaction::where('user_id', $user->id)
            ->whereIn('transaction_type', $types)
            ->where('status', 'success')
            ->whereBetween('created_at', [$start, $end])
            ->sum('amount');
    }

    /**
     * Helper to sum transactions on specific date
     */
    private function sumTransactionsOnDate($user, $type, $date)
    {
        return Transaction::where('user_id', $user->id)
            ->where('transaction_type', $type)
            ->where('status', 'success')
            ->whereDate('created_at', $date)
            ->sum('amount');
    }

    /**
     * Helper to sum multiple transaction types on specific date
     */
    private function sumTransactionsOnDateTypes($user, $types, $date)
    {
        return Transaction::where('user_id', $user->id)
            ->whereIn('transaction_type', $types)
            ->where('status', 'success')
            ->whereDate('created_at', $date)
            ->sum('amount');
    }

    /**
     * Helper to sum transactions in specific month
     */
    private function sumTransactionsInMonth($user, $type, $date)
    {
        return Transaction::where('user_id', $user->id)
            ->where('transaction_type', $type)
            ->where('status', 'success')
            ->whereYear('created_at', $date->year)
            ->whereMonth('created_at', $date->month)
            ->sum('amount');
    }

    /**
     * Helper to sum multiple transaction types in specific month
     */
    private function sumTransactionsInMonthTypes($user, $types, $date)
    {
        return Transaction::where('user_id', $user->id)
            ->whereIn('transaction_type', $types)
            ->where('status', 'success')
            ->whereYear('created_at', $date->year)
            ->whereMonth('created_at', $date->month)
            ->sum('amount');
    }

    /**
     * Calculate statistics summary
     */
    private function calculateStatisticsSummary($data)
    {
        return [
            'total_funding' => array_sum(array_column($data, 'funding')),
            'total_withdrawals' => array_sum(array_column($data, 'withdrawals')),
            'total_purchases' => array_sum(array_column($data, 'purchases')),
            'net_flow' => array_sum(array_column($data, 'total'))
        ];
    }
}
