<?php

namespace App\Http\Controllers;

use App\Services\ClubKonnectService;
use App\Models\Transaction;
use App\Models\Product;
use App\Models\Wallet;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Inertia\Inertia;

class DataController extends Controller
{
    protected $clubKonnect;

    public function __construct(ClubKonnectService $clubKonnect)
    {
        $this->clubKonnect = $clubKonnect;
    }

    public function index()
    {
        // Service list in order
        $Service = $this->getNetworks();

        // Get authenticated user
        $user = auth()->user();

        // Ensure wallet exists or create it
        $wallet = $user->wallet()->first();

        if (!$wallet) {
            $wallet = $user->wallet()->create(['balance' => 0]);
        }

        return Inertia::render('Products/BuyData', [
            'networks' => $networks,
            'wallet' => [
                'balance' => $wallet->balance,
            ]
        ]);
    }

    public function getPlans($network)
    {
        try {
            Log::info("Fetching plans for network: {$network}");

            $plans = Product::where('type', 'data')
                ->where('network_code', $network)
                ->where('is_active', true)
                ->where('is_available', true)
                ->orderBy('price')
                ->get()
                ->map(function ($product) {
                    return [
                        'id' => $product->provider_plan_id,
                        'description' => $product->description,
                        'price' => (float) $product->price,
                        'display_price' => '₦' . number_format($product->price, 2),
                        'validity' => $product->validity ?? '',
                        'data_size' => $product->data_size ?? '',
                    ];
                });

            Log::info("Found {$plans->count()} plans for network {$network}");

            return response()->json([
                'success' => true,
                'plans' => $plans
            ]);
        } catch (\Exception $e) {
            Log::error("Error fetching plans for network {$network}: " . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to load plans',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    private function getNetworkName($code): string
    {
        $networks = [
            '01' => 'MTN',
            '02' => 'GLO',
            '04' => 'Airtel',
            '03' => '9mobile',
        ];

        return $networks[$code] ?? 'Unknown Network';
    }

    public function purchase(Request $request)
    {
        $request->validate([
            'network' => 'required|string|in:01,02,03,04',
            'plan_id' => 'required|string',
            'phone_numbers' => 'required|string',
            'bypass_validator' => 'boolean',
        ]);

        Log::info('Data purchase request:', $request->all());

        // Parse phone numbers
        $phoneNumbers = $this->parsePhoneNumbers($request->phone_numbers);

        if (empty($phoneNumbers)) {
            return response()->json([
                'success' => false,
                'message' => 'Please enter valid phone numbers'
            ], 422);
        }

        // Get plan details
        $plan = Product::where('type', 'data')
            ->where('provider_plan_id', $request->plan_id)
            ->where('network_code', $request->network)
            ->where('is_active', true)
            ->first();

        if (!$plan) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid data plan selected'
            ], 422);
        }

        // Check user's wallet balance
        $user = $request->user();
        $totalAmount = $plan->price * count($phoneNumbers);

        $wallet = $user->wallet()->first();
        if (!$wallet) {
            return response()->json([
                'success' => false,
                'message' => 'Wallet not found for user'
            ], 422);
        }

        $walletBalance = $wallet->balance;

        if ($walletBalance < $totalAmount) {
            return response()->json([
                'success' => false,
                'message' => 'Insufficient balance. Required: ₦' .
                    number_format($totalAmount, 2) .
                    ', Available: ₦' . number_format($walletBalance, 2)
            ], 422);
        }

        // ============ CRITICAL: Check ClubKonnect balance BEFORE any processing ============
        try {
            // This will throw exception if ClubKonnect balance is insufficient
            $this->clubKonnect->ensureSufficientBalance($totalAmount);

            Log::info('ClubKonnect balance check PASSED - proceeding with purchase', [
                'required_amount' => $totalAmount
            ]);
        } catch (\Exception $e) {
            // ClubKonnect has insufficient balance - BLOCK ALL TRANSACTIONS
            Log::error('❌ PURCHASE BLOCKED - ClubKonnect insufficient balance: ' . $e->getMessage(), [
                'user_id' => $user->id,
                'required_amount' => $totalAmount,
                'phone_count' => count($phoneNumbers)
            ]);

            // Return ONLY generic message to user
            return response()->json([
                'success' => false,
                'message' => 'Service unavailable, try again later'
            ], 422);
        }
        // ============ END CRITICAL CHECK ============

        // If we reach here, ClubKonnect has sufficient balance
        // Now process each phone number
        $networkName = $this->getNetworkName($request->network);
        $results = [];
        $successful = 0;
        $failed = 0;

        foreach ($phoneNumbers as $phone) {
            // Validate phone number (unless bypassed)
            if (!$request->bypass_validator && !$this->validatePhoneNumber($phone)) {
                $results[] = [
                    'phone' => $phone,
                    'success' => false,
                    'message' => 'Invalid phone number format'
                ];
                $failed++;
                continue;
            }

            // Generate unique request ID
            $requestId = 'DATA' . time() . rand(1000, 9999);

            try {
                // Call ClubKonnect API
                $apiResult = $this->clubKonnect->buyDataBundle(
                    $request->network,
                    $request->plan_id,
                    '0' . $phone,
                    $requestId
                );

                Log::info('ClubKonnect API response:', $apiResult);

                // Check if API call was successful
                if (!($apiResult['success'] ?? false)) {
                    $errorMessage = $apiResult['message'] ?? 'API call failed';
                    throw new \Exception($errorMessage);
                }

                // Get the actual API response data
                $result = $apiResult['data'] ?? $apiResult;

                // Get original status from API
                $originalStatus = $result['status'] ?? ($result['orderstatus'] ?? 'pending');
                $normalizedStatus = $this->normalizeTransactionStatus($originalStatus);

                // Check if transaction was successful
                $isSuccessful = $normalizedStatus === 'success';

                if ($isSuccessful) {
                    // Deduct from user wallet
                    $wallet->decrement('balance', $plan->price);

                    // Record successful transaction
                    Transaction::create([
                        'user_id' => $user->id,
                        'reference' => $requestId,
                        'type' => 'data',
                        'status' => $normalizedStatus,
                        'original_status' => $originalStatus,
                        'amount' => $plan->price,
                        'amount_charged' => $plan->price,
                        'network' => $networkName,
                        'phone' => $phone,
                        'plan_code' => $request->plan_id,
                        'plan_description' => $plan->description,
                        'order_id' => $result['orderid'] ?? $result['order_id'] ?? null,
                        'request_id' => $requestId,
                        'api_response' => json_encode($result),
                        'remark' => $result['orderremark'] ?? $result['message'] ?? null,
                    ]);

                    $results[] = [
                        'phone' => $phone,
                        'success' => true,
                        'message' => 'Data bundle purchased successfully',
                        'order_id' => $result['orderid'] ?? $result['order_id'] ?? null,
                        'status' => $normalizedStatus
                    ];
                    $successful++;
                } else {
                    // Transaction failed - don't deduct money
                    Transaction::create([
                        'user_id' => $user->id,
                        'reference' => $requestId,
                        'type' => 'data',
                        'status' => $normalizedStatus,
                        'original_status' => $originalStatus,
                        'amount' => $plan->price,
                        'amount_charged' => $plan->price,
                        'network' => $networkName,
                        'phone' => $phone,
                        'plan_code' => $request->plan_id,
                        'plan_description' => $plan->description,
                        'order_id' => $result['orderid'] ?? $result['order_id'] ?? null,
                        'request_id' => $requestId,
                        'api_response' => json_encode($result),
                        'remark' => $result['orderremark'] ?? $result['message'] ?? null,
                    ]);

                    // Show generic message for failed transactions
                    $results[] = [
                        'phone' => $phone,
                        'success' => false,
                        'message' => 'Service unavailable, try again later',
                        'order_id' => $result['orderid'] ?? $result['order_id'] ?? null,
                        'status' => $normalizedStatus
                    ];
                    $failed++;
                }
            } catch (\Exception $e) {
                Log::error('Error processing phone ' . $phone . ': ' . $e->getMessage());

                // Record error transaction
                Transaction::create([
                    'user_id' => $user->id,
                    'reference' => $requestId,
                    'type' => 'data',
                    'status' => 'failed',
                    'original_status' => 'error',
                    'amount' => $plan->price,
                    'amount_charged' => $plan->price,
                    'network' => $networkName,
                    'phone' => $phone,
                    'plan_code' => $request->plan_id,
                    'plan_description' => $plan->description,
                    'order_id' => null,
                    'request_id' => $requestId,
                    'api_response' => json_encode(['error' => $e->getMessage()]),
                    'remark' => 'API Error: ' . $e->getMessage(),
                ]);

                $results[] = [
                    'phone' => $phone,
                    'success' => false,
                    'message' => 'Service unavailable, try again later',
                    'status' => 'failed'
                ];
                $failed++;
            }
        }

        // Get updated wallet balance
        $wallet->refresh();

        return response()->json([
            'success' => true,
            'message' => 'Data purchase processed',
            'successful' => $successful,
            'failed' => $failed,
            'results' => $results,
            'wallet_balance' => $wallet->balance
        ]);
    }

    // This method is for pre-check (optional)
    public function checkVendorBalance(Request $request)
    {
        try {
            $requiredAmount = $request->input('amount', 0);

            if ($requiredAmount <= 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Service unavailable, try again later'
                ], 422);
            }

            // Check ClubKonnect balance
            $this->clubKonnect->ensureSufficientBalance($requiredAmount);

            return response()->json([
                'success' => true,
                'message' => 'Service available'
            ]);
        } catch (\Exception $e) {
            Log::error('Vendor balance check failed: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Service unavailable, try again later'
            ], 422);
        }
    }

    private function normalizeTransactionStatus($status)
    {
        $status = strtoupper(trim($status));

        if (in_array($status, ['ORDER_RECEIVED', 'SUCCESSFUL', 'SUCCESS', 'COMPLETED', '100', '200'])) {
            return 'success';
        }

        if (in_array($status, ['PENDING', 'PROCESSING', 'IN_PROGRESS'])) {
            return 'pending';
        }

        if (in_array($status, ['FAILED', 'INSUFFICIENT_BALANCE', 'NETWORK_ERROR', 'ERROR', 'CANCELLED', 'DECLINED', '400', '500'])) {
            return 'failed';
        }

        return 'pending';
    }

    private function parsePhoneNumbers($phoneNumbers)
    {
        $phoneNumbers = preg_replace('/\s+/', '', $phoneNumbers);
        $numbers = preg_split('/[,|\n]/', $phoneNumbers, -1, PREG_SPLIT_NO_EMPTY);

        $cleanedNumbers = [];
        foreach ($numbers as $number) {
            $cleanNumber = preg_replace('/[^\d]/', '', $number);

            if (str_starts_with($cleanNumber, '0')) {
                $cleanNumber = substr($cleanNumber, 1);
            }

            if (str_starts_with($cleanNumber, '234')) {
                $cleanNumber = substr($cleanNumber, 3);
            }

            if (strlen($cleanNumber) === 10) {
                $cleanedNumbers[] = $cleanNumber;
            } else {
                Log::warning('Invalid phone number format: ' . $number . ' -> ' . $cleanNumber);
            }
        }

        return $cleanedNumbers;
    }

    private function validatePhoneNumber($phone)
    {
        return preg_match('/^(70|80|81|90|91)\d{8}$/', $phone);
    }

    public function callback(Request $request)
    {
        $orderId = $request->get('orderid');
        $statusCode = $request->get('statuscode');
        $orderStatus = $request->get('orderstatus');
        $orderRemark = $request->get('orderremark');

        Log::info('ClubKonnect Data Callback', $request->all());

        $transaction = Transaction::where('order_id', $orderId)->first();

        if ($transaction) {
            $normalizedStatus = $this->normalizeTransactionStatus($orderStatus);

            $transaction->update([
                'status' => $normalizedStatus,
                'original_status' => $orderStatus,
                'remark' => $orderRemark,
                'completed_at' => now(),
            ]);

            if ($normalizedStatus === 'failed') {
                $wallet = Wallet::where('user_id', $transaction->user_id)->first();
                if ($wallet) {
                    $wallet->increment('balance', $transaction->amount);
                    Log::info('Refund processed for failed transaction', [
                        'order_id' => $orderId,
                        'amount' => $transaction->amount,
                        'wallet_id' => $wallet->id
                    ]);
                }
            }
        }

        return response()->json(['status' => 'ok']);
    }

    private function getNetworks()
    {
        return [
            ['code' => '01', 'name' => 'MTN'],
            ['code' => '04', 'name' => 'Airtel'],
            ['code' => '02', 'name' => 'Glo'],
            ['code' => '03', 'name' => '9mobile'],
        ];
    }
}
