<?php

namespace App\Http\Controllers;

use App\Services\ClubKonnectService;
use App\Models\Transaction;
use App\Models\Product;
use App\Models\Wallet;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Inertia\Inertia;

class CableTVController extends Controller
{
    protected $clubKonnect;

    public function __construct(ClubKonnectService $clubKonnect)
    {
        $this->clubKonnect = $clubKonnect;
    }

    public function index()
    {
        $providers = $this->getCableProviders();

        $user = auth()->user();
        $wallet = $user->wallet()->first();

        if (!$wallet) {
            $wallet = $user->wallet()->create(['balance' => 0]);
        }

        return Inertia::render('Products/BuyCableTV', [
            'providers' => $providers,
            'wallet' => [
                'balance' => $wallet->balance,
            ]
        ]);
    }

    public function getPackages($provider)
    {
        try {
            Log::info("Fetching packages for cable provider: {$provider}");

            $packages = Product::where('type', 'cable')
                ->where('network_code', $provider)
                ->where('is_active', true)
                ->where('is_available', true)
                ->orderBy('price')
                ->get()
                ->map(function ($product) {
                    return [
                        'id' => $product->id,
                        'package_code' => $product->provider_plan_id,
                        'name' => $product->name,
                        'price' => (float) $product->price,
                        'display_price' => '₦' . number_format($product->price, 2),
                        'description' => $product->description,
                        'validity' => $product->validity ?? '',
                    ];
                });

            Log::info("Found {$packages->count()} packages for provider {$provider}");

            return response()->json([
                'success' => true,
                'packages' => $packages
            ]);
        } catch (\Exception $e) {
            Log::error("Error fetching packages for provider {$provider}: " . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to load packages'
            ], 500);
        }
    }

    public function verifySmartcard(Request $request)
    {
        $request->validate([
            'provider' => 'required|string|in:dstv,gotv,startimes,showmax',
            'smartcard_number' => 'required|string|min:10|max:20',
        ]);

        try {
            $smartcardNumber = trim($request->smartcard_number);

            Log::info('Verifying smartcard:', [
                'provider' => $request->provider,
                'smartcard' => substr($smartcardNumber, 0, 6) . '...'
            ]);

            // Call ClubKonnect verification API
            $result = $this->clubKonnect->verifyCableTV(
                $request->provider,
                $smartcardNumber
            );

            Log::info('Verification response:', $result);

            if (isset($result['success']) && $result['success'] === false) {
                return response()->json([
                    'success' => false,
                    'message' => $result['message'] ?? 'Verification failed'
                ]);
            }

            // Check for customer name in response
            $customerName = $result['customer_name'] ?? null;

            if ($customerName && $customerName !== 'INVALID_SMARTCARDNO') {
                return response()->json([
                    'success' => true,
                    'info' => [
                        'customer_name' => $customerName,
                        'status' => 'success',
                        'message' => 'Smartcard verified successfully'
                    ]
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid smartcard number'
                ]);
            }
        } catch (\Exception $e) {
            Log::error('Smartcard verification error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Verification service unavailable'
            ], 500);
        }
    }

    public function purchase(Request $request)
    {
        $request->validate([
            'provider' => 'required|string|in:dstv,gotv,startimes,showmax',
            'package_code' => 'required|string',
            'smartcard_number' => 'required|string|min:10|max:20',
            'phone_number' => 'required|string|regex:/^0[7-9][0-1]\d{8}$/',
            'request_id' => 'nullable|string',
            'callback_url' => 'nullable|url',
        ]);

        Log::info('Cable TV purchase request:', [
            'provider' => $request->provider,
            'package_code' => $request->package_code,
            'smartcard' => substr($request->smartcard_number, 0, 6) . '...',
            'phone' => substr($request->phone_number, 0, 6) . '...'
        ]);

        // Get package details
        $package = Product::where('type', 'cable')
            ->where('provider_plan_id', $request->package_code)
            ->where('network_code', $request->provider)
            ->where('is_active', true)
            ->first();

        if (!$package) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid package selected'
            ], 422);
        }

        // Check user's wallet balance
        $user = $request->user();
        $totalAmount = $package->price;

        $wallet = $user->wallet()->first();
        if (!$wallet) {
            return response()->json([
                'success' => false,
                'message' => 'Wallet not found'
            ], 422);
        }

        if ($wallet->balance < $totalAmount) {
            return response()->json([
                'success' => false,
                'message' => 'Insufficient balance. Required: ₦' .
                    number_format($totalAmount, 2) .
                    ', Available: ₦' . number_format($wallet->balance, 2)
            ], 422);
        }

        // ============ CRITICAL: Check ClubKonnect balance ============
        try {
            $this->clubKonnect->ensureSufficientBalance($totalAmount);

            Log::info('ClubKonnect balance check PASSED - proceeding with cable purchase', [
                'required_amount' => $totalAmount
            ]);
        } catch (\Exception $e) {
            Log::error('❌ CABLE PURCHASE BLOCKED - ClubKonnect insufficient balance: ' . $e->getMessage(), [
                'user_id' => $user->id,
                'required_amount' => $totalAmount,
                'provider' => $request->provider
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Service unavailable, try again later'
            ], 422);
        }
        // ============ END CRITICAL CHECK ============

        // Generate request ID if not provided
        $requestId = $request->request_id ?: 'CABLE' . time() . rand(1000, 9999);

        try {
            // Call ClubKonnect API for cable subscription
            $apiResult = $this->clubKonnect->buyCableTV(
                $request->provider,
                $request->package_code,
                $request->smartcard_number,
                $request->phone_number,
                $requestId,
                $request->callback_url
            );

            Log::info('ClubKonnect Cable API response:', $apiResult);

            // Check if API call was successful
            if (!($apiResult['success'] ?? false)) {
                $errorMessage = $apiResult['message'] ?? 'API call failed';
                throw new \Exception($errorMessage);
            }

            // Get the actual API response data
            $result = $apiResult['data'] ?? $apiResult;

            // Get original status from API
            $originalStatus = $result['status'] ?? ($result['orderstatus'] ?? 'pending');
            $statusCode = $result['statuscode'] ?? '100';
            $normalizedStatus = $this->normalizeTransactionStatus($originalStatus, $statusCode);

            // Check if transaction was successful
            $isSuccessful = $normalizedStatus === 'success';

            if ($isSuccessful) {
                // Deduct from user wallet
                $wallet->decrement('balance', $totalAmount);

                // Record successful transaction
                Transaction::create([
                    'user_id' => $user->id,
                    'reference' => $requestId,
                    'type' => 'cable',
                    'status' => $normalizedStatus,
                    'original_status' => $originalStatus,
                    'amount' => $totalAmount,
                    'amount_charged' => $totalAmount,
                    'network' => strtoupper($request->provider),
                    'phone' => $request->phone_number,
                    'smartcard_number' => $request->smartcard_number,
                    'plan_code' => $request->package_code,
                    'plan_description' => $package->description,
                    'order_id' => $result['orderid'] ?? $result['order_id'] ?? null,
                    'request_id' => $requestId,
                    'api_response' => json_encode($result),
                    'remark' => $result['orderremark'] ?? $result['message'] ?? 'Cable subscription',
                ]);

                $wallet->refresh();

                return response()->json([
                    'success' => true,
                    'message' => 'Subscription successful',
                    'order_id' => $result['orderid'] ?? $result['order_id'] ?? null,
                    'status' => $normalizedStatus,
                    'wallet_balance' => $wallet->balance
                ]);
            } else {
                // Transaction failed - record but don't deduct money
                Transaction::create([
                    'user_id' => $user->id,
                    'reference' => $requestId,
                    'type' => 'cable',
                    'status' => $normalizedStatus,
                    'original_status' => $originalStatus,
                    'amount' => $totalAmount,
                    'amount_charged' => $totalAmount,
                    'network' => strtoupper($request->provider),
                    'phone' => $request->phone_number,
                    'smartcard_number' => $request->smartcard_number,
                    'plan_code' => $request->package_code,
                    'plan_description' => $package->description,
                    'order_id' => $result['orderid'] ?? $result['order_id'] ?? null,
                    'request_id' => $requestId,
                    'api_response' => json_encode($result),
                    'remark' => $result['orderremark'] ?? $result['message'] ?? 'Failed',
                ]);

                return response()->json([
                    'success' => false,
                    'message' => $result['orderremark'] ?? 'Subscription failed',
                    'order_id' => $result['orderid'] ?? $result['order_id'] ?? null,
                    'status' => $normalizedStatus,
                    'wallet_balance' => $wallet->balance
                ]);
            }
        } catch (\Exception $e) {
            Log::error('Cable subscription error: ' . $e->getMessage());

            // Record error transaction
            Transaction::create([
                'user_id' => $user->id,
                'reference' => $requestId,
                'type' => 'cable',
                'status' => 'failed',
                'original_status' => 'error',
                'amount' => $totalAmount,
                'amount_charged' => $totalAmount,
                'network' => strtoupper($request->provider),
                'phone' => $request->phone_number,
                'smartcard_number' => $request->smartcard_number,
                'plan_code' => $request->package_code,
                'plan_description' => $package->description,
                'order_id' => null,
                'request_id' => $requestId,
                'api_response' => json_encode(['error' => $e->getMessage()]),
                'remark' => 'API Error: ' . $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Service unavailable, try again later',
                'status' => 'failed'
            ], 500);
        }
    }

    private function normalizeTransactionStatus($status, $statusCode = null)
    {
        $status = strtoupper(trim($status));

        // Check status codes first
        if ($statusCode) {
            if (in_array($statusCode, ['100', '200'])) {
                return 'success';
            }
            if (in_array($statusCode, ['400', '500'])) {
                return 'failed';
            }
        }

        if (in_array($status, ['ORDER_RECEIVED', 'ORDER_COMPLETED', 'SUCCESSFUL', 'SUCCESS', 'COMPLETED'])) {
            return 'success';
        }

        if (in_array($status, ['PENDING', 'PROCESSING', 'IN_PROGRESS', 'ORDER_ONHOLD'])) {
            return 'pending';
        }

        if (in_array($status, ['FAILED', 'INSUFFICIENT_BALANCE', 'NETWORK_ERROR', 'ERROR', 'CANCELLED', 'DECLINED'])) {
            return 'failed';
        }

        return 'pending';
    }

    public function callback(Request $request)
    {
        $orderId = $request->get('orderid');
        $statusCode = $request->get('statuscode');
        $orderStatus = $request->get('orderstatus');
        $orderRemark = $request->get('orderremark');
        $requestId = $request->get('requestid') ?? $request->get('request_id');

        Log::info('ClubKonnect Cable Callback', $request->all());

        $transaction = Transaction::where('order_id', $orderId)
            ->orWhere('request_id', $requestId)
            ->first();

        if ($transaction) {
            $normalizedStatus = $this->normalizeTransactionStatus($orderStatus, $statusCode);

            $transaction->update([
                'status' => $normalizedStatus,
                'original_status' => $orderStatus,
                'remark' => $orderRemark,
                'completed_at' => now(),
            ]);

            // Refund if failed
            if ($normalizedStatus === 'failed') {
                $wallet = Wallet::where('user_id', $transaction->user_id)->first();
                if ($wallet) {
                    $wallet->increment('balance', $transaction->amount);
                    Log::info('Cable refund processed for failed transaction', [
                        'order_id' => $orderId,
                        'amount' => $transaction->amount,
                        'wallet_id' => $wallet->id
                    ]);
                }
            }
        }

        return response()->json(['status' => 'ok']);
    }

    private function getCableProviders()
    {
        return [
            ['code' => 'dstv', 'name' => 'DStv', 'display_name' => 'DStv'],
            ['code' => 'gotv', 'name' => 'GOtv', 'display_name' => 'GOtv'],
            ['code' => 'startimes', 'name' => 'StarTimes', 'display_name' => 'StarTimes'],
            ['code' => 'showmax', 'name' => 'Showmax', 'display_name' => 'Showmax'],
        ];
    }
}
