<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Http\Requests\Auth\LoginRequest;
use App\Models\UserMessage;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Route;
use Inertia\Inertia;
use Inertia\Response;

class AuthenticatedSessionController extends Controller
{
    /**
     * Display the login view.
     */
    public function create(): Response
    {
        return Inertia::render('Auth/Login', [
            'canResetPassword' => Route::has('password.request'),
            'status' => session('status'),
        ]);
    }

    /**
     * Handle an incoming authentication request.
     */
    public function store(LoginRequest $request): RedirectResponse
    {
        $request->authenticate();

        $request->session()->regenerate();

        $user = Auth::user();

        // Add login notification logic
        $this->handleLoginNotifications($user);

        return redirect()->intended(route('dashboard', absolute: false));
    }

    /**
     * Handle login notifications and messages
     */
    protected function handleLoginNotifications($user): void
    {
        // Add welcome message if needed
        $this->addWelcomeMessage($user);

        // Check for unread messages
        $unreadCount = $user->unreadMessages()->count();
        session(['unread_notifications_count' => $unreadCount]);

        // Check for system announcements
        $systemAnnouncements = $this->checkSystemAnnouncements($user);

        // Calculate total unread messages
        $totalUnread = $unreadCount + $systemAnnouncements;

        // Set session flags if there are notifications
        if ($totalUnread > 0) {
            session()->flash('show_notifications', true);
            session()->flash('has_unread_messages', true);

            // Store in session for middleware access
            session(['unread_notifications_count' => $totalUnread]);

            // If there are critical messages, set immediate flag
            $criticalMessages = $user->messages()
                ->unread()
                ->where('type', 'error')
                ->where(function ($query) {
                    $query->whereNull('expires_at')
                        ->orWhere('expires_at', '>', now());
                })
                ->exists();

            if ($criticalMessages) {
                session()->flash('show_critical_notifications', true);
            }
        }

        // Update last login timestamp
        $user->update(['last_login_at' => now()]);
    }

    /**
     * Add welcome message for user
     */
    protected function addWelcomeMessage($user): void
    {
        // Check if user has any previous welcome message today
        $hasWelcomeToday = $user->messages()
            ->where('title', 'like', '%Welcome%')
            ->whereDate('created_at', today())
            ->exists();

        if (!$hasWelcomeToday) {
            $message = $this->getWelcomeMessage($user);

            $user->messages()->create([
                'title' => $message['title'],
                'message' => $message['text'],
                'type' => 'success',
                'expires_at' => now()->addDays(3),
            ]);
        }
    }

    /**
     * Generate appropriate welcome message
     */
    protected function getWelcomeMessage($user): array
    {
        $lastLogin = $user->last_login_at;

        if (!$lastLogin) {
            return [
                'title' => '👋 Welcome to ' . config('app.name') . '!',
                'text' => "Welcome aboard! We're excited to have you here. Get started by exploring the dashboard."
            ];
        }

        $daysSinceLastLogin = $lastLogin->diffInDays(now());

        if ($daysSinceLastLogin > 30) {
            return [
                'title' => '🎉 Welcome Back!',
                'text' => "We've missed you! It's been $daysSinceLastLogin days since your last visit. Check out what's new!"
            ];
        }

        if ($daysSinceLastLogin > 7) {
            return [
                'title' => '👍 Welcome Back!',
                'text' => "Good to see you again! Your last login was " . $lastLogin->format('M j, Y') . "."
            ];
        }

        return [
            'title' => '👋 Welcome Back!',
            'text' => "Welcome back! Last login: " . $lastLogin->diffForHumans() . "."
        ];
    }

    /**
     * Check for system announcements and assign to user
     */
    protected function checkSystemAnnouncements($user): int
    {
        $count = 0;

        // Get active system announcements
        $systemAnnouncements = UserMessage::whereNull('user_id')
            ->where('expires_at', '>', now())
            ->whereDoesntHave('assignedUsers', function ($query) use ($user) {
                $query->where('user_id', $user->id);
            })
            ->get();

        foreach ($systemAnnouncements as $announcement) {
            // Check if similar message already exists for user
            $exists = $user->messages()
                ->where('title', $announcement->title)
                ->where('message', $announcement->message)
                ->exists();

            if (!$exists) {
                // Create a copy for this user
                $user->messages()->create([
                    'title' => $announcement->title,
                    'message' => $announcement->message,
                    'type' => $announcement->type,
                    'expires_at' => $announcement->expires_at,
                    'actions' => $announcement->actions,
                ]);

                $count++;
            }

            // Mark original as assigned to this user
            $announcement->assignedUsers()->syncWithoutDetaching([$user->id => ['is_read' => false]]);
        }

        return $count;
    }

    /**
     * Destroy an authenticated session.
     */
    public function destroy(Request $request): RedirectResponse
    {
        Auth::guard('web')->logout();

        $request->session()->invalidate();

        $request->session()->regenerateToken();

        return redirect('/');
    }
}
