<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\UserMessage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class NotificationController extends Controller
{
    /**
     * Get unread notifications for current user
     */
    public function unread(Request $request)
    {
        $user = Auth::user();

        // Get user-specific messages
        $userMessages = $user->messages()
            ->active()
            ->unread()
            ->orderBy('created_at', 'desc')
            ->get();

        // Get system announcements assigned to this user
        $systemMessages = UserMessage::whereNull('user_id')
            ->whereHas('users', function ($query) use ($user) {
                $query->where('user_id', $user->id)
                    ->where('is_read', false);
            })
            ->active()
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($message) use ($user) {
                // Convert system message to user message format
                return [
                    'id' => $message->id,
                    'title' => $message->title,
                    'message' => $message->message,
                    'type' => $message->type,
                    'is_read' => $message->users()->where('user_id', $user->id)->first()->pivot->is_read ?? false,
                    'created_at' => $message->created_at->toISOString(),
                    'expires_at' => $message->expires_at?->toISOString(),
                    'actions' => $message->actions,
                    'is_system' => true,
                ];
            });

        // Merge both collections
        $allMessages = $userMessages->merge($systemMessages)->sortByDesc('created_at');

        // Format response
        $notifications = $allMessages->take(10)->map(function ($message) {
            return [
                'id' => $message['id'] ?? $message->id,
                'title' => $message['title'] ?? $message->title,
                'message' => $message['message'] ?? $message->message,
                'type' => $message['type'] ?? $message->type,
                'is_read' => $message['is_read'] ?? $message->is_read,
                'created_at' => $message['created_at'] ?? $message->created_at->toISOString(),
                'expires_at' => $message['expires_at'] ?? $message->expires_at?->toISOString(),
                'actions' => $message['actions'] ?? $message->actions,
                'is_system' => $message['is_system'] ?? false,
            ];
        });

        // Count unread messages
        $unreadCount = $user->unreadMessages()->count() +
            UserMessage::whereNull('user_id')
            ->whereHas('users', function ($query) use ($user) {
                $query->where('user_id', $user->id)
                    ->where('is_read', false);
            })
            ->count();

        return response()->json([
            'notifications' => $notifications,
            'unread_count' => $unreadCount,
        ]);
    }

    /**
     * Mark a notification as read
     */
    public function markAsRead(Request $request, $messageId)
    {
        $user = Auth::user();

        // Check if it's a user-specific message
        $userMessage = $user->messages()->where('id', $messageId)->first();

        if ($userMessage) {
            $userMessage->update(['is_read' => true]);
            return response()->json(['success' => true]);
        }

        // Check if it's a system announcement
        $systemMessage = UserMessage::whereNull('user_id')
            ->where('id', $messageId)
            ->first();

        if ($systemMessage) {
            // Update pivot table
            $systemMessage->users()->updateExistingPivot($user->id, ['is_read' => true]);
            return response()->json(['success' => true]);
        }

        return response()->json(['error' => 'Message not found'], 404);
    }

    /**
     * Mark all notifications as read
     */
    public function markAllAsRead(Request $request)
    {
        $user = Auth::user();

        // Mark user-specific messages
        $user->unreadMessages()->update(['is_read' => true]);

        // Mark system announcements
        $systemMessages = UserMessage::whereNull('user_id')
            ->whereHas('users', function ($query) use ($user) {
                $query->where('user_id', $user->id)
                    ->where('is_read', false);
            })
            ->get();

        foreach ($systemMessages as $message) {
            $message->users()->updateExistingPivot($user->id, ['is_read' => true]);
        }

        return response()->json(['success' => true]);
    }

    /**
     * Get notification statistics
     */
    public function stats()
    {
        $user = Auth::user();

        $stats = [
            'total_unread' => $user->unreadMessages()->count(),
            'unread_today' => $user->unreadMessages()
                ->whereDate('created_at', today())
                ->count(),
            'unread_this_week' => $user->unreadMessages()
                ->whereBetween('created_at', [now()->startOfWeek(), now()->endOfWeek()])
                ->count(),
            'total_messages' => $user->messages()->count(),
        ];

        return response()->json($stats);
    }
}
