<?php

namespace App\Http\Controllers;

use App\Services\ClubKonnectService;
use App\Models\Transaction;
use App\Models\Wallet;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Inertia\Inertia;

class AirtimeController extends Controller
{
    protected $clubKonnect;

    public function __construct(ClubKonnectService $clubKonnect)
    {
        $this->clubKonnect = $clubKonnect;
    }

    public function index()
    {
        $networks = [
            ['code' => '01', 'name' => 'MTN'],
            ['code' => '02', 'name' => 'GLO'],
            ['code' => '04', 'name' => 'Airtel'],
            ['code' => '03', 'name' => '9mobile'],
        ];

        $user = auth()->user();
        $wallet = $user->wallet()->first();

        if (!$wallet) {
            $wallet = $user->wallet()->create(['balance' => 0]);
        }

        return Inertia::render('Products/BuyAirtime', [
            'networks' => $networks,
            'wallet' => [
                'balance' => $wallet->balance,
            ]
        ]);
    }

    private function getNetworkName($code): string
    {
        $networks = [
            '01' => 'MTN',
            '02' => 'GLO',
            '04' => 'Airtel',
            '03' => '9mobile',
        ];

        return $networks[$code] ?? 'Unknown Network';
    }

    public function purchase(Request $request)
    {
        $request->validate([
            'network' => 'required|string|in:01,02,03,04',
            'amount' => 'required|numeric|min:50|max:200000',
            'phone' => 'required|digits:11|starts_with:0',
        ]);

        Log::info('Airtime purchase request:', $request->all());

        $user = $request->user();
        $wallet = $user->wallet()->first();

        if (!$wallet) {
            return response()->json([
                'success' => false,
                'message' => 'Wallet not found'
            ], 422);
        }

        $originalAmount = floatval($request->amount);

        // Calculate discount
        $discountPercent = $this->getDiscountPercent($user);
        $discountAmount = $originalAmount * ($discountPercent / 100);
        $finalAmount = $originalAmount - $discountAmount;

        // Validate user balance
        if ($wallet->balance < $finalAmount) {
            return response()->json([
                'success' => false,
                'message' => 'Insufficient balance. Required: ₦' .
                    number_format($finalAmount, 2) .
                    ', Available: ₦' . number_format($wallet->balance, 2)
            ], 422);
        }

        // ============ CRITICAL: Check ClubKonnect balance BEFORE any processing ============
        try {
            // This will throw exception if ClubKonnect balance is insufficient
            $this->clubKonnect->ensureSufficientBalance($originalAmount);

            Log::info('ClubKonnect balance check PASSED - proceeding with airtime purchase', [
                'required_amount' => $originalAmount
            ]);
        } catch (\Exception $e) {
            // ClubKonnect has insufficient balance - BLOCK TRANSACTION
            Log::error('❌ AIRTIME PURCHASE BLOCKED - ClubKonnect insufficient balance: ' . $e->getMessage(), [
                'user_id' => $user->id,
                'required_amount' => $originalAmount,
                'phone' => $request->phone
            ]);

            // Return ONLY generic message to user
            return response()->json([
                'success' => false,
                'message' => 'Service unavailable, try again later'
            ], 422);
        }
        // ============ END CRITICAL CHECK ============

        $requestId = 'AT' . time() . rand(1000, 9999);

        try {
            // Call ClubKonnect API
            $result = $this->clubKonnect->buyAirtime(
                $request->network,
                $originalAmount,
                $request->phone,
                $requestId
            );

            Log::info('ClubKonnect Airtime API response:', $result);

            // Check if API call was successful
            if (!($result['success'] ?? false)) {
                // Check for insufficient balance error
                $errorMessage = $result['message'] ?? 'API call failed';
                if (str_contains(strtoupper($errorMessage), 'INSUFFICIENT_BALANCE')) {
                    // Log internally but show generic message
                    Log::error('Vendor balance error during airtime transaction: ' . $errorMessage);
                    throw new \Exception('Service unavailable, try again later');
                }
                throw new \Exception($errorMessage);
            }

            // Get the actual API response data
            $apiData = $result['data'] ?? $result;

            // Check for vendor insufficient balance in the response
            $statusCode = $apiData['statuscode'] ?? '';
            $apiMessage = $apiData['message'] ?? $apiData['status'] ?? '';

            if (
                $statusCode == 'INSUFFICIENT_BALANCE' ||
                str_contains(strtoupper($apiMessage), 'INSUFFICIENT_BALANCE')
            ) {

                // Log internally but show generic message
                Log::error('Vendor balance error in airtime API response: ' . $apiMessage);
                throw new \Exception('Service unavailable, try again later');
            }

            // Check if transaction was successful
            $status = $apiData['status'] ?? 'pending';
            $isSuccessful = in_array(strtoupper($status), ['ORDER_RECEIVED', 'SUCCESSFUL', 'SUCCESS', 'COMPLETED', '100', '200']);

            if ($isSuccessful) {
                // Deduct from user wallet
                $wallet->decrement('balance', $finalAmount);

                // Record successful transaction
                Transaction::create([
                    'user_id' => $user->id,
                    'reference' => $requestId,
                    'type' => 'airtime',
                    'status' => 'success',
                    'original_status' => $status,
                    'amount' => $originalAmount,
                    'amount_charged' => $finalAmount,
                    'network' => $this->getNetworkName($request->network),
                    'phone' => $request->phone,
                    'plan_description' => 'Airtime recharge',
                    'order_id' => $apiData['orderid'] ?? $apiData['order_id'] ?? null,
                    'request_id' => $requestId,
                    'api_response' => json_encode($apiData),
                    'remark' => $apiData['message'] ?? 'Airtime recharge successful',
                ]);

                // Get updated wallet balance
                $wallet->refresh();

                return response()->json([
                    'success' => true,
                    'message' => 'Airtime purchased successfully',
                    'wallet_balance' => $wallet->balance,
                    'order_id' => $apiData['orderid'] ?? $apiData['order_id'] ?? null
                ]);
            } else {
                // Transaction failed at provider side
                Transaction::create([
                    'user_id' => $user->id,
                    'reference' => $requestId,
                    'type' => 'airtime',
                    'status' => 'failed',
                    'original_status' => $status,
                    'amount' => $originalAmount,
                    'amount_charged' => $finalAmount,
                    'network' => $this->getNetworkName($request->network),
                    'phone' => $request->phone,
                    'plan_description' => 'Airtime recharge',
                    'order_id' => $apiData['orderid'] ?? $apiData['order_id'] ?? null,
                    'request_id' => $requestId,
                    'api_response' => json_encode($apiData),
                    'remark' => $apiData['message'] ?? 'Transaction failed',
                ]);

                return response()->json([
                    'success' => false,
                    'message' => 'Service unavailable, try again later',
                    'order_id' => $apiData['orderid'] ?? $apiData['order_id'] ?? null
                ]);
            }
        } catch (\Exception $e) {
            Log::error('Error processing airtime purchase: ' . $e->getMessage());

            // Record error transaction
            Transaction::create([
                'user_id' => $user->id,
                'reference' => $requestId,
                'type' => 'airtime',
                'status' => 'failed',
                'original_status' => 'error',
                'amount' => $originalAmount,
                'amount_charged' => $finalAmount,
                'network' => $this->getNetworkName($request->network),
                'phone' => $request->phone,
                'plan_description' => 'Airtime recharge',
                'order_id' => null,
                'request_id' => $requestId,
                'api_response' => json_encode(['error' => $e->getMessage()]),
                'remark' => 'API Error: ' . $e->getMessage(),
            ]);

            // Return generic message
            return response()->json([
                'success' => false,
                'message' => 'Service unavailable, try again later'
            ], 422);
        }
    }

    /**
     * Get discount percentage based on user type
     */
    private function getDiscountPercent($user): float
    {
        // Default discount of 2%
        return 2.0;

        // You can implement your logic here based on user type/role
        // Example:
        // if ($user->hasRole('distributor')) {
        //     return 3.0;
        // } elseif ($user->hasRole('upgraded')) {
        //     return 2.5;
        // }
    }

    public function callback(Request $request)
    {
        // Log the full callback for debugging
        Log::info('ClubKonnect Airtime Callback Received', $request->all());

        // Get parameters according to ClubKonnect documentation
        $orderId = $request->get('orderid');
        $statusCode = $request->get('statuscode');
        $orderStatus = $request->get('orderstatus');
        $orderRemark = $request->get('orderremark');
        $requestId = $request->get('requestid');

        if (empty($orderId) && empty($requestId)) {
            Log::error('Airtime callback missing both orderId and requestId', $request->all());
            return response()->json(['error' => 'Missing order identifier'], 400);
        }

        // Find transaction
        $transaction = null;
        if (!empty($orderId)) {
            $transaction = Transaction::where('order_id', $orderId)->first();
        }

        if (!$transaction && !empty($requestId)) {
            $transaction = Transaction::where('request_id', $requestId)->first();
        }

        if (!$transaction) {
            Log::error('Airtime transaction not found for callback', [
                'order_id' => $orderId,
                'request_id' => $requestId
            ]);
            return response()->json(['error' => 'Transaction not found'], 404);
        }

        // Determine new status
        $newStatus = 'pending';
        if (!empty($statusCode)) {
            if ($statusCode == '100' || $statusCode == '200') {
                $newStatus = 'success';
            } elseif (in_array($statusCode, ['400', '401', '402', '403', '404', '405', '406', '407', '408', '409'])) {
                $newStatus = 'failed';
            }
        } elseif (!empty($orderStatus)) {
            $statusFromOrderStatus = strtolower($orderStatus);
            if (str_contains($statusFromOrderStatus, 'completed') || str_contains($statusFromOrderStatus, 'success')) {
                $newStatus = 'success';
            } elseif (str_contains($statusFromOrderStatus, 'failed')) {
                $newStatus = 'failed';
            }
        }

        // Update the transaction
        $transaction->update([
            'status' => $newStatus,
            'original_status' => $orderStatus ?? $transaction->original_status,
            'remark' => $orderRemark ?? $transaction->remark,
        ]);

        Log::info('Airtime transaction updated', [
            'transaction_id' => $transaction->id,
            'new_status' => $newStatus,
            'status_code' => $statusCode,
            'order_status' => $orderStatus,
        ]);

        // Process refund if transaction failed
        if ($newStatus === 'failed') {
            $wallet = Wallet::where('user_id', $transaction->user_id)->first();
            if ($wallet) {
                $wallet->increment('balance', $transaction->amount_charged);
                Log::info('Refund processed for failed airtime transaction', [
                    'order_id' => $orderId,
                    'amount' => $transaction->amount_charged,
                    'wallet_id' => $wallet->id
                ]);
            }
        }

        return response()->json(['status' => 'ok']);
    }
}
