<?php

namespace App\Filament\Resources\Announcements\Tables;

use App\Models\UserMessage;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\BadgeColumn;
use Filament\Tables\Filters;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\Filter;
// Import Table Actions correctly
use Filament\Tables\Actions\EditAction;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\BulkAction;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\DeleteBulkAction;

class AnnouncementsTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('title')
                    ->searchable()
                    ->sortable()
                    ->limit(50),

                BadgeColumn::make('type')
                    ->colors([
                        'info' => 'primary',
                        'success' => 'success',
                        'warning' => 'warning',
                        'error' => 'danger',
                    ]),

                IconColumn::make('is_read')
                    ->label('Read')
                    ->boolean()
                    ->sortable(),

                TextColumn::make('user.name')
                    ->label('User')
                    ->sortable()
                    ->searchable(),

                TextColumn::make('expires_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->defaultSort('created_at', 'desc')
            ->filters([
                SelectFilter::make('type')
                    ->options([
                        'info' => 'Information',
                        'success' => 'Success',
                        'warning' => 'Warning',
                        'error' => 'Error',
                    ]),

                Filter::make('unread')
                    ->query(fn($query) => $query->where('is_read', false))
                    ->label('Unread Only'),

                Filter::make('expired')
                    ->query(fn($query) => $query->where('expires_at', '<', now()))
                    ->label('Expired'),
            ])
            ->actions([
                EditAction::make(),
                Action::make('markAsRead')
                    ->label('Mark as Read')
                    ->icon('heroicon-o-check')
                    ->action(function (UserMessage $record) {
                        $record->update(['is_read' => true]);
                    })
                    ->visible(fn(UserMessage $record) => !$record->is_read)
                    ->color('success'),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                    BulkAction::make('markAsRead')
                        ->label('Mark as Read')
                        ->icon('heroicon-o-check')
                        ->action(function ($records) {
                            $records->each->update(['is_read' => true]);
                        })
                        ->color('success'),
                ]),
            ]);
    }
}
