<?php

namespace App\Filament\Pages;

use App\Models\Setting;
use Filament\Pages\Page;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Components\TextEntry;
use Filament\Schemas\Components\Toggle;
use Filament\Schemas\Components\FileUpload;
use Filament\Schemas\Contracts\HasSchemas;
use Illuminate\Support\Facades\Cache;

class Settings extends Page implements HasSchemas
{
    public ?array $data = [];

    protected static ?string $title = 'Settings';
    
    public static function getNavigationIcon(): ?string
    {
        return 'heroicon-o-cog';
    }

    public static function getNavigationSort(): ?int
    {
        return 1;
    }

    public static function getNavigationLabel(): string
    {
        return 'Settings';
    }

    public function mount(): void
    {
        $this->data = $this->getSettings();
        }

    public function getSchemas(): array
    {
        return [
            Section::make('General Settings')
                ->schema([
                    TextEntry::make('app.name')
                        ->label('Application Name')
                        ->required()
                        ->maxLength(255)
                        ->default($this->data['app.name'] ?? ''),

                    TextEntry::make('app.url')
                        ->label('Application URL')
                        ->required()
                        ->url()
                        ->default($this->data['app.url'] ?? ''),

                    TextEntry::make('app.timezone')
                        ->label('Timezone')
                        ->options($this->getTimezones())
                        ->searchable()
                        ->required()
                        ->default($this->data['app.timezone'] ?? config('app.timezone'))
                ])
                ->columns(2),

            Section::make('Branding')
                ->schema([
                    FileUpload::make('branding.logo')
                        ->label('Logo')
                        ->disk('public')
                        ->directory('branding')
                        ->image()
                        ->maxSize(2048)
                        ->acceptedFileTypes(['image/png', 'image/jpeg', 'image/svg+xml'])
                        ->default($this->data['branding.logo'] ?? null),

                    FileUpload::make('branding.favicon')
                        ->label('Favicon')
                        ->disk('public')
                        ->directory('branding')
                        ->maxSize(1024)
                        ->acceptedFileTypes(['image/x-icon', 'image/png', 'image/svg+xml'])
                        ->default($this->data['branding.favicon'] ?? null)
                ])
                ->columns(2),

            Section::make('Monnify Payment Gateway')
                ->schema([
                    Toggle::make('payment.monnify.sandbox')
                        ->label('Sandbox Mode')
                        ->helperText('Enable for testing, disable for live transactions')
                        ->default(!empty($this->data['payment.monnify.sandbox'])),

                    TextEntry::make('payment.monnify.base_url')
                        ->label('Base URL')
                        ->url()
                        ->required()
                        ->default($this->data['payment.monnify.base_url'] ?? ''),

                    TextEntry::make('payment.monnify.api_key')
                        ->label('API Key')
                        ->password()
                        ->revealable()
                        ->default($this->data['payment.monnify.api_key'] ?? ''),

                    TextEntry::make('payment.monnify.secret_key')
                        ->label('Secret Key')
                        ->password()
                        ->revealable()
                        ->default($this->data['payment.monnify.secret_key'] ?? ''),

                    TextEntry::make('payment.monnify.contract_code')
                        ->label('Contract Code')
                        ->default($this->data['payment.monnify.contract_code'] ?? '')
                ])
                ->columns(2),

            Section::make('ClubKonnect')
                ->schema([
                    Toggle::make('payment.clubkonnect.test_mode')
                        ->label('Test Mode'),

                    TextEntry::make('payment.clubkonnect.user_id')
                        ->label('User ID'),

                    TextEntry::make('payment.clubkonnect.api_key')
                        ->label('API Key')
                        ->password()
                        ->revealable(),

                    TextEntry::make('payment.clubkonnect.test_success_rate')
                        ->label('Test Success Rate (%)')
                        ->numeric()
                        ->minValue(0)
                        ->maxValue(100)
                ])
                ->columns(2),

            Section::make('Company Bank Details')
                ->schema([
                    TextEntry::make('company.bank.name')
                        ->label('Bank Name')
                        ->maxLength(255),

                    TextEntry::make('company.bank.account_name')
                        ->label('Account Name')
                        ->maxLength(255),

                    TextEntry::make('company.bank.account_number')
                        ->label('Account Number'),

                    TextEntry::make('company.bank.transfer_notes')
                        ->label('Transfer Instructions')
                        ->textarea()
                        ->rows(3)
                ])
                ->columns(2),
        ];
    }


    public function save(): void
    {
        foreach ($this->data as $key => $value) {
            $setting = Setting::where('key', $key)->first();

            if ($setting) {
                $isSensitive = in_array($key, [
                    'payment.monnify.api_key',
                    'payment.monnify.secret_key',
                    'payment.clubkonnect.api_key',
                ]);

                if ($isSensitive && !empty($value)) {
                    $setting->setEncryptedValue($value);
                } else {
                    $setting->update(['value' => $value]);
                }
            }
        }

        Cache::forget('app_settings');
        $this->notify('success', 'Settings saved successfully.');
    }

    protected function getSettings(): array
    {
        $settings = [];

        try {
            foreach (Setting::all() as $setting) {
                $settings[$setting->key] = $setting->is_encrypted
                    ? $setting->getDecryptedValue()
                    : $setting->value;
            }
        } catch (\Exception $e) {
            // ignore if table doesn't exist
        }

        return $settings;
    }

    protected function getTimezones(): array
    {
        return collect(timezone_identifiers_list())
            ->mapWithKeys(fn($zone) => [$zone => $zone])
            ->toArray();
    }
}
