<?php
// app/Console/Commands/SyncClubKonnectTransactions.php

namespace App\Console\Commands;

use App\Models\Transaction;
use App\Models\ApiProvider;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Http;

class SyncClubKonnectTransactions extends Command
{
    protected $signature = 'clubkonnect:sync-transactions';
    protected $description = 'Sync pending ClubKonnect transactions';

    public function handle()
    {
        $provider = ApiProvider::where('code', 'clubkonnect')->first();
        
        if (!$provider) {
            $this->error('ClubKonnect provider not found');
            return;
        }
        
        // Get pending transactions from the last 24 hours
        $transactions = Transaction::where('api_provider_id', $provider->id)
            ->where('status', 'pending')
            ->where('created_at', '>=', now()->subDay())
            ->get();
        
        $this->info("Found {$transactions->count()} pending ClubKonnect transactions");
        
        foreach ($transactions as $transaction) {
            $this->syncTransaction($provider, $transaction);
        }
        
        $this->info('Sync completed');
    }
    
    private function syncTransaction($provider, $transaction)
    {
        $orderId = $transaction->provider_reference;
        
        if (!$orderId) {
            $this->warn("Transaction {$transaction->id} has no order ID");
            return;
        }
        
        $config = $provider->config ?? [];
        $requestData = [
            'UserID' => $provider->secret_key,
            'APIKey' => $provider->api_key,
            'OrderID' => $orderId,
        ];
        
        try {
            $response = Http::timeout(30)
                ->get($provider->api_url . ($config['query_endpoint'] ?? '/APIQueryV1.asp'), $requestData);
            
            $responseData = $response->json();
            
            $statusCode = $responseData['statuscode'] ?? null;
            
            if ($statusCode === '200') { // ORDER_COMPLETED
                $transaction->update([
                    'status' => 'success',
                    'details' => array_merge($transaction->details ?? [], [
                        'sync_response' => $responseData,
                        'sync_at' => now()->toDateTimeString(),
                    ])
                ]);
                $this->info("Transaction {$orderId} marked as success");
            } elseif (in_array($statusCode, ['300', '400'])) { // FAILED or CANCELLED
                $transaction->update([
                    'status' => 'failed',
                    'details' => array_merge($transaction->details ?? [], [
                        'sync_response' => $responseData,
                        'sync_at' => now()->toDateTimeString(),
                    ])
                ]);
                $this->warn("Transaction {$orderId} marked as failed");
            }
            
        } catch (\Exception $e) {
            $this->error("Failed to sync transaction {$orderId}: " . $e->getMessage());
        }
    }
}